import basis_set_exchange
import numpy as np
import pytest
import torch
from pyscf import gto

from mldft.ml.data.components.basis_info import BasisInfo
from mldft.ml.data.components.convert_transforms import to_torch
from mldft.ml.data.components.of_data import OFData
from mldft.ofdft.energies import Energies

# Basis set from: https://doi.org/10.48550/arXiv.2207.03587 (via Basis Set Exchange)
# sigmaTZHF basis for Hellmann-Feynman forces, it is an optimized basis set with a
# very small error between the HF and analytic forces
# => if the HF forces are correctly implemented, the analytic forces should be very close to the
#    HF forces calculated with this basis set
#
SIGMA_TZHF = """
BASIS "ao basis" SPHERICAL PRINT
#BASIS SET: (66s,60p,12d,3f) -> [6s,6p,3d,1f]
H    S
   3877.0000                -0.7830456701E+00
    576.3000                 0.9340134323E+00
    131.2000                -0.2540690663E+00
     36.9800                 0.1126282677E+00
     11.7300                -0.1070596431E+01
      3.9200                 0.1391763637E+01
      1.3590                 0.3847972828E+01
      0.9351                -0.7381478510E+01
      0.5111                 0.3543777956E+01
      0.1991                -0.4822381396E+00
      0.0779                 0.1526792416E-01
H    S
   3877.0000                -0.3211288303E+00
    576.3000                 0.3963404401E+00
    131.2000                -0.1797150150E+00
     36.9800                 0.1726414649E+00
     11.7300                -0.1097923035E+01
      3.9200                 0.4364354328E+00
      1.3590                -0.3959284477E+01
      0.9351                 0.5925117323E+01
      0.5111                -0.2013723848E+01
      0.1991                 0.5852190783E-01
      0.0779                -0.1044156013E+00
H    S
   3877.0000                 0.1635837668E-04
    576.3000                 0.1305468718E-03
    131.2000                 0.6730846148E-03
     36.9800                 0.2930889590E-02
     11.7300                 0.1110916630E-01
      3.9200                 0.4147441563E-01
      1.3590                 0.1286073260E+00
      0.9351                 0.5070224816E-07
      0.5111                 0.3016545186E+00
      0.1991                 0.4476183889E+00
      0.0779                 0.2100360237E+00
H    S
   3877.0000                 0.6353341336E-04
    576.3000                 0.4500791880E-03
    131.2000                 0.2688329850E-02
     36.9800                 0.9726601400E-02
     11.7300                 0.4493053499E-01
      3.9200                 0.1287485200E+00
      1.3590                 0.5302905103E+00
      0.9351                -0.8978786992E-06
      0.5111                 0.5000089725E+00
      0.1991                -0.4732310755E+00
      0.0779                -0.5820833216E+00
H    S
   3877.0000                -0.2965626227E+00
    576.3000                 0.3156167240E+00
    131.2000                 0.1190397236E+00
     36.9800                -0.2866807800E+00
     11.7300                 0.1316562302E+01
      3.9200                -0.2369212932E+01
      1.3590                 0.1103464754E+02
      0.9351                -0.1855074449E+02
      0.5111                 0.1319555129E+02
      0.1991                -0.6306876718E+01
      0.0779                 0.2153916741E+01
H    S
   3877.0000                 0.1456749685E-03
    576.3000                 0.1300918968E-02
    131.2000                 0.5557381099E-02
     36.9800                 0.3002096362E-01
     11.7300                 0.7914946274E-01
      3.9200                 0.3095052514E+00
      1.3590                -0.1272157336E+01
      0.9351                 0.2014479349E-06
      0.5111                 0.1770819722E+00
      0.1991                 0.2195335512E+01
      0.0779                -0.1895351640E+01
H    P
    576.3000                 0.9413152271E-08
    131.2000                 0.2060780430E-07
     36.9800                -0.4596384468E-08
     11.7300                -0.2167370263E-07
      3.9200                 0.3418492337E-01
      2.6390                -0.2284166131E-06
      1.3590                 0.3404282151E+00
      0.5111                 0.5440700162E+00
      0.1991                 0.2150345018E+00
      0.0779                -0.6800004817E-07
H    P
    576.3000                 0.1364908337E-07
    131.2000                -0.3637865831E-07
     36.9800                 0.2903909618E-07
     11.7300                -0.3255921134E-07
      3.9200                -0.1917883736E+00
      2.6390                -0.9103130007E-07
      1.3590                -0.1029940804E+01
      0.5111                 0.6492738982E+00
      0.1991                 0.4424354902E+00
      0.0779                -0.2315216065E-07
H    P
    576.3000                -0.4409585562E-01
    131.2000                -0.7393168441E-01
     36.9800                -0.2307526158E+00
     11.7300                -0.3154357054E+00
      3.9200                -0.1281413318E+01
      2.6390                 0.8631096728E+00
      1.3590                 0.5928072434E+00
      0.5111                -0.1461233606E+00
      0.1991                -0.6552515093E+00
      0.0779                 0.6799574201E+00
H    P
    576.3000                 0.5363834167E-02
    131.2000                -0.3708868585E-01
     36.9800                 0.7362052594E-01
     11.7300                -0.3272842833E+00
      3.9200                 0.2929296154E+01
      2.6390                -0.4967595560E+01
      1.3590                 0.2358018978E+01
      0.5111                 0.2490653324E+00
      0.1991                -0.3230276894E+00
      0.0779                -0.6131121505E+00
H    P
    576.3000                 0.6715239786E-03
    131.2000                -0.2045475230E+00
     36.9800                 0.1853587490E+00
     11.7300                -0.1537562431E+01
      3.9200                 0.8547977081E+01
      2.6390                -0.1191882268E+02
      1.3590                 0.6908775961E+01
      0.5111                -0.4062721313E+01
      0.1991                 0.2851007771E+01
      0.0779                -0.1149413803E+01
H    P
    576.3000                 0.2497267372E-01
    131.2000                 0.2209949246E-01
     36.9800                 0.1648417993E+00
     11.7300                 0.5373492886E-02
      3.9200                 0.4135097170E+01
      2.6390                -0.7018993794E+01
      1.3590                 0.3972859437E+01
      0.5111                -0.9370016686E+00
      0.1991                -0.6805555496E+00
      0.0779                 0.1025675152E+01
H    D
      3.9200                 0.6871086645E-08
      1.3590                 0.6068328458E+00
      0.5111                 0.4553896887E+00
      0.1991                 0.5618425491E-01
H    D
      3.9200                -0.5808455412E+00
      1.3590                -0.5442741761E+00
      0.5111                 0.7982163445E+00
      0.1991                 0.1209528152E+00
H    D
      3.9200                 0.1028522180E+01
      1.3590                -0.1066648616E+01
      0.5111                 0.1725370985E+00
      0.1991                 0.6589690699E+00
H    F
      1.3590                 0.7467794630E+00
      0.5111                 0.3436862864E+00
      0.1991                 0.2646563025E-01
#BASIS SET: (136s,128p,55d,12f,3g) -> [8s,8p,5d,3f,1g]
C    S
 193600.0000                 0.1011104244E-04
  28940.0000                 0.7916907814E-04
   6521.0000                 0.4243474616E-03
   1813.0000                 0.1847417234E-02
    575.9000                 0.6940015415E-02
    201.9000                 0.2291260824E-01
     76.8700                 0.6492496441E-01
     31.4800                 0.1523177365E+00
     13.5600                 0.2837643223E+00
      5.9930                 0.3607772738E+00
      2.6790                 0.2053896021E+00
      1.1410                 0.3140353717E-01
      0.4733                 0.3256747284E-01
      0.3500                 0.6910749824E-06
      0.1913                 0.3516011991E-01
      0.1300                 0.1079609090E-06
      0.0751                 0.6558204553E-02
C    S
 193600.0000                 0.2966931799E-05
  28940.0000                 0.2275967052E-04
   6521.0000                 0.1226362976E-03
   1813.0000                 0.5313639188E-03
    575.9000                 0.2014166147E-02
    201.9000                 0.6664613367E-02
     76.8700                 0.1932931883E-01
     31.4800                 0.4697918722E-01
     13.5600                 0.9636577141E-01
      5.9930                 0.1529837224E+00
      2.6790                 0.1464772173E+00
      1.1410                -0.8351549502E-01
      0.4733                -0.4658813308E+00
      0.3500                 0.4869807518E-06
      0.1913                -0.5025132619E+00
      0.1300                -0.1516449381E-06
      0.0751                -0.9490082445E-01
C    S
 193600.0000                 0.6477477891E-01
  28940.0000                -0.1014805920E+00
   6521.0000                 0.3654276802E+00
   1813.0000                -0.8058891629E+00
    575.9000                 0.1041131734E+01
    201.9000                -0.4404430203E+00
     76.8700                -0.4694122184E+00
     31.4800                 0.1398993644E+01
     13.5600                 0.1923936422E+00
      5.9930                -0.1337797816E+01
      2.6790                 0.4636756693E+00
      1.1410                -0.2140575281E+01
      0.4733                 0.1495379123E+02
      0.3500                -0.1831966660E+02
      0.1913                 0.9713171817E+01
      0.1300                -0.6177652111E+01
      0.0751                 0.2068713726E+01
C    S
 193600.0000                 0.3631663325E-05
  28940.0000                 0.1905405944E-04
   6521.0000                 0.1327209469E-03
   1813.0000                 0.4697868965E-03
    575.9000                 0.2132689937E-02
    201.9000                 0.6154558547E-02
     76.8700                 0.2054903874E-01
     31.4800                 0.4342675119E-01
     13.5600                 0.1164749486E+00
      5.9930                 0.1531293796E+00
      2.6790                 0.3034823633E+00
      1.1410                -0.6610890058E+00
      0.4733                -0.1159408897E+01
      0.3500                 0.7783322180E-06
      0.1913                 0.1121518347E+01
      0.1300                 0.6903609657E-06
      0.0751                 0.3691724581E+00
C    S
 193600.0000                 0.8367468024E-01
  28940.0000                -0.1353408021E+00
   6521.0000                 0.4767168026E+00
   1813.0000                -0.1083968145E+01
    575.9000                 0.1361070975E+01
    201.9000                -0.7937955941E+00
     76.8700                -0.6803830364E+00
     31.4800                 0.7001921656E+00
     13.5600                 0.1737135755E+00
      5.9930                -0.2672126492E+01
      2.6790                 0.7329185675E+01
      1.1410                -0.1278194946E+02
      0.4733                 0.3591456117E+02
      0.3500                -0.3864083895E+02
      0.1913                 0.1984395556E+02
      0.1300                -0.1250756044E+02
      0.0751                 0.3412389333E+01
C    S
 193600.0000                -0.1328588223E-05
  28940.0000                -0.5726210208E-04
   6521.0000                -0.9257143307E-04
   1813.0000                -0.9954849451E-03
    575.9000                -0.2405706092E-02
    201.9000                -0.1088099218E-01
     76.8700                -0.2712440999E-01
     31.4800                -0.8177486490E-01
     13.5600                -0.1629856227E+00
      5.9930                -0.3485310891E+00
      2.6790                -0.2706604102E+00
      1.1410                 0.2907004813E+01
      0.4733                -0.2930406810E+01
      0.3500                -0.3334295716E-05
      0.1913                 0.3079715263E+00
      0.1300                -0.5447434827E-06
      0.0751                 0.7976995000E+00
C    S
 193600.0000                 0.1279023470E+00
  28940.0000                -0.2004950301E+00
   6521.0000                 0.7151228054E+00
   1813.0000                -0.1591583372E+01
    575.9000                 0.1960574589E+01
    201.9000                -0.7769359557E+00
     76.8700                -0.1567184656E+01
     31.4800                 0.2831918212E+01
     13.5600                -0.1472843895E+01
      5.9930                -0.1163930189E+01
      2.6790                 0.2156899774E+01
      1.1410                -0.1208326421E+01
      0.4733                 0.4751586484E+01
      0.3500                -0.9013632544E+01
      0.1913                 0.7319175564E+01
      0.1300                -0.1277091723E+01
      0.0751                -0.1729257891E+01
C    S
 193600.0000                -0.9213612726E-02
  28940.0000                 0.6871444359E-02
   6521.0000                -0.4129038237E-01
   1813.0000                 0.7539221613E-01
    575.9000                -0.8384562432E-01
    201.9000                -0.5058055848E-01
     76.8700                -0.7914823803E-01
     31.4800                -0.2624203212E+00
     13.5600                -0.3987891082E+00
      5.9930                 0.2031857554E+01
      2.6790                -0.1068102262E+01
      1.1410                -0.5777110488E+01
      0.4733                 0.4716645929E+02
      0.3500                -0.6493699092E+02
      0.1913                 0.3972245242E+02
      0.1300                -0.1758392895E+02
      0.0751                 0.1019061238E+01
C    P
 193600.0000                -0.1732102746E-03
  28940.0000                 0.1139475366E-01
   6521.0000                -0.1239160738E-01
   1813.0000                 0.4337861936E-01
    575.9000                 0.2548704947E-01
    201.9000                 0.6310928297E-01
     76.8700                 0.1838242602E+00
     31.4800                 0.2177517554E+00
     13.5600                 0.4154123775E+00
      5.9930                 0.4545577451E+00
      2.6790                -0.9873957898E-01
      1.1410                -0.2427706535E+01
      0.8000                 0.2605492430E+01
      0.4733                -0.5666345082E+00
      0.1913                 0.6304597550E-01
      0.0751                -0.1075643643E+00
C    P
 193600.0000                -0.9982614503E-02
  28940.0000                 0.5998241938E-01
   6521.0000                -0.9196069227E-01
   1813.0000                 0.2091619211E+00
    575.9000                -0.1656147733E-01
    201.9000                 0.2224941688E+00
     76.8700                 0.3556389023E+00
     31.4800                 0.5209689595E+00
     13.5600                -0.4458963667E+00
      5.9930                -0.3826675935E+00
      2.6790                -0.1002395683E+01
      1.1410                 0.7300490464E+01
      0.8000                -0.9569569112E+01
      0.4733                 0.4316856337E+01
      0.1913                -0.1210512813E+01
      0.0751                 0.4253500899E+00
C    P
 193600.0000                 0.3105378200E-08
  28940.0000                 0.3758397939E-08
   6521.0000                -0.1294040507E-07
   1813.0000                 0.1684359565E-07
    575.9000                 0.4011394416E-08
    201.9000                -0.2070738447E-03
     76.8700                -0.7222720978E-03
     31.4800                -0.3101326650E-02
     13.5600                -0.1082175355E-01
      5.9930                -0.3135128004E-01
      2.6790                -0.9072986754E-01
      1.1410                -0.2220755234E+00
      0.8000                 0.1869367464E-06
      0.4733                -0.3658731703E+00
      0.1913                -0.3710010434E+00
      0.0751                -0.1533655167E+00
C    P
 193600.0000                -0.6833197794E-01
  28940.0000                 0.3390874571E+00
   6521.0000                -0.8142905026E+00
   1813.0000                 0.1031175715E+01
    575.9000                -0.7107030634E+00
    201.9000                -0.3694820481E+00
     76.8700                 0.9631909816E+00
     31.4800                -0.3077175251E+00
     13.5600                -0.7281488752E+00
      5.9930                 0.5554029322E+00
      2.6790                 0.6670503484E+00
      1.1410                -0.4971089734E+00
      0.8000                -0.5653512766E+00
      0.4733                 0.7702249452E-01
      0.1913                 0.9843363861E+00
      0.0751                -0.7683339206E+00
C    P
 193600.0000                -0.5656746590E-01
  28940.0000                 0.2822151616E+00
   6521.0000                -0.6880824512E+00
   1813.0000                 0.8568804014E+00
    575.9000                -0.6065910391E+00
    201.9000                -0.3569014801E+00
     76.8700                 0.7963793349E+00
     31.4800                -0.3032942831E+00
     13.5600                -0.3996563256E+00
      5.9930                 0.8234443563E+00
      2.6790                -0.5862858040E+00
      1.1410                -0.5071396490E+01
      0.8000                 0.8999091681E+01
      0.4733                -0.4369095630E+01
      0.1913                 0.8531949467E-01
      0.0751                 0.5310772985E+00
C    P
 193600.0000                -0.7753388336E-08
  28940.0000                -0.2011764689E-07
   6521.0000                -0.7718719807E-08
   1813.0000                 0.1248617437E-07
    575.9000                -0.3497114475E-07
    201.9000                -0.3735191700E-03
     76.8700                -0.3067024401E-02
     31.4800                -0.7064459812E-02
     13.5600                -0.4240877532E-01
      5.9930                -0.9947408465E-01
      2.6790                -0.4657305933E+00
      1.1410                -0.6526394999E+00
      0.8000                 0.1981998324E-06
      0.4733                 0.1290683329E+01
      0.1913                -0.2148060452E-01
      0.0751                -0.6332210703E+00
C    P
 193600.0000                -0.1418344697E-07
  28940.0000                 0.4424535443E-07
   6521.0000                -0.4747637265E-07
   1813.0000                -0.1702349071E-07
    575.9000                 0.4856249495E-07
    201.9000                -0.2667331263E-03
     76.8700                -0.7753574803E-03
     31.4800                -0.4464527985E-02
     13.5600                -0.1152182089E-01
      5.9930                -0.4234145145E-01
      2.6790                -0.9322297726E-01
      1.1410                -0.4635969608E+00
      0.8000                -0.3985025030E-07
      0.4733                -0.4451098919E+00
      0.1913                 0.6370906030E+00
      0.0751                 0.4495783507E+00
C    P
 193600.0000                 0.8886415301E-02
  28940.0000                -0.1994777459E-01
   6521.0000                 0.2939549756E-01
   1813.0000                -0.5631326194E-01
    575.9000                 0.8155448306E-01
    201.9000                -0.1353017632E+00
     76.8700                 0.2008825429E+00
     31.4800                -0.3663378198E+00
     13.5600                 0.8168834618E+00
      5.9930                -0.5648585315E+00
      2.6790                -0.1617703662E+00
      1.1410                -0.3713044563E+01
      0.8000                 0.9524638426E+01
      0.4733                -0.8345658082E+01
      0.1913                 0.4195504992E+01
      0.0751                -0.1689735987E+01
C    D
    201.9000                 0.3216960939E-02
     76.8700                 0.2822481975E-01
     31.4800                 0.3284714794E-01
     13.5600                 0.1573712342E+00
      5.9930                 0.2447560176E+00
      2.6790                 0.3020293477E+00
      1.8500                 0.6380826155E+00
      1.1410                -0.5089590419E+00
      0.4733                -0.8340305865E+00
      0.1913                 0.8798506794E+00
      0.0751                -0.7484140239E-01
C    D
    201.9000                -0.2736934804E-07
     76.8700                 0.8974445729E-07
     31.4800                -0.1098962534E-06
     13.5600                 0.1744604872E-06
      5.9930                -0.2980168831E-06
      2.6790                 0.5106739470E-01
      1.8500                -0.1254898984E-05
      1.1410                 0.2719969617E+00
      0.4733                 0.5701788771E+00
      0.1913                 0.2708747696E+00
      0.0751                -0.1140139427E-07
C    D
    201.9000                 0.1362670057E-07
     76.8700                -0.3531738130E-07
     31.4800                 0.5200266629E-07
     13.5600                -0.8932103718E-07
      5.9930                 0.1062090675E-06
      2.6790                 0.1515176587E+00
      1.8500                 0.3817772787E-06
      1.1410                 0.8174438038E+00
      0.4733                -0.1941662369E+00
      0.1913                -0.7027481242E+00
      0.0751                -0.1153458122E-07
C    D
    201.9000                 0.1329138274E-01
     76.8700                 0.2548935299E-01
     31.4800                 0.9549146664E-01
     13.5600                 0.1554443445E+00
      5.9930                 0.4060780354E+00
      2.6790                 0.1948878154E+00
      1.8500                -0.2860776857E+00
      1.1410                -0.7208384365E+00
      0.4733                 0.8753995238E+00
      0.1913                 0.1690956615E-01
      0.0751                -0.8161730533E+00
C    D
    201.9000                -0.2038782494E-01
     76.8700                -0.9290124664E-02
     31.4800                -0.9347715054E-01
     13.5600                -0.9959561663E-01
      5.9930                -0.2343466879E+00
      2.6790                -0.2223142511E+01
      1.8500                 0.4249198364E+01
      1.1410                -0.2006206360E+01
      0.4733                -0.5253086542E+00
      0.1913                 0.1347659250E+01
      0.0751                -0.1134783375E+01
C    F
      2.6790                 0.1344696530E-07
      1.1410                -0.4479149846E+00
      0.4733                -0.5600690243E+00
      0.1913                -0.1496056065E+00
C    F
      2.6790                 0.5332105190E+00
      1.1410                 0.5355787009E+00
      0.4733                -0.4893617055E+00
      0.1913                -0.3638657677E+00
C    F
      2.6790                -0.1207652461E+01
      1.1410                 0.1187902422E+01
      0.4733                -0.1714183278E+00
      0.1913                -0.5367628332E+00
C    G
      1.1410                -0.5981342503E+00
      0.4733                -0.4816864317E+00
      0.1913                -0.8180600652E-01
#BASIS SET: (128s,128p,50d,12f,3g) -> [8s,8p,5d,3f,1g]
N    S
 218500.0000                -0.1027972418E-04
  32650.0000                -0.8036178867E-04
   7405.0000                -0.4257000381E-03
   2075.0000                -0.1827938168E-02
    666.7000                -0.6738493315E-02
    236.1000                -0.2191161634E-01
     89.8800                -0.6215251860E-01
     36.1700                -0.1458773364E+00
     15.1400                -0.2559537826E+00
      6.5250                -0.2522391255E+00
      4.5000                -0.4725938878E-07
      2.8290                -0.8212287328E-01
      1.2080                -0.1036489740E+00
      0.5135                -0.2356722181E+00
      0.2170                -0.1791425003E+00
      0.0889                -0.2476035018E-01
N    S
 218500.0000                -0.8738520364E-01
  32650.0000                 0.3424276142E+00
   7405.0000                 0.1993761901E+00
   2075.0000                -0.1565698973E+01
    666.7000                 0.1571749403E+01
    236.1000                 0.3030296715E+00
     89.8800                -0.6113538002E+00
     36.1700                -0.7495926235E+00
     15.1400                 0.1056017664E+01
      6.5250                 0.2430896917E+01
      4.5000                -0.4196543445E+01
      2.8290                 0.7981935840E+00
      1.2080                 0.1293787261E+01
      0.5135                -0.6403272996E+00
      0.2170                -0.3452482997E+00
      0.0889                 0.4302396370E+00
N    S
 218500.0000                 0.3318595847E-01
  32650.0000                 0.3382550649E-01
   7405.0000                 0.5551714804E-01
   2075.0000                -0.2372905288E+00
    666.7000                 0.1906407282E+00
    236.1000                 0.4239897420E+00
     89.8800                -0.2005038991E-01
     36.1700                 0.7655305746E+00
     15.1400                 0.9186797065E+00
      6.5250                -0.1262162495E+02
      4.5000                 0.1718309700E+02
      2.8290                -0.6387264170E+01
      1.2080                -0.7923124067E+00
      0.5135                 0.1977943118E+01
      0.2170                -0.1721782092E+01
      0.0889                 0.7463845793E+00
N    S
 218500.0000                -0.8646098370E-05
  32650.0000                -0.6737469929E-04
   7405.0000                -0.3569780628E-03
   2075.0000                -0.1534695271E-02
    666.7000                -0.5663053214E-02
    236.1000                -0.1853415537E-01
     89.8800                -0.5314394072E-01
     36.1700                -0.1287790822E+00
     15.1400                -0.2425666317E+00
      6.5250                -0.3015843846E+00
      4.5000                -0.5854096122E-06
      2.8290                -0.1415679765E+00
      1.2080                 0.1892252654E+00
      0.5135                 0.4472929202E+00
      0.2170                 0.3456064111E+00
      0.0889                 0.4703952428E-01
N    S
 218500.0000                -0.3632004644E-05
  32650.0000                -0.2919739355E-04
   7405.0000                -0.1536616787E-03
   2075.0000                -0.6549033741E-03
    666.7000                -0.2480331285E-02
    236.1000                -0.7999441025E-02
     89.8800                -0.2426837776E-01
     36.1700                -0.6008499226E-01
     15.1400                -0.1357213974E+00
      6.5250                -0.2160566394E+00
      4.5000                 0.9124235743E-07
      2.8290                -0.1627515346E+00
      1.2080                 0.1116076712E+01
      0.5135                 0.5089533502E+00
      0.2170                -0.1139603111E+01
      0.0889                -0.1999885355E+00
N    S
 218500.0000                -0.4835053533E-05
  32650.0000                -0.4260196087E-04
   7405.0000                -0.2008561079E-03
   2075.0000                -0.1049839243E-02
    666.7000                -0.3044593621E-02
    236.1000                -0.1344685877E-01
     89.8800                -0.2899930645E-01
     36.1700                -0.1157513757E+00
     15.1400                -0.1499425366E+00
      6.5250                -0.6163201440E+00
      4.5000                 0.1011677464E-05
      2.8290                 0.8850890903E+00
      1.2080                 0.1811940421E+01
      0.5135                -0.3249573671E+01
      0.2170                 0.1191749590E+01
      0.0889                 0.4675339778E+00
N    S
 218500.0000                -0.1584510604E-01
  32650.0000                 0.1223241102E+00
   7405.0000                 0.8176936538E-01
   2075.0000                -0.5970157828E+00
    666.7000                 0.5413746579E+00
    236.1000                 0.2314726386E+00
     89.8800                -0.3889441816E+00
     36.1700                -0.3880478879E+00
     15.1400                 0.7129509600E+00
      6.5250                -0.6964298733E+01
      4.5000                 0.1109024753E+02
      2.8290                -0.3899302874E+01
      1.2080                -0.1332797541E+01
      0.5135                 0.2877484947E+00
      0.2170                 0.1528215905E+01
      0.0889                -0.1269638888E+01
N    S
 218500.0000                -0.1339118063E-01
  32650.0000                 0.2113457652E-01
   7405.0000                 0.8020741498E-02
   2075.0000                -0.6904718497E-01
    666.7000                 0.1219164247E+00
    236.1000                -0.5698637574E-01
     89.8800                 0.3079971408E+00
     36.1700                -0.4471611979E+00
     15.1400                 0.1916327888E+01
      6.5250                -0.6709645711E+01
      4.5000                 0.9507519267E+01
      2.8290                -0.7589004911E+01
      1.2080                 0.7105664302E+01
      0.5135                -0.7900931248E+01
      0.2170                 0.6416874617E+01
      0.0889                -0.2759451152E+01
N    P
 218500.0000                 0.3643646984E+00
  32650.0000                 0.1508872473E-01
   7405.0000                -0.1113673734E+01
   2075.0000                 0.6513994403E+00
    666.7000                -0.4317625080E+00
    236.1000                 0.8869532880E+00
     89.8800                -0.7032253571E+00
     36.1700                 0.1116707255E+00
     15.1400                 0.2171601499E+00
      6.5250                -0.6471008147E+00
      2.8290                 0.3892408008E+00
      1.2080                 0.3030824065E+00
      0.7000                -0.1520395749E+00
      0.5135                -0.3161584913E+00
      0.2170                 0.1513758512E+00
      0.0889                 0.4646174969E-01
N    P
 218500.0000                 0.8198773575E-01
  32650.0000                 0.7992509388E-02
   7405.0000                -0.2354184258E+00
   2075.0000                 0.1723138737E+00
    666.7000                -0.4576845766E-01
    236.1000                 0.3149601698E+00
     89.8800                 0.1025685880E-01
     36.1700                 0.3811194731E+00
     15.1400                 0.3124711535E+00
      6.5250                 0.4217403686E+00
      2.8290                -0.5973538208E+00
      1.2080                -0.3361738069E+00
      0.7000                -0.2008678625E+00
      0.5135                 0.9832828584E+00
      0.2170                -0.3784928578E+00
      0.0889                -0.7107134860E-01
N    P
 218500.0000                -0.6908418105E-08
  32650.0000                -0.1471708343E-07
   7405.0000                 0.2051617722E-07
   2075.0000                -0.3955355404E-07
    666.7000                 0.5168134010E-07
    236.1000                -0.3036104552E-03
     89.8800                -0.1024164555E-02
     36.1700                -0.4759618420E-02
     15.1400                -0.1656589068E-01
      6.5250                -0.5080781398E-01
      2.8290                -0.1383078177E+00
      1.2080                -0.2775518427E+00
      0.7000                -0.1232718914E-06
      0.5135                -0.3661140927E+00
      0.2170                -0.3029715052E+00
      0.0889                -0.9765171455E-01
N    P
 218500.0000                 0.1151798740E-07
  32650.0000                 0.1146557408E-07
   7405.0000                -0.2317884924E-07
   2075.0000                 0.1941133626E-07
    666.7000                 0.1639040413E-08
    236.1000                -0.7367795158E-03
     89.8800                -0.3936795788E-02
     36.1700                -0.1230346913E-01
     15.1400                -0.6747761315E-01
      6.5250                -0.1803420526E+00
      2.8290                -0.7383859963E+00
      1.2080                 0.9280159985E-01
      0.7000                 0.1308258509E-05
      0.5135                 0.1245492634E+01
      0.2170                -0.6011660777E+00
      0.0889                -0.3658033805E+00
N    P
 218500.0000                 0.1690487133E-01
  32650.0000                 0.1118020187E-01
   7405.0000                -0.4358886375E-01
   2075.0000                 0.1135960631E+00
    666.7000                -0.3619688161E-01
    236.1000                 0.4317612751E+00
     89.8800                -0.3515543514E+00
     36.1700                 0.1232582375E+01
     15.1400                -0.3022003236E+01
      6.5250                 0.3166593792E+01
      2.8290                -0.2040596003E+01
      1.2080                 0.3956876530E+01
      0.7000                -0.1242244735E+02
      0.5135                 0.1165578161E+02
      0.2170                -0.3362439582E+01
      0.0889                 0.1019864358E+01
N    P
 218500.0000                -0.6961349038E-01
  32650.0000                 0.5821993813E-02
   7405.0000                 0.2381405592E+00
   2075.0000                -0.7297272444E-01
    666.7000                 0.1610215702E+00
    236.1000                 0.5204194239E-01
     89.8800                 0.3091842150E+00
     36.1700                 0.4897813059E+00
     15.1400                -0.5919861193E+00
      6.5250                 0.9143840708E-01
      2.8290                -0.1090777816E+01
      1.2080                 0.2685977296E+01
      0.7000                -0.2404220808E+01
      0.5135                 0.9167470909E+00
      0.2170                -0.8103040320E+00
      0.0889                 0.8462909844E+00
N    P
 218500.0000                -0.8991717476E-08
  32650.0000                -0.2985867373E-08
   7405.0000                -0.6568998826E-08
   2075.0000                 0.2681511771E-09
    666.7000                -0.1224101467E-07
    236.1000                -0.4149620856E-03
     89.8800                -0.1431027530E-02
     36.1700                -0.6664747149E-02
     15.1400                -0.2343845029E-01
      6.5250                -0.6857575075E-01
      2.8290                -0.2251201352E+00
      1.2080                -0.5724317993E+00
      0.7000                -0.4534271935E-06
      0.5135                -0.3727331403E-01
      0.2170                 0.6765980336E+00
      0.0889                 0.2637891694E+00
N    P
 218500.0000                -0.3180620684E-01
  32650.0000                 0.3295433661E-02
   7405.0000                 0.1203008155E+00
   2075.0000                -0.4448603537E-01
    666.7000                 0.1321874357E+00
    236.1000                -0.4195885829E-01
     89.8800                 0.3953641056E+00
     36.1700                -0.1573293162E+00
     15.1400                 0.5847600128E+00
      6.5250                -0.1935381305E+01
      2.8290                 0.2296213257E+01
      1.2080                -0.8174902112E+00
      0.7000                -0.7038246016E+00
      0.5135                -0.2995299844E+00
      0.2170                 0.1918881580E+01
      0.0889                -0.1365262688E+01
N    D
    236.1000                -0.3733571788E-01
     89.8800                -0.7806380894E-01
     36.1700                -0.2142725875E+00
     15.1400                -0.4863166818E+00
      6.5250                -0.5042811704E+00
      2.8290                 0.1464549016E+01
      1.2080                -0.1036177338E+01
      0.5135                -0.2424587842E-01
      0.2170                 0.4740100105E+00
      0.0889                -0.8007285451E-01
N    D
    236.1000                -0.8093653676E-02
     89.8800                -0.2901805470E-01
     36.1700                -0.5596046587E-01
     15.1400                -0.2115960142E+00
      6.5250                -0.4052289977E+00
      2.8290                -0.6502544145E+00
      1.2080                 0.9572344577E+00
      0.5135                -0.2949740911E-01
      0.2170                -0.4994465171E+00
      0.0889                 0.2995135403E-01
N    D
    236.1000                -0.3843848561E-08
     89.8800                -0.1950634703E-07
     36.1700                 0.2076925352E-07
     15.1400                -0.1350652078E-07
      6.5250                -0.2555697455E-07
      2.8290                 0.1234760088E+00
      1.2080                 0.3908172574E+00
      0.5135                 0.5047745838E+00
      0.2170                 0.1529950549E+00
      0.0889                 0.1169304836E-07
N    D
    236.1000                 0.1626284641E-07
     89.8800                -0.1645658367E-07
     36.1700                -0.1385091872E-07
     15.1400                 0.7849993740E-08
      6.5250                -0.1803260427E-07
      2.8290                -0.4447375395E+00
      1.2080                -0.6563233555E+00
      0.5135                 0.6205546539E+00
      0.2170                 0.4062293773E+00
      0.0889                -0.8535757518E-08
N    D
    236.1000                -0.1413508001E-01
     89.8800                -0.2559329052E-01
     36.1700                -0.6700278164E-01
     15.1400                -0.1267792737E+00
      6.5250                 0.9092200961E-01
      2.8290                 0.6677308037E+00
      1.2080                -0.1505236812E+01
      0.5135                 0.1579429542E+01
      0.2170                -0.4742578128E+00
      0.0889                -0.7515304676E+00
N    F
      2.8290                 0.1513748736E+00
      1.2080                 0.5264483063E+00
      0.5135                 0.4741986394E+00
      0.2170                 0.2316907900E-08
N    F
      2.8290                -0.3439440700E+00
      1.2080                -0.5345889088E+00
      0.5135                 0.5626852410E+00
      0.2170                 0.5245876247E+00
N    F
      2.8290                 0.9315142003E+00
      1.2080                -0.4360055994E+00
      0.5135                -0.6631273282E+00
      0.2170                 0.9292963147E+00
N    G
      2.8290                -0.2589624262E+00
      1.2080                -0.5884919914E+00
      0.5135                -0.3456071354E+00
#BASIS SET: (128s,128p,50d,12f,3g) -> [8s,8p,5d,3f,1g]
O    S
 254500.0000                -0.1305098514E-04
  38080.0000                -0.1283106325E-03
   8262.0000                -0.6353663990E-03
   2405.0000                -0.2681641183E-02
    764.0000                -0.1022173787E-01
    266.3000                -0.3366650580E-01
    101.0000                -0.9274117350E-01
     41.6400                -0.2017024752E+00
     18.2600                -0.3300574008E+00
      8.2510                -0.3330279263E+00
      3.7120                -0.1266798733E+00
      2.7500                -0.9138518189E-04
      1.6160                -0.1373154563E-01
      0.6891                -0.1969192461E-01
      0.2875                -0.1290641780E-01
      0.1144                -0.1852591718E-02
O    S
 254500.0000                -0.3466167655E-05
  38080.0000                -0.3394329576E-04
   8262.0000                -0.1678514014E-03
   2405.0000                -0.7105510388E-03
    764.0000                -0.2717910901E-02
    266.3000                -0.9071321289E-02
    101.0000                -0.2571092300E-01
     41.6400                -0.5990144302E-01
     18.2600                -0.1129639464E+00
      8.2510                -0.1604932509E+00
      3.7120                -0.8275302708E-01
      2.7500                -0.2892508840E-04
      1.6160                 0.2218381671E+00
      0.6891                 0.5095565501E+00
      0.2875                 0.3832063582E+00
      0.1144                 0.4879509738E-01
O    S
 254500.0000                 0.1850365909E+00
  38080.0000                -0.4937956939E+00
   8262.0000                 0.7964241620E+00
   2405.0000                -0.8060945450E+00
    764.0000                -0.4524213784E+00
    266.3000                 0.1866629604E+01
    101.0000                -0.6921373082E+00
     41.6400                -0.1416747147E+01
     18.2600                 0.1054610285E+01
      8.2510                -0.6857777881E+00
      3.7120                 0.6201291828E+01
      2.7500                -0.7793891063E+01
      1.6160                 0.2730670504E+01
      0.6891                -0.1190017374E+01
      0.2875                 0.1501817688E+01
      0.1144                -0.9388846329E+00
O    S
 254500.0000                 0.1006606565E+00
  38080.0000                -0.2719137357E+00
   8262.0000                 0.4435107256E+00
   2405.0000                -0.4110555503E+00
    764.0000                -0.2591982183E+00
    266.3000                 0.1220145426E+01
    101.0000                -0.2435134857E+00
     41.6400                -0.4077953446E-01
     18.2600                 0.4970076955E+00
      8.2510                -0.9414297299E+00
      3.7120                 0.2421744652E+00
      2.7500                -0.1354392213E+01
      1.6160                 0.2142961699E+01
      0.6891                -0.3592609138E+00
      0.2875                -0.1310592761E+01
      0.1144                 0.1111419517E+01
O    S
 254500.0000                 0.3912630748E-05
  38080.0000                 0.3769189363E-04
   8262.0000                 0.1912286420E-03
   2405.0000                 0.7888888929E-03
    764.0000                 0.3096379361E-02
    266.3000                 0.1011110854E-01
    101.0000                 0.3017410196E-01
     41.6400                 0.6846043981E-01
     18.2600                 0.1465071292E+00
      8.2510                 0.2202311586E+00
      3.7120                 0.1373509372E+00
      2.7500                 0.3160903901E-04
      1.6160                -0.1180789086E+01
      0.6891                -0.3941145973E+00
      0.2875                 0.1113862005E+01
      0.1144                 0.1714636926E+00
O    S
 254500.0000                 0.5639153132E-05
  38080.0000                 0.5804656092E-04
   8262.0000                 0.2469419681E-03
   2405.0000                 0.1274538861E-02
    764.0000                 0.3798089801E-02
    266.3000                 0.1759297116E-01
    101.0000                 0.3443370224E-01
     41.6400                 0.1442493770E+00
     18.2600                 0.1345919472E+00
      8.2510                 0.7553658447E+00
      3.7120                -0.1318220882E+01
      2.7500                 0.7068365920E-04
      1.6160                -0.1384978240E+01
      0.6891                 0.3161432182E+01
      0.2875                -0.1392603807E+01
      0.1144                -0.3118412840E+00
O    S
 254500.0000                -0.5913830208E-01
  38080.0000                 0.1439004688E+00
   8262.0000                -0.2372095366E+00
   2405.0000                 0.3632826046E+00
    764.0000                -0.1986490037E+00
    266.3000                 0.3473929929E+00
    101.0000                -0.9022907716E+00
     41.6400                 0.2784564049E+01
     18.2600                -0.3976312620E+01
      8.2510                 0.9040148640E+00
      3.7120                 0.5271167368E+01
      2.7500                -0.1476887856E+01
      1.6160                -0.6524629155E+01
      0.6891                 0.6660364907E+01
      0.2875                -0.4340142911E+01
      0.1144                 0.1537395724E+01
O    S
 254500.0000                -0.4160462967E-01
  38080.0000                 0.1079179603E+00
   8262.0000                -0.1741818747E+00
   2405.0000                 0.2376398841E+00
    764.0000                 0.1536483751E-02
    266.3000                -0.5003068325E-01
    101.0000                 0.4362109223E-01
     41.6400                 0.1120007750E+01
     18.2600                -0.3184257719E+00
      8.2510                -0.3487967753E+01
      3.7120                 0.1152949189E+02
      2.7500                -0.1224194516E+02
      1.6160                 0.5599768469E+01
      0.6891                -0.4511020807E+01
      0.2875                 0.4327293711E+01
      0.1144                -0.2158919599E+01
O    P
 254500.0000                 0.1464517883E-02
  38080.0000                -0.3195187333E-03
   8262.0000                -0.5093018808E-02
   2405.0000                 0.2563643342E-01
    764.0000                 0.4736412537E-01
    266.3000                 0.8738330666E-01
    101.0000                 0.1767823273E+00
     41.6400                 0.3152427665E+00
     18.2600                 0.3290496951E+00
      8.2510                 0.4045123637E+00
      3.7120                -0.4413381608E+00
      1.6160                -0.8527320681E+00
      0.6891                 0.1678173221E+01
      0.4900                -0.7118927920E+00
      0.2875                -0.1478046585E+00
      0.1144                -0.1524693649E+00
O    P
 254500.0000                 0.8464786787E-03
  38080.0000                -0.6124326698E-01
   8262.0000                -0.1785199711E+00
   2405.0000                -0.3710746728E-01
    764.0000                -0.9496267534E-01
    266.3000                -0.4170000812E+00
    101.0000                -0.7178597260E-01
     41.6400                -0.3372370850E+00
     18.2600                 0.6820688623E+00
      8.2510                -0.1142911203E+01
      3.7120                 0.3017789632E+01
      1.6160                -0.5211713920E+01
      0.6891                 0.9647084612E+01
      0.4900                -0.8265950360E+01
      0.2875                 0.1919970426E+01
      0.1144                -0.3138710517E+00
O    P
 254500.0000                 0.5736038387E-08
  38080.0000                 0.2261262507E-08
   8262.0000                -0.1512299893E-07
   2405.0000                 0.3726681515E-07
    764.0000                -0.1564432948E-06
    266.3000                -0.4193307361E-03
    101.0000                -0.1389041542E-02
     41.6400                -0.5982370080E-02
     18.2600                -0.1896364764E-01
      8.2510                -0.5498688154E-01
      3.7120                -0.1427690185E+00
      1.6160                -0.2762392921E+00
      0.6891                -0.3571278628E+00
      0.4900                -0.7622990963E-06
      0.2875                -0.3016722503E+00
      0.1144                -0.1073931410E+00
O    P
 254500.0000                 0.1995681974E-03
  38080.0000                 0.3750469579E-01
   8262.0000                 0.8157890813E-01
   2405.0000                 0.7568031584E-01
    764.0000                 0.2097560846E-03
    266.3000                 0.4979027205E+00
    101.0000                -0.4307054502E+00
     41.6400                 0.1236286423E+01
     18.2600                -0.3591788226E+01
      8.2510                 0.4491108668E+01
      3.7120                -0.3146804510E+01
      1.6160                 0.2269857674E+01
      0.6891                -0.7520126074E+01
      0.4900                 0.1036876774E+02
      0.2875                -0.4920642125E+01
      0.1144                 0.9559739007E+00
O    P
 254500.0000                -0.4398135884E-02
  38080.0000                 0.3631906116E-01
   8262.0000                 0.1110688256E+00
   2405.0000                -0.4051831599E-01
    764.0000                -0.1000282702E+00
    266.3000                 0.8502586560E-01
    101.0000                -0.4734740906E+00
     41.6400                 0.7126304360E-01
     18.2600                -0.7090129274E+00
      8.2510                 0.3172399707E+01
      3.7120                -0.4803511138E+01
      1.6160                 0.4507463431E+01
      0.6891                -0.5471399925E+01
      0.4900                 0.5020841568E+01
      0.2875                -0.2260802741E+01
      0.1144                 0.8849131287E+00
O    P
 254500.0000                 0.7579190856E-08
  38080.0000                -0.1873498953E-07
   8262.0000                 0.3405576633E-07
   2405.0000                -0.1467497765E-06
    764.0000                 0.3866388706E-06
    266.3000                 0.9842668178E-03
    101.0000                 0.5232239534E-02
     41.6400                 0.1444007188E-01
     18.2600                 0.7686501222E-01
      8.2510                 0.1839277008E+00
      3.7120                 0.7654754728E+00
      1.6160                -0.2205540232E+00
      0.6891                -0.1170480871E+01
      0.4900                 0.2055817009E-05
      0.2875                 0.6162539512E+00
      0.1144                 0.3457744920E+00
O    P
 254500.0000                 0.6117155840E-09
  38080.0000                 0.1415281288E-07
   8262.0000                -0.1682052160E-07
   2405.0000                 0.5422966990E-07
    764.0000                -0.2010167567E-06
    266.3000                -0.6092876751E-03
    101.0000                -0.2017002001E-02
     41.6400                -0.8832755340E-02
     18.2600                -0.2805700837E-01
      8.2510                -0.8080967534E-01
      3.7120                -0.2529832338E+00
      1.6160                -0.5528526481E+00
      0.6891                 0.1322448113E-01
      0.4900                -0.1492336226E-05
      0.2875                 0.6411002013E+00
      0.1144                 0.2610894111E+00
O    P
 254500.0000                 0.5155555401E-02
  38080.0000                -0.6253758097E-01
   8262.0000                -0.2053844000E+00
   2405.0000                 0.5417315112E-01
    764.0000                 0.1549688713E-01
    266.3000                -0.1115909330E+00
    101.0000                 0.1556719905E+00
     41.6400                 0.5400219216E+00
     18.2600                -0.4047841544E+00
      8.2510                -0.8539330303E+00
      3.7120                 0.1963417717E+01
      1.6160                -0.3460452431E+01
      0.6891                 0.9319598008E+01
      0.4900                -0.8480319166E+01
      0.2875                 0.5859192433E+00
      0.1144                 0.1141842327E+01
O    D
    266.3000                 0.9539564585E-08
    101.0000                -0.3149912838E-07
     41.6400                 0.4758652787E-07
     18.2600                -0.5729187661E-07
      8.2510                 0.7208710746E-07
      3.7120                -0.1516580209E+00
      1.6160                -0.4130112296E+00
      0.6891                -0.4832104913E+00
      0.2875                -0.1247064026E+00
      0.1144                -0.1398646658E-07
O    D
    266.3000                -0.1879620930E-07
    101.0000                 0.4560342057E-08
     41.6400                -0.4814995336E-07
     18.2600                 0.4053387664E-07
      8.2510                -0.2243849784E-07
      3.7120                 0.5033767078E+00
      1.6160                 0.5566730803E+00
      0.6891                -0.6282452079E+00
      0.2875                -0.4000645363E+00
      0.1144                -0.1963220125E-07
O    D
    266.3000                -0.3766168592E-01
    101.0000                -0.7738611412E-01
     41.6400                -0.1955673852E+00
     18.2600                -0.4646271860E+00
      8.2510                -0.4688041503E+00
      3.7120                 0.4880594015E+00
      1.6160                -0.3382468137E+00
      0.6891                 0.5269034069E+00
      0.2875                -0.2844335123E+00
      0.1144                -0.3632182685E+00
O    D
    266.3000                -0.2628291795E-01
    101.0000                -0.4689196110E-01
     41.6400                -0.1423224070E+00
     18.2600                -0.2771691874E+00
      8.2510                -0.3578784305E+00
      3.7120                 0.1107511166E+01
      1.6160                -0.4431743935E+00
      0.6891                -0.7308208468E+00
      0.2875                 0.7438026557E+00
      0.1144                 0.3039855269E+00
O    D
    266.3000                -0.2174006100E-01
    101.0000                -0.1911990081E-01
     41.6400                -0.9445361415E-01
     18.2600                -0.9857295383E-01
      8.2510                 0.2355952126E+00
      3.7120                 0.1181020610E+01
      1.6160                -0.1901201007E+01
      0.6891                 0.1085511428E+01
      0.2875                 0.1949541951E+00
      0.1144                -0.6827839617E+00
O    F
      3.7120                 0.1707387097E+00
      1.6160                 0.5350159884E+00
      0.6891                 0.4480026423E+00
      0.2875                -0.3510715724E-07
O    F
      3.7120                -0.7263082968E+00
      1.6160                -0.2251714344E+00
      0.6891                 0.7309588870E+00
      0.2875                 0.1467783098E+00
O    F
      3.7120                 0.7850506594E+00
      1.6160                -0.7311789324E+00
      0.6891                -0.2728005070E+00
      0.2875                 0.9996613477E+00
O    G
      3.7120                 0.2836837003E+00
      1.6160                 0.5865124451E+00
      0.6891                 0.3207201559E+00
END
"""


@pytest.fixture(scope="module")
def get_sigmanzhf_basisset():
    def get_basis(mol: gto.Mole) -> dict:
        atom_symb = [mol.atom_pure_symbol(i) for i in range(mol.natm)]
        atom_symb = set(atom_symb)

        return {key: gto.basis.parse_nwchem.parse(SIGMA_TZHF, symb=key) for key in atom_symb}

    return get_basis


@pytest.fixture(scope="module", params=[(1, 2, 3, 4)])
def energies(molecule_medium, request):
    """Returns an Energies object for each given molecule."""
    kinetic, hartree, xc, nuclear_attraction = request.param
    return Energies(
        kinetic=kinetic,
        hartree=hartree,
        xc=xc,
        nuclear_attraction=nuclear_attraction,
        mol=molecule_medium,
    )


@pytest.fixture(scope="module")
def basis_info():
    """Returns a BasisInfo object for the 6-31G basis set."""
    basis_name = "6-31G"
    atomic_numbers = list(range(1, 30))
    elements = [gto.param.ELEMENTS[atomic_number] for atomic_number in atomic_numbers]
    nwchem_string = basis_set_exchange.api.get_basis(basis_name, elements=elements, fmt="nwchem")
    basis_info = BasisInfo.from_nwchem(nwchem_string)
    return basis_info


@pytest.fixture(scope="module")
def ofdata_three_atoms_and_more(molecule_three_atoms_and_more, basis_info):
    """Returns an OFData object for molecules with more than three atoms containing a BasisInfo
    object."""
    mol = molecule_three_atoms_and_more
    pos = mol.atom_coords()
    atomic_numbers = mol.atom_charges()
    coeffs = np.random.rand(mol.nao)
    ofdata = OFData.construct_new(
        basis_info,
        pos,
        atomic_numbers,
        coeffs,
        add_irreps=True,
        dual_basis_integrals="infer_from_basis",
    )
    # basis transformation tests require float64 to be accurate enough
    ofdata = to_torch(ofdata, float_dtype=torch.float64)
    return ofdata


@pytest.fixture(scope="module")
def ofdata_single(basis_info):
    """Returns an OFData object for a single molecule containing a BasisInfo object."""
    # Ethanol molecule (C2H5OH)
    mol = gto.M(
        atom="""    H      1.8853     -0.0401      1.0854
                    C      1.2699     -0.0477      0.1772
                    H      1.5840      0.8007     -0.4449
                    H      1.5089     -0.9636     -0.3791
                    C     -0.2033      0.0282      0.5345
                    H     -0.4993     -0.8287      1.1714
                    H     -0.4235      0.9513      1.1064
                    O     -0.9394      0.0157     -0.6674
                    H     -1.8540      0.0626     -0.4252"""
    )
    pos = mol.atom_coords()
    atomic_numbers = mol.atom_charges()
    coeffs = np.random.rand(mol.nao)
    ofdata = OFData.construct_new(
        basis_info,
        pos,
        atomic_numbers,
        coeffs,
        add_irreps=True,
        dual_basis_integrals="infer_from_basis",
    )
    ofdata.coeffs = np.random.rand(ofdata.n_basis_per_atom.sum())
    # basis transformation tests require float64 to be accurate enough
    ofdata = to_torch(ofdata, float_dtype=torch.float64)
    return ofdata
