function [nodes, weights] = exact_quadrature_parameters(h, order)
% Return exact quadrature weights for the interval [0,h] for certain
% rules.
%
% order = 29,13,9,7,5,3  gives Gauss quadrature of specified order
% order = 'rectangle', 'midpoint' or 'trapezoidal' give those 2nd,
% 3rd and 3rd-order rules, respectively.

%
% This file is part of the M-M.E.S.S. project
% (http://www.mpi-magdeburg.mpg.de/projects/mess).
% Copyright (c) 2009-2025 Jens Saak, Martin Koehler, Peter Benner and others.
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%

switch order

    case 29

        xi0 = [ ...
               -0.10805494870734366206624465021983; ...
               0.10805494870734366206624465021983; ...
               -0.31911236892788976043567182416847; ...
               0.31911236892788976043567182416847; ...
               -0.51524863635815409196529071855118; ...
               0.51524863635815409196529071855118; ...
               -0.68729290481168547014801980301933; ...
               0.68729290481168547014801980301933; ...
               -0.82720131506976499318979474265039; ...
               0.82720131506976499318979474265039; ...
               -0.92843488366357351733639113937787; ...
               0.92843488366357351733639113937787; ...
               -0.98628380869681233884159726670405; ...
               0.98628380869681233884159726670405];

        weights0 = [ ...
                    0.21526385346315779019587644331626; ...
                    0.21526385346315779019587644331626; ...
                    0.20519846372129560396592406566121; ...
                    0.20519846372129560396592406566121; ...
                    0.18553839747793781374171659012515; ...
                    0.18553839747793781374171659012515; ...
                    0.15720316715819353456960193862384; ...
                    0.15720316715819353456960193862384; ...
                    0.12151857068790318468941480907247; ...
                    0.12151857068790318468941480907247; ...
                    0.08015808715976020980563327706285; ...
                    0.08015808715976020980563327706285; ...
                    0.03511946033175186303183287613819; ...
                    0.03511946033175186303183287613819];

    case 13

        xi0 = [ ...
               0.66120938646626451366139959501990; ...
               -0.66120938646626451366139959501990; ...
               -0.23861918608319690863050172168071; ...
               0.23861918608319690863050172168071; ...
               -0.93246951420315202781230155449399; ...
               0.93246951420315202781230155449399];

        weights0 = [ ...
                    0.36076157304813860756983351383771; ...
                    0.36076157304813860756983351383771; ...
                    0.46791393457269104738987034398955; ...
                    0.46791393457269104738987034398955; ...
                    0.17132449237917034504029614217273; ...
                    0.17132449237917034504029614217273];

    case 9

        xi0 = [ ...
               -0.33998104358485626480266575910324; ...
               0.33998104358485626480266575910324; ...
               -0.86113631159405257522394648889280; ...
               0.86113631159405257522394648889280];

        weights0 = [ ...
                    0.65214515486254614262693605077800; ...
                    0.65214515486254614262693605077800; ...
                    0.34785484513745385737306394922199; ...
                    0.34785484513745385737306394922199];

    case 7

        xi0 = [ ...
               0; ...
               -0.77459666924148337703585307995647; ...
               0.77459666924148337703585307995647];

        weights0 = [ ...
                    0.88888888888888888888888888888888; ...
                    0.55555555555555555555555555555555; ...
                    0.55555555555555555555555555555555];

    case 5

        xi0 = [ ...
               -0.57735026918962576450914878050195; ...
               0.57735026918962576450914878050195];

        weights0 = [1; 1];

    case 3

        xi0 = 0;

        weights0 = 2;

    case 'midpoint'

        xi0 = 0;

        weights0 = 2;

    case 'trapezoidal'

        xi0 = [-1; 1];

        weights0 = [1; 1];

    case 'rectangle'

        xi0 = 1;

        weights0 = 2;

    otherwise

        mess_err(opts, 'exact_quadrature_parameters:Unknown order');
end

% Conversion from [-1, 1] to [0, h]:
weights = weights0 * h / 2;
nodes = h / 2 * (xi0 + 1);

% Sort nodes in ascending order to allow for iterative computing in our
% particular use-case

[nodes, ind] = sort(nodes, 'ascend');
weights = weights(ind);
end
