function eqn = mess_get_bilinear_rail(k)
%% function eqn = mess_get_bilinear_rail(k) reads in data for rail example
% s.a. FenicsRail model hosted at MORWiki
% (https://modelreduction.org)
%
% DESCRIPTION:
%   This function extracts the mat-files to get the bilinear steel rail
%   model described by
%
%       E*x' = A*x + B*u + N{1}*x*u_1 + ... + N{6}*x*u_6,              (1a)
%          y = C*x,                                                    (1b)
%
%   for different refinement sizes. Analogous to the Oberwolfach
%   Collection steel profile the model features 7 inputs and 6 outputs.
%
% Input:
%   k       number of instance
%           k = 0 -> n =       109
%           k = 1 -> n =       371
%           k = 2 -> n =     1,357
%           k = 3 -> n =     5,177
%           k = 4 -> n =    20,209
%           k = 5 -> n =    79,841
%           k = 6 -> n =   317,377
%           k = 7 -> n = 1,265,537
%           k = 8 -> n = 5,054,209
%
% Output:
% eqn       structure for use with the 'default' usfs with members
%   E_      Sparse Matrix
%   A_      Sparse Matrix
%   B       Dense Matrix
%   C       Dense Matrix
%   N       cell array of bilinear matrices of dimensions n x n from (1a),
%           note that the seventh bilinear term is zero
%   haveE   indicates that E is not the identity
%
% Data generated by:
%   https://gitlab.mpi-magdeburg.mpg.de/models/fenicsrail
%
% REFERENCES:
%  [1] J. Saak, Efficient numerical solution of large scale
%      algebraic matrix equations in PDE control and model order
%      reduction, Dissertation, Technische Universität Chemnitz,
%      Chemnitz, Germany (Jul. 2009).
%      URL http://nbn-resolving.de/urn:nbn:de:bsz:ch1-200901642
%  [2] J. Saak, Effiziente numerische Lösung eines
%      Optimalsteuerungsproblems für die Abkühlung von Stahlprofilen,
%      Diplomarbeit, Fachbereich 3/Mathematik und Informatik,
%      Universität Bremen, D-28334 Bremen (Sep. 2003).
%  [3] P. Benner, J. Saak, A semi-discretized heat transfer model
%      for optimal cooling of steel profiles, in: P. Benner,
%      V. Mehrmann, D. Sorensen (Eds.), Dimension Reduction of
%      Large-Scale Systems, Vol. 45 of Lecture Notes in Computational Science and Engineering,
%      Springer-Verlag, Berlin/Heidelberg, Germany, 2005,
%      pp. 353–356. https://doi.org/10.1007/3-540-27909-1_19.

%
% This file is part of the M-M.E.S.S. project
% (http://www.mpi-magdeburg.mpg.de/projects/mess).
% Copyright (c) 2009-2025 Jens Saak, Martin Koehler, Peter Benner and others.
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%

% Model Parameters
lambda = 26.4;
c = 7620.0;
rho = 654.0;
gamma_k = 7.0164;

u_ext = 0.02;

alpha = lambda / (c * rho);
robin = 1.0 / (c * rho);

data = mess_load_rail(k);

eqn.E_ = data.M;

eqn.A_ = -(alpha * data.S + gamma_k * robin * data.M_GAMMA_6);

eqn.N_ = {-robin * data.M_GAMMA_0, ...
          -robin * data.M_GAMMA_1, ...
          -robin * data.M_GAMMA_2, ...
          -robin * data.M_GAMMA_3, ...
          -robin * data.M_GAMMA_4, ...
          -robin * data.M_GAMMA_5, ...
          sparse(size(eqn.A_, 1), size(eqn.A_, 1))};

eqn.B = robin * [u_ext * data.B_0', ...
                 u_ext * data.B_1', ...
                 u_ext * data.B_2', ...
                 u_ext * data.B_3', ...
                 u_ext * data.B_4', ...
                 u_ext * data.B_5', ...
                 gamma_k * data.B_6'];

eqn.C                      = zeros(6, size(eqn.A_, 1));
eqn.C(1, [4, 22, 60])      = [-1, -1, 3];
eqn.C(2, [2, 3, 63])       = [-1, -1, 2];
eqn.C(3, [43, 51])         = [-1, 1];
eqn.C(4, [47, 55])         = [-1, 1];
eqn.C(5, [9, 16, 92])      = [-1, -1, 2];
eqn.C(6, [10, 15, 34, 83]) = [-1, -1, -1, 3];

eqn.haveE = true;
