function Lyapunov_rail_LDL_ADI(k, shifts, implicit, istest)
% Computes the solution of the generalized Lyapunov equation via the
% low-rank ADI iteration in both ZZ^T [1,2] and LDL^T[3] formulation for the
% selective cooling of Steel profiles application described in [4,5,6].
%
% Usage:      Lyapunov_rail_LDL_ADI(k,shifts,istest)
%
% Inputs:
%
% k           refinement level of the model to use
%             (0 - 5, i.e. 109 - 79841 Dofs)
%             (optional, defaults to 2, i.e. 1357 Dofs)
%
% shifts      ADI shift selection strategy. Possible values:
%              'heur'        Penzl's heuristic shifts
%              'Wachspress'  Wachspress shifts, optimally solving the dense
%                            shift selection problem.
%              'projection'  projection shifts [7]
%             (optional, defaults to 'heur')
%
% implicit    projection shifts with implicit computation of projected A
%             matrix
%
% istest      flag to determine whether this demo runs as a CI test or
%             interactive demo
%             (optional, defaults to 0, i.e. interactive demo)
%
% References:
%
% [1] P. Benner, J.-R. Li, T. Penzl, Numerical solution of large-scale
%     Lyapunov equations, Riccati equations, and linear-quadratic optimal
%     control problems, Numer. Lin. Alg. Appl. 15 (9) (2008) 755–777.
%     https://doi.org/10.1002/nla.622
%
% [2] P. Kürschner, Efficient low-rank solution of large-scale matrix
%     equations, Dissertation, Otto-von-Guericke-Universität, Magdeburg,
%     Germany, shaker Verlag, ISBN 978-3-8440-4385-3 (Apr. 2016).
%     URL http://hdl.handle.net/11858/00-001M-0000-0029-CE18-2
%
% [3] N. Lang, H. Mena, J. Saak, On the benefits of the LDLT factorization
%     for large-scale differential matrix equation solvers, Linear Algebra
%     Appl. 480 (2015) 44–71.
%     https://doi.org/10.1016/j.laa.2015.04.006
%
% [4] J. Saak, Effiziente numerische Lösung eines
%     Optimalsteuerungsproblems für die Abkühlung von Stahlprofilen,
%     Diplomarbeit, Fachbereich 3/Mathematik und Informatik, Universität
%     Bremen, D-28334 Bremen (Sep. 2003).
%     https://doi.org/10.5281/zenodo.1187040
%
% [5] P. Benner, J. Saak, A semi-discretized heat transfer model for
%     optimal cooling of steel profiles, in: P. Benner, V. Mehrmann, D.
%     Sorensen (Eds.), Dimension Reduction of Large-Scale Systems, Vol. 45
%     of Lecture Notes in Computational Science and Engineering,
%     Springer-Verlag, Berlin/Heidelberg, Germany, 2005, pp. 353–356.
%     https://doi.org/10.1007/3-540-27909-1_19
%
% [6] J. Saak, Efficient numerical solution of large scale algebraic matrix
%     equations in PDE control and model order reduction, Dissertation,
%     Technische Universität Chemnitz, Chemnitz, Germany (Jul. 2009).
%     URL http://nbn-resolving.de/urn:nbn:de:bsz:ch1-200901642
%
% [7] P. Benner, P. Kürschner, J. Saak, Self-generating and efficient shift
%     parameters in ADI methods for large Lyapunov and Sylvester equations,
%     Electron. Trans. Numer. Anal. 43 (2014) 142–162.
%     URL http://etna.mcs.kent.edu/volumes/2011-2020/vol43/abstract.php?vol=43&pages=142-162

%
% This file is part of the M-M.E.S.S. project
% (http://www.mpi-magdeburg.mpg.de/projects/mess).
% Copyright (c) 2009-2025 Jens Saak, Martin Koehler, Peter Benner and others.
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%

%%
narginchk(0, 4);
if nargin < 1
    k = 2;
end
if nargin < 2
    shifts = 'wachspress';
end
if nargin < 3
    implicit = 0;
end
if nargin < 4
    istest = false;
end
%%
% set operation
opts = struct();
[oper, opts] = operatormanager(opts, 'default');
% Problem data
eqn = mess_get_linear_rail(k);
%%

% ADI tolerances and maximum iteration number
opts.adi.maxiter      = 100;
opts.adi.res_tol      = 1e-12;
opts.adi.rel_diff_tol = 1e-16;
opts.adi.info         = 1;

eqn.type = 'T';

%%
% Heuristic shift parameters via basic Arnoldi
n = oper.size(eqn, opts);
opts.shifts.num_Ritz    = 50;
opts.shifts.num_hRitz   = 25;
opts.shifts.num_desired = 6;

opts.shifts.b0 = ones(n, 1);
switch lower(shifts)

    case 'heur'
        opts.shifts.method       = 'heur';

    case 'wachspress'
        opts.shifts.method       = 'wachspress';
        opts.shifts.wachspress   = 'T';

    case 'projection'
        opts.shifts.method       = 'projection';
        opts.shifts.implicitVtAV = implicit;
end
opts.norm = 'fro';

%%
mess_fprintf(opts, '########################\n');
mess_fprintf(opts, '# ADI with ZZ^T:        \n');
mess_fprintf(opts, '########################\n');
t_mess_lradi = tic;
out = mess_lradi(eqn, opts, oper);
t_elapsed1 = toc(t_mess_lradi);
mess_fprintf(opts, 'mess_lradi took %6.2f seconds \n', t_elapsed1);

if istest
    if min(out.res) >= opts.adi.res_tol
        mess_err(opts, 'TEST:accuracy', ...
                 'unexpectedly inaccurate result');
    end
else
    figure(1);
    semilogy(out.res, 'LineWidth', 3);
    xlabel('number of iterations');
    ylabel('normalized residual norm');
    pause(1);
end
[mZ, nZ] = size(out.Z);
mess_fprintf(opts, 'size out.Z: %d x %d\n\n', mZ, nZ);

%% Set LDL fields
opts.LDL_T = true;
eqn.T = diag([4, 4, 9, 9, 16, 16]);
eqn.W = eqn.C' * diag([4, 4, 9, 9, 16, 16].^(-0.5));

%%
mess_fprintf(opts, '########################\n');
mess_fprintf(opts, '# ADI with LDL^T:       \n');
mess_fprintf(opts, '########################\n');
t_mess_lradi = tic;
out1 = mess_lradi(eqn, opts, oper);
t_elapsed2 = toc(t_mess_lradi);
mess_fprintf(opts, 'mess_lradi took %6.2f seconds \n', t_elapsed2);

if istest
    if min(out.res) >= opts.adi.res_tol
        mess_err(opts, 'TEST:accuracy', ...
                 'unexpectedly inaccurate result');
    end
else
    figure(2);
    semilogy(out1.res, 'LineWidth', 3);
    xlabel('number of iterations');
    ylabel('normalized residual norm');
    pause(1);
end
[mZ, nZ] = size(out1.Z);
mess_fprintf(opts, 'size out1.Z: %d x %d\n\n', mZ, nZ);

%% Difference of Lyapunov solutions
if k < 3
    % This is mainly for consistency checking on our continuous
    % integration tests.
    % NEVER FORM SUCH DYADIC PRODUCTS IN PRODUCTION CODE!!!
    err = norm(out.Z * out.Z' - out1.Z * out1.D * out1.Z') / ...
          norm(out.Z * out.Z');
    mess_fprintf(opts, ...
                 ['Relative difference between solution with and without ' ...
                  'LDL^T: \t %g\n'], ...
                 err);
    if err > 1e-11
        if implicit
            shifts = [shifts '(implicit)'];
        end
        mess_err(opts, 'TEST:accuracy', ...
                 ['unexpectedly inaccurate result relative difference', ...
                  ' %e > 1e-12 in case %s'], ...
                 err, shifts);
    end
end
