## Manuscript "Trends in global biodiversity scenarios" - February 2025
## Aschi et al.
## Descriptive statistics of identified scenario literature
library(ggplot2)
library(readr)
library(dplyr)
library(ggpubr)
library(forcats)
library(readxl)

# Plot Sceanrio Type------------------------------------------
#Load in Scenario Type
ScenarioType <- read_csv("Metadata/Scenario_type/scenario_type.csv")

# Define the desired order of levels
class_order <- c("Bending the Curve", "Zero Net Loss", "Reduce Loss", "Baseline", "Worst Case", "Unclassified")

#Plot Sceanrio Type
ggplot(ScenarioType, aes(x = number, y = factor(scenario_type, levels = class_order), fill = scenario_type)) +
  geom_bar(width = .7, stat ="identity")+
  scale_fill_manual(values = c("Bending the Curve" = "darkolivegreen4", 
                               "Net Zero Loss" = "lightblue", 
                               "Reduce Loss" = "lightseagreen", 
                               "Baseline" = "goldenrod1", 
                               "Worst Case" = "#D55E00",
                               "Unclassified" = "lightgrey")) +
  geom_text(aes(label=number), size = 3.5, position = position_stack(vjust = 0.5), colour = "black")+
  scale_x_continuous(breaks=seq(0, 110, 50)) +
  theme_bw() +
  theme(aspect.ratio = 2/7)+
  theme(axis.line = element_line(size=0.3),
        axis.ticks = element_line(size=0.3),
        plot.background = element_blank(),
        panel.grid.major = element_blank(),
        panel.grid.minor = element_blank(),
        panel.border = element_blank()) +
  theme(legend.position = "none") +
  theme(axis.title.y = element_blank(),
        axis.ticks.y = element_blank(),
        axis.text.y = element_text(size = 11))+
  theme(axis.title.x = element_text(vjust=-1.5))+
  ggtitle("Scenario Type") +
  xlab ("Number of Scenarios")


#-------------------------------------------
# Plot publication per year per scenario type
Scen_peryear<- read_csv("Metadata/Scenario_type/scenario_peryear.csv")

# Reshape the data for ggplot
df_long <- tidyr::pivot_longer(Scen_peryear, 
                               cols = -c(year, record_id), names_to = "Category", values_to = "Count")%>%
  na.omit() 

# Define the order of the categories
category_order <- c("Bending the Curve", "Zero Net Loss", "Reduce Loss", "Baseline", "Worst Case", "Unclassified")

# Convert "Category" to a factor with default levels
df_long$Category <- fct_relevel(df_long$Category, category_order)

# Create the plot
ggplot(df_long, aes(x = Count, y = factor(year), fill = Category)) +
  geom_bar(stat = "identity", position = "stack") +
  scale_fill_manual(values = c("Bending the Curve" = "darkolivegreen4", 
                               "Zero Net Loss" = "lightblue", 
                               "Reduce Loss" = "lightseagreen", 
                               "Baseline" = "goldenrod1", 
                               "Worst Case" = "#D55E00",
                               "Unclassified" = "lightgrey")) +
  labs(x = "Number of Scenarios", y = "Year") +
  theme_bw() +
  theme(axis.line = element_line(size=0.3),
        axis.ticks = element_line(size=0.3),
        plot.background = element_blank(),
        panel.grid.major = element_blank(),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(),
        legend.key.size = unit(0.3, "cm"))+
  ggtitle("Scenarios per year by scenario type") +
  theme(legend.position = "right") +
  theme(axis.title.x = element_text(vjust=-1.5))+
  theme(axis.title.y = element_text(vjust=2))+
  guides(fill = guide_legend(title = "Category")) +
  theme(aspect.ratio = 0.6)

#-------------------------------------------
# Plot publication per year 
Scen_peryear<- read_csv("Metadata/Scenario_type/YearPublication.csv")


# Aggregate the data to get the count of record_id per year
Scen_peryear_counts <- Scen_peryear %>% group_by(year) %>% summarize(count = n_distinct(record_id))

# Create a histogram of the count of record_id per year
ggplot(Scen_peryear_counts, aes(x = factor(year), y = count, fill = factor(year))) +
  geom_col(fill = "darkolivegreen", width = .4) +
  coord_flip()+
  labs(x = "Year", y = "Papers", fill = "Year") +
  theme_minimal() +
  theme(legend.position = "none")+
  theme(axis.line = element_line(size=0.2),
        axis.ticks = element_line(size=0.3),
        plot.background = element_blank(),
        panel.grid.major = element_blank(),
        panel.grid.minor = element_blank(),
        panel.border = element_blank())+
  scale_y_continuous(breaks = seq(0,10,1)) +
  ggtitle("Papers published per year (n = 57)") +
  theme(axis.title.y = element_text(vjust=2))+
  theme(axis.title.x = element_text(vjust=-1.5))+
  theme(aspect.ratio = 0.6)

#-------------------------------------------
# Plot BDMs per # papers

BDMs<- read_csv("Metadata/BDMs/BDM_contraction.csv")


# Reshape the data for ggplot
BDMs_long <- tidyr::pivot_longer(BDMs, cols = -c(record_id, ), names_to = "BDMs", values_to = "Count") %>% na.omit()

# Create the bar plot
ggplot(BDMs_long, aes(x = BDMs, y = Count, fill = BDMs)) +
  geom_col(fill = "darkolivegreen", width = .4) +
  coord_flip()+
  labs(title = "Biodiversity Models (tot n = 21)", x = "Biodiversity Models", y = "Count") +
  labs(x = "BDMs", y = "Papers", fill = "BDMs") +
  theme_minimal() +
  theme(legend.position = "none")+
  theme(axis.line = element_line(size=0.3),
        axis.ticks = element_line(size=0.3),
        plot.background = element_blank(),
        panel.grid.major = element_blank(),
        panel.grid.minor = element_blank(),
        panel.border = element_blank())+
  scale_y_continuous(breaks = seq(0,12,2)) +
  theme(axis.title.y = element_text(vjust=2))+
  theme(axis.title.x = element_text(vjust=-1.5))+
  theme(aspect.ratio = 0.6)





#-------------------------------------------
# PlotModels per # papers

Models<- read_csv("Metadata/Models/Models.csv")

# OtherModels<-na.omit(OtherModels)
# Reshape the data for ggplot
Models_long <- tidyr::pivot_longer(Models, cols = -c(record_id, ), names_to = "Models", values_to = "Count") %>% na.omit()


# Create the bar plot

ggplot(Models_long, aes(x= reorder(Models, -Count), y = Count, fill = Models)) +
  geom_col(fill = "darkolivegreen", width = .3) +
  coord_flip()+
  labs(title = "Top 5 models and others (tot n = 33)", x = "Models", y = "Count") +
  labs(x = "Models", y = "Papers", fill = "BDMs") +
  theme_minimal() +
  theme(legend.position = "none")+
  theme(axis.line = element_line(size=0.3),
        axis.ticks = element_line(size=0.3),
        plot.background = element_blank(),
        panel.grid.major = element_blank(),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(), 
        axis.text.y = element_text(size = 8))+
  scale_y_continuous(breaks = seq(0,44,2)) +
  theme(axis.title.y = element_text(vjust=2))+
  theme(axis.title.x = element_text(vjust=-1.5))+
  theme(aspect.ratio = 0.6)

#-------------------------------------------
# Plot BDI per # papers

bdi<-read_csv("Metadata/BDIs/BDI_plots.csv")

# Reshape the data for ggplot
bdi_long <- tidyr::pivot_longer(bdi, cols = -c(Record_id, Title, Authors, Year), names_to = "BDI", values_to = "Count") %>%
  na.omit() 

# Create the bar plot

ggplot(bdi_long, aes(x= reorder(BDI, -Count), y = Count, fill = BDI)) +
  geom_col(fill = "darkolivegreen", width = .3) +
  coord_flip()+
  labs(title = "Biodiversity Indicators (tot n = 35)", x = "BDIs", y = "Count") +
  labs(x = "BDIs", y = "Papers", fill = "BDIs") +
  theme_minimal() +
  theme(legend.position = "none")+
  theme(axis.line = element_line(size=0.3),
        axis.ticks = element_line(size=0.3),
        plot.background = element_blank(),
        panel.grid.major = element_blank(),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(), 
        axis.text.y = element_text(size = 8))+
  scale_y_continuous(breaks = seq(0,40,2)) +
  theme(axis.title.y = element_text(vjust=2))+
  theme(axis.title.x = element_text(vjust=-1.5))

#-------------------------------------------
# Plot Drivers per # papers

Drivers<- read_csv("Metadata/Drivers/drivers.csv")

# Reshape the data for ggplot
Drivers_long <- tidyr::pivot_longer(Drivers, cols = -c(record_id, year), names_to = "drivers", values_to = "Count") %>%
  na.omit()

drivers_order <- c("LUC", "CC", 
                   "Fragmentation", "Road Disturbance", 
                   "Hunitng", "Nitrogen Deposition",
                   "Nutrient Emission to Water", "Hydrological Disturbance",
                   "Encroachment", "Infastructure Development",
                   "Natural Habitat & Wetland Conversion", "Biomass Harvest","Urban Land Expasion",
                   "Human Population Density",
                   "Pollution")

# Create the bar plot
ggplot(Drivers_long, aes(x = reorder(drivers, -Count), y = Count, fill = drivers)) +
  geom_col(fill = "darkolivegreen", width = .3) +
  coord_flip()+
  labs(title = "Drivers of Biodiversity Loss (tot n = 15)", x = "Models", y = "Count") +
  labs(x = "Drivers", y = "Papers", fill = "Drivers") +
  theme_minimal() +
  theme(legend.position = "none")+
  theme(axis.line = element_line(size=0.3),
        axis.ticks = element_line(size=0.3),
        plot.background = element_blank(),
        panel.grid.major = element_blank(),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(),
        axis.text.y = element_text(size = 8))+
  scale_y_continuous(breaks = seq(0,44,2)) +
  scale_x_discrete(limits = drivers_order) +
  theme(axis.title.y = element_text(vjust=2))+
  theme(axis.title.x = element_text(vjust=-1.5))+
  theme(aspect.ratio = 0.6)

