% This file performs a dynamic analysis of a SDOF
% by making use of the Newmark integration algorithm (from DT)

function  [U_dyn, U_d, U_dd, F_stat, F_dyn, time_num, TMP_failure_label, i_end] = integration_SDOF( ...
               U_dd_ground, u0, ud0, udd0, time_gm, m, psi, model_param)

        %% model properties

        %model parameters
        d_1 = model_param(1,1);
        d_2 = model_param(2,1);
        d_u = model_param(3,1);
        f_1 = model_param(4,1);
        f_0 = model_param(5,1);

        %compute initial stiffness
        kini   =  f_1/d_1;

        %fix time integration step, can be different from time of the ground motion
        dt     = 0.001; %min([d_1/2.,(d_2-d_1)/2.,(d_u-d_2)/4.]);

        Solver = 1; % gamma=1/2,beta=1/4 constant acceleration; gamma=1/2,beta=1/6 linear acceleration

        %%  INTEGRATE EQUATIONS OF MOTION

        % initial conditions of the system
        U_dyn(1) = u0;    % initial displacement
        U_d(1)   = ud0;    % Initial velocity
        U_dd(1)  = udd0;    % initial accelerations

        % Solution algorithm, Chopra's book at page 174 (Newmark's Method)
        % 1) 'Newton Average Acceleration', 2)'Newton Linear Acceleration'
        switch Solver
            case 1
                gamma = 1/2;
                beta  = 1/4;
            case 2
                gamma = 1/2;
                beta  = 1/6;
            otherwise
                error('The selected solution algorithm does not exist')
        end

        %% INITIAL CALCULATIONS
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % 1) numerical time history, according to the user defined dt

        record_dt = time_gm(2) - time_gm(1);        %dt of the record
        num_div_Udd_ground = ceil(record_dt/dt);    %number of points to introduce between each input point
        %(both concerning time and acceleration)

        if num_div_Udd_ground == 1 
            time_num = time_gm;
        else
            time_num = (time_gm(1):record_dt/num_div_Udd_ground:time_gm(end))';
        end

        %updated time history considering the user defined numerical dt

        U_dd_ground_num = zeros(size(time_num));      %calculations to obtain a numerical ground acceleration
        U_dd_ground_num(1) = U_dd_ground(1);

        index = 2;

        for i = 1:length(U_dd_ground)-2
            lin_fit = linspace(U_dd_ground(i),U_dd_ground(i+1),num_div_Udd_ground+1);   %acceleration included in between input accelerations
            U_dd_ground_num(index:index+num_div_Udd_ground-1) = lin_fit(2:end)';        %updated acceleration history considering the user defined numerical dt
            index = index + num_div_Udd_ground;
        end

        N_time_steps = length(U_dd_ground_num);

        % plot imposed ground acceleration vs numerical
        %figure
        %plot(time_gm,U_dd_ground,'b-x','markersize',6); hold on
        %plot(time_num,U_dd_ground_num,'r:o','markersize',2)


        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % 2) eigenvalue analysis
        [ ~ ,wn_2] = eig(kini,m);
        wn   = sqrt(diag(wn_2));        % compute omega from eigenvalues [rad/s]
        freq = wn/(2*pi);               % compute eigenfrequencies [Hz]
        Tn   = 2*pi/wn;                 % fundamental periods [s]

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % 2) computation of required initial and constant quantities
        % damping matrix
        c = 2*psi*wn*m;
        % dynamic coefficients
        a = m/(beta*dt) + gamma*c/(beta);
        b = m/(2*beta) + dt*c*(gamma/(2*beta)-1);

        %% CALCULATIONS FOR EACH TIME STEP

        delta_U_dd_ground_num = diff(U_dd_ground_num);

        %initalization quantities for NR
        k(1)      = kini;
        F_stat(1) = 0;
        F_dyn(1)  = 0; 
        u_current = 0.;
        i = 0;
        TMP_failure_label = 0;

        while all([i < N_time_steps-1, abs(u_current) < abs(d_u)])
            %keep on integrating up to the moment in which either the record is
            %finished, or the SDOF has failed

            i = i+1;

            delta_pbar(i) = -m*(delta_U_dd_ground_num(i)) + a*U_d(i) + b*U_dd(i);
            k_bar_T(i)    = k(i) + gamma*c/(beta*dt) + m/(beta*dt^2);

            % step 2.4 in chopra's book, p.189: 5.7 ANALYSIS OF NONLINEAR RESPONSE: NEWMARK'S METHOD
            % initialization for NR cycles
            dR = delta_pbar(i);
            u_current  = U_dyn(i);
            fs_current = F_stat(i);
            k_current  = k(i);

            gg = 0;

            while abs(dR) > 1E-07

                gg = gg + 1;

                if gg > 100
                    disp('it seems there is no convergence')
                end

                delta_U_current = dR/k_bar_T(i);
                u_past          = u_current;
                fs_past         = fs_current;
                k_past          = k_current;
                u_current       = u_current + delta_U_current;

                [fs_current,k_current] = Trilinear_Mat_model(u_current,U_dyn(i),F_stat(i),model_param);

                df = fs_current - fs_past + a/dt*delta_U_current;
                dR = dR - df;

            end

            %update solution to the current time step
            U_dyn(i+1)   = u_current;
            F_stat(i+1)  = fs_current;
            k(i+1)       = k_current;

            %compute deltas at the current time step
            delta_U(i)   = U_dyn(i+1) - U_dyn(i);
            delta_U_d(i) = gamma/(beta*dt)*delta_U(i) - gamma*U_d(i)/beta + dt*U_dd(i)*(1-gamma/(2*beta));
            delta_U_dd(i)= 1/(beta*dt^2)*delta_U(i) - U_d(i)/(beta*dt) - U_dd(i)/(2*beta);

            %update velocities and accelerations with values from current time step
            U_d(i+1)     = U_d(i)  + delta_U_d(i);
            U_dd(i+1)    = U_dd(i) + delta_U_dd(i);
            
            %update total inertia force vector
            F_dyn(i+1)   = F_stat(i+1) + c*U_d(i+1); 

        end

        % check if the wall failed or not
        if abs(u_current) > abs(d_u)
            TMP_failure_label = 1;
        else
            TMP_failure_label = 0;
        end

        % record last step before exiting the loop
        i_end = i + 1;

end