#include <valarray>

#include "xtensor/xarray.hpp"
#include "xtensor/xio.hpp"
#include "xtensor/xview.hpp"
#include "xtensor/xbuilder.hpp"
#include "xtensor/xcomplex.hpp"
#include "xtensor.hpp"

#include <Vars.h>
#include <makeDicts.h>

using namespace std;

double factorial (int n)
{
	double x = 1;
	for(double i = 1; i <=n; i++) {
		x *= i;
	}
	return x;
}

valarray<double> phi_minMod (valarray<double> r)
{
	int Nx = r.size();
	valarray<double> phi (Nx);
	for (int i=0; i < Nx; i++) {
		phi[i] = fmax(0, fmin(1, r[i]) );
	}
	return phi;	
}

double phi_minMod_double (double r)
{
	double phi = fmax(0, fmin(1, r) );
	return phi;	
}


valarray<double> xGrid_maker (Vars variables, bool isStaggeredGrid)
{
	valarray<double> x_grid;
	if (isStaggeredGrid) { // if staggered grid is used, place x_grid on cell bounds instead of cell centers
		x_grid = make_xGrid_staggered (variables.Nx, variables.x_min, variables.x_max); // the CD grid = x_min+dx:dx:x_max (hence not including x_min, thus h[end] = x_min)
	} else {
		x_grid = make_xGrid (variables.Nx, variables.x_min, variables.x_max);	// this grid is on the cell center points
	}
	return x_grid;
}

valarray<double> interp1_lin(valarray<double> xq, valarray<double> yq, valarray<double> x)
{
	int Nx = x.size();
	int Nxq = xq.size();
	if (Nx == Nxq) { return yq; }
	

	
	
	double dxq = xq[1] - xq[0];
	
	// make full vectors, including both ends (x=0, x=L)
	valarray<double> xq_full (Nxq + 2);
	valarray<double> yq_full (Nxq + 2);
	xq_full[0] = 0;
	double yq_ends = 0.5 * (yq[0] + yq[Nxq-1]);
	yq_full[0] = yq_ends;
	for (int i = 1; i < Nxq + 1 ; i++) {
		xq_full[i] = xq[i-1];
		yq_full[i] = yq[i-1];
	}

	xq_full[Nxq+1] = xq_full[Nxq] + dxq/2;
	yq_full[Nxq+1] = yq_ends;

	valarray<double> y(Nx);
		
	valarray<double> yq_minOne = yq_full.cshift(-1);
	valarray<double> yq_plusOne = yq_full.cshift(1);
	valarray<double> xq_minOne = xq_full.cshift(-1);
	double a0, a1;
	int i = 0;
	int Nxq_full = xq_full.size();

	for (int j=1; j < Nxq_full; j++) { 
		while (x[i] < xq_full[j]) {

			a0 = yq_minOne[j];

			a1 = (yq_full[j] - yq_minOne[j]) / (xq_full[j] - xq_minOne[j]);

			y[i] = a0 + a1 * (x[i] - xq_minOne[j]);
			
			if (i == Nx-1) {
				return y;
			} else {			
				i++;
			}
		}
		
	}
	return y; 
}


valarray<double> derivative_upWind_secondOrder(valarray<double> y, double dx) 
{
	valarray<double> dy;
	dy = (3*y  - 4*y.cshift(-1) + y.cshift(-2)) / (2*dx); 
	return dy;		
}

valarray<double> derivative_upWind_thirdOrder(valarray<double> y, double dx) 
{
	valarray<double> dy;
	dy = (11/18 * y - y.cshift(-1) + 0.5 * y.cshift(-2) - 1/9* y.cshift(-3)) / (dx/3); 
	return dy;		
}

valarray<double> derivative_upWind(valarray<double> y, double dx) 
{
	valarray<double> dy;
	dy = (y - y.cshift(-1)) / dx; 
	return dy;		
}

valarray<double> derivative_downWind(valarray<double> y, double dx) 
{
	valarray<double> dy;
	dy = (y.cshift(1) - y) / dx; 
	return dy;		
}

valarray<double> derivative_center(valarray<double> y, double dx) 
{
	valarray<double> dy;
	dy = (y.cshift(1) - y.cshift(-1)) / (2*dx); 
	return dy;		
}

xt::xarray<double> valArrayTo_xarray (valarray<double> x)
{	
	int Nx = x.size();
	xt::xarray<double> y = xt::ones<double>({1, Nx});
	for (int i=0; i < x.size(); i++) {
		y(0,i) = x[i];
	}
	return y;
}

valarray<double> xarrayTo_valArray (xt::xarray<double> x)
{	
	int Nx = x.shape(0);
	valarray<double> y (Nx);
	for (int i=0; i < Nx; i++) {
		y[i] = x(i,0);
	}
	return y;
}

xt::xarray<double> c_leftshift_xarray (xt::xarray<double> x, int shift)
{
	xt::xarray<double> y = xt::zeros<double>({x.shape(0), x.shape(1)});
	for (int i = 0; i < x.shape(1)-shift; i++) {
		xt::col(y,i) = xt::col(x,i+shift);
	}	
	for (int i = x.shape(1)-shift; i < x.shape(1); i++) {
		xt::col(y,i) = xt::col(x,i - x.shape(1) + shift);
	}
	return y;
}

xt::xarray<double> c_rightshift_xarray (xt::xarray<double> x, int shift)
{
	xt::xarray<double> y = xt::zeros<double>({x.shape(0), x.shape(1)});
	for (int i = shift; i < x.shape(1); i++) {
		xt::col(y,i) = xt::col(x,i-shift);
	}	
	for (int i = 0; i < shift; i++) {
		xt::col(y,i) = xt::col(x,i + x.shape(1) - shift);
	}
	return y;
}

valarray<double> make_random_topography (int LastConstituent, valarray<double> x_grid, Vars variables, int j, int increment, int startWaveMode)
{
	valarray<double> h (variables.Nx);
	valarray<double> phases (LastConstituent);
	std::uniform_real_distribution<double> phase_distribution(-M_PI, M_PI);
	default_random_engine eng(j); // this one always generates the same sequence of numbers
	for (int i=startWaveMode; i <= LastConstituent; i+=increment)
	{
		phases[i-1] = phase_distribution(eng);
//		cout << phases[i-1] << " ";
		h = h + variables.bedAmp / (LastConstituent/increment) * sin (x_grid*2*i*M_PI / variables.x_max - phases[i-1]);
	}
 	return h;
}

