/****************************************************************************
**
** Copyright (C) 2011 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** This file is part of the Qt3D module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** GNU Lesser General Public License Usage
** This file may be used under the terms of the GNU Lesser General Public
** License version 2.1 as published by the Free Software Foundation and
** appearing in the file LICENSE.LGPL included in the packaging of this
** file. Please review the following information to ensure the GNU Lesser
** General Public License version 2.1 requirements will be met:
** http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU General
** Public License version 3.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of this
** file. Please review the following information to ensure the GNU General
** Public License version 3.0 requirements will be met:
** http://www.gnu.org/copyleft/gpl.html.
**
** Other Usage
** Alternatively, this file may be used in accordance with the terms and
** conditions contained in a signed written agreement between you and Nokia.
**
**
**
**
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qgllightmodel.h"

QT_BEGIN_NAMESPACE

/*!
    \class QGLLightModel
    \brief The QGLLightModel class defines the lighting model to use for the scene.
    \since 4.8
    \ingroup qt3d
    \ingroup qt3d::painting
*/

/*!
    \qmlclass LightModel QGLLightModel
    \brief The LightModel item defines the lighting model to use for the scene.
    \since 4.8
    \ingroup qt3d::qml3d

    \sa Light
*/

/*!
    \enum QGLLightModel::Model
    This enum defines the type of lighting model to use: one-sided or two-sided.

    \value OneSided One-sided lighting, with the front face material used for both front and back faces.
    \value TwoSided Two-sided lighting, with separate front and back face materials.
*/

/*!
    \enum QGLLightModel::ColorControl
    This enum controls the number of colors to be generated by the lighting computation.

    \value SingleColor A single color is generated by the lighting computation.
    \value SeparateSpecularColor A separate specular color computation is
           performed and then summed into the pixel color after texture mapping.
*/

/*!
    \enum QGLLightModel::ViewerPosition
    This enum defines the position of the viewer for the purposes of lighting calculations.

    \value ViewerAtInfinity The viewer is at infinity along the -z axis.
    \value LocalViewer The viewer is at the local origin in eye coordinates.
*/

class QGLLightModelPrivate
{
public:
    QGLLightModelPrivate()
        : model(QGLLightModel::OneSided),
          colorControl(QGLLightModel::SingleColor),
          viewerPosition(QGLLightModel::ViewerAtInfinity)
    {
        ambientSceneColor.setRgbF(0.2, 0.2, 0.2, 1.0);
    }

    QGLLightModel::Model model;
    QGLLightModel::ColorControl colorControl;
    QGLLightModel::ViewerPosition viewerPosition;
    QColor ambientSceneColor;
};

/*!
    Constructs a light model object with default values and attach
    it to \a parent.
*/
QGLLightModel::QGLLightModel(QObject *parent)
    : QObject(parent)
    , d_ptr(new QGLLightModelPrivate)
{
}

/*!
    Destroys this light model.
*/
QGLLightModel::~QGLLightModel()
{
}

/*!
    \property QGLLightModel::model
    \brief the lighting model to use, either OneSided or TwoSided.
    The default is OneSided.

    \sa modelChanged()
*/

/*!
    \qmlproperty enumeration LightModel::model
    The lighting model to use, either OneSided or TwoSided.
    The default is OneSided.
*/

QGLLightModel::Model QGLLightModel::model() const
{
    Q_D(const QGLLightModel);
    return d->model;
}

void QGLLightModel::setModel(QGLLightModel::Model value)
{
    Q_D(QGLLightModel);
    if (d->model != value) {
        d->model = value;
        emit modelChanged();
        emit lightModelChanged();
    }
}

/*!
    \property QGLLightModel::colorControl
    \brief the color control mode, either SingleColor or
    SeparateSpecularColor.  The default value is SingleColor.

    If SingleColor is specified, then a single color is calculated
    by the lighting computation for a vertex.  If SeparateSpecularColor
    is specified, then a separate specular color computation is
    performed and then summed into the pixel color after texture mapping.

    \sa colorControlChanged()
*/

/*!
    \qmlproperty enumeration LightModel::colorControl
    The color control mode, either SingleColor or
    SeparateSpecularColor.  The default value is SingleColor.

    If SingleColor is specified, then a single color is calculated
    by the lighting computation for a vertex.  If SeparateSpecularColor
    is specified, then a separate specular color computation is
    performed and then summed into the pixel color after texture mapping.
*/

QGLLightModel::ColorControl QGLLightModel::colorControl() const
{
    Q_D(const QGLLightModel);
    return d->colorControl;
}

void QGLLightModel::setColorControl(QGLLightModel::ColorControl value)
{
    Q_D(QGLLightModel);
    if (d->colorControl != value) {
        d->colorControl = value;
        emit colorControlChanged();
        emit lightModelChanged();
    }
}

/*!
    \property QGLLightModel::viewerPosition
    \brief the viewer position, either ViewerAtInfinity or LocalViewer.
    The default value is ViewerAtInfinity.

    \sa viewerPositionChanged()
*/

/*!
    \qmlproperty enumeration LightModel::viewerPosition
    The viewer position, either ViewerAtInfinity or LocalViewer.
    The default value is ViewerAtInfinity.
*/

QGLLightModel::ViewerPosition QGLLightModel::viewerPosition() const
{
    Q_D(const QGLLightModel);
    return d->viewerPosition;
}

void QGLLightModel::setViewerPosition(QGLLightModel::ViewerPosition value)
{
    Q_D(QGLLightModel);
    if (d->viewerPosition != value) {
        d->viewerPosition = value;
        emit viewerPositionChanged();
        emit lightModelChanged();
    }
}

/*!
    \property QGLLightModel::ambientSceneColor
    \brief the ambient color of the entire scene.  The default value
    is (0.2, 0.2, 0.2, 1.0).

    \sa ambientSceneColorChanged()
*/

/*!
    \qmlproperty color LightModel::ambientSceneColor
    The ambient color of the entire scene.  The default value
    is (0.2, 0.2, 0.2, 1.0).
*/

QColor QGLLightModel::ambientSceneColor() const
{
    Q_D(const QGLLightModel);
    return d->ambientSceneColor;
}

void QGLLightModel::setAmbientSceneColor(const QColor& value)
{
    Q_D(QGLLightModel);
    if (d->ambientSceneColor != value) {
        d->ambientSceneColor = value;
        emit ambientSceneColorChanged();
        emit lightModelChanged();
    }
}

/*!
    \fn void QGLLightModel::modelChanged()

    This signal is emitted when model() changes.

    \sa model(), lightModelChanged()
*/

/*!
    \fn void QGLLightModel::colorControlChanged()

    This signal is emitted when colorControl() changes.

    \sa colorControl(), lightModelChanged()
*/

/*!
    \fn void QGLLightModel::viewerPositionChanged()

    This signal is emitted when viewerPosition() changes.

    \sa viewerPosition(), lightModelChanged()
*/

/*!
    \fn void QGLLightModel::ambientSceneColorChanged()

    This signal is emitted when ambientSceneColor() changes.

    \sa ambientSceneColor(), lightModelChanged()
*/

/*!
    \fn void QGLLightModel::lightModelChanged()

    This signal is emitted when one of model(), colorControl(),
    viewerPosition(), or ambientSceneColor() changes.
*/

QT_END_NAMESPACE
