"""
Mechanism for integrating code generation into walberla's CMake system.
CMake needs to determine which C++ source files are generated by which Python script.
The list of files should be available fast, without running the generation process itself.
Thus all code generation function are registered at a single class that manages this process.

Usage example:
    from pystencils_walberla.cmake_integration import codegen
    codegen.register(['MyClass.h', 'MyClass.cpp'], functionReturningTwoStringsForHeaderAndCpp)

"""
import argparse
import json
import os

__all__ = ['CodeGeneration', 'ManualCodeGenerationContext']

DEFAULT_CMAKE_VARS = {'WALBERLA_BUILD_WITH_OPENMP': False,
                      'WALBERLA_OPTIMIZE_FOR_LOCALHOST': False,
                      'WALBERLA_DOUBLE_ACCURACY': True,
                      'WALBERLA_BUILD_WITH_MPI': True,
                      'WALBERLA_BUILD_WITH_CUDA': False,
                      'WALBERLA_BUILD_WITH_HIP': False,
                      "CODEGEN_CFG": ""}

PARSE_HELPER = {"on":  True,  "1": True,  "yes": True,  "true":  True,
                "off": False, "0": False, "no":  False, "false": False}


class CodeGeneration:
    def __init__(self):
        parser = argparse.ArgumentParser(description='Code Generation script for waLBerla.')
        parser.add_argument('-f', '--files', nargs='*',
                            help='List all files that will be generated with absolute path',
                            default=[])
        parser.add_argument('-c', '--cmake-args', type=json.loads,
                            help='Provide CMake configuration (will be used in the codegen config)',
                            default=DEFAULT_CMAKE_VARS)
        parser.add_argument('-l', '--list-only',
                            help="Script will not generate files but list files it would generated without this option")
        args = parser.parse_args()

        cmake_args = {key: PARSE_HELPER.get(str(value).lower(), value) for key, value in args.cmake_args.items()}

        self.context = CodeGenerationContext(cmake_args)
        self.expected_files = args.files
        self.list_only = True if args.list_only else False

    def __enter__(self):
        return self.context

    def __exit__(self, *args):
        if self.expected_files and (set(self.context.files_written) != set(self.expected_files)):
            expected = set(os.path.realpath(f) for f in self.expected_files)
            written = set(os.path.realpath(f) for f in self.context.files_written)
            only_in_cmake = expected - written
            only_generated = written - expected
            error_message = "Generated files (OUT_FILES) specified not correctly " \
                            + "in cmake with 'waLBerla_generate_target_from_python'\n"
            if only_in_cmake:
                error_message += "Files only specified in CMake {}\n".format(
                    [os.path.basename(p) for p in only_in_cmake])
            if only_generated:
                error_message += "Unexpected generated files {}\n".format([os.path.basename(p) for p in only_generated])
            if len(only_in_cmake) > 0 or len(only_generated) > 0:
                raise ValueError(error_message)


class CodeGenerationContext:
    def __init__(self, cmake_vars):
        self.files_written = []
        self.openmp = cmake_vars['WALBERLA_BUILD_WITH_OPENMP']
        self.optimize_for_localhost = cmake_vars['WALBERLA_OPTIMIZE_FOR_LOCALHOST']
        self.mpi = cmake_vars['WALBERLA_BUILD_WITH_MPI']
        self.double_accuracy = cmake_vars['WALBERLA_DOUBLE_ACCURACY']
        self.cuda = cmake_vars['WALBERLA_BUILD_WITH_CUDA']
        self.hip = cmake_vars['WALBERLA_BUILD_WITH_HIP']
        self.gpu = self.cuda or self.hip
        self.config = cmake_vars['CODEGEN_CFG'].strip()

    def write_file(self, name, content):
        self.files_written.append(os.path.abspath(name))
        with open(name, 'w') as f:
            f.write(content)


class ManualCodeGenerationContext:
    """Context for testing - does not actually write files but puts them into a public dict
    Environment parameters like if OpenMP, MPI or CPU-specific optimization should be used can be explicitly passed
    to constructor instead of getting them from CMake
    """

    def __init__(self, openmp=False, optimize_for_localhost=False, mpi=True, double_accuracy=True,
                 cuda=False, hip=False):
        self.openmp = openmp
        self.optimize_for_localhost = optimize_for_localhost
        self.mpi = mpi
        self.double_accuracy = double_accuracy
        self.files = dict()
        self.cuda = cuda
        self.hip = hip
        self.gpu = self.cuda or self.hip
        self.config = ""

    def write_file(self, name, content):
        self.files[name] = content

    def write_all_files(self):
        for name, content in self.files.items():
            with open(name, 'w') as f:
                f.write(content)
        self.files = dict()

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        pass
