#!/usr/bin/env python3
import sys, os
import pandas as pd
import matplotlib.pyplot as plt

def time4(filename):
   # Find the fourth line with a Time value
   # Return the average time value (column 8)
   counter = 0
   for line in open(filename):
    if "Time" in line:
      counter +=1
      if counter > 3:
        return float( line.split(' ')[8] )

def get_time_data(avx512dir, avx2dir, cores):
    # Read average timings from avx512 and avx2 directories
    # and return as a dataframe
    avx512 = []
    avx2 = []
    for ncores in cores:
      folder = "{}{}/log/".format(avx512dir,ncores)
      filename = os.listdir(folder)[0]
      avx512.append(time4(folder+filename))
      
      folder = "{}{}/log/".format(avx2dir,ncores)
      filename = os.listdir(folder)[0]
      avx2.append(time4(folder+filename))
    
    time_dict = {'Cores': cores, 'AVX-512': avx512, 'AVX2': avx2}
    return pd.DataFrame.from_dict(time_dict)

def plot_scaling(data, labels, filename):
    # Plot results of a scaling study from a list of dataframe
    # and label each with the corresponding entry in labels
    # Save to filename
    plt.xlabel('Number of cores')
    plt.ylabel('Relative Performance')
    colours = ['black','red','blue']
    symbols = ['-o','-x','-*']
    for i,[d,l] in enumerate(zip(data,labels)):
        index = i%len(colours)
        plt.plot(d['Cores'], d['AVX2']/d['AVX-512'], symbols[index], label=l, color=colours[index])
    plt.xscale('log', basex=2)
    plt.gca().set_aspect('equal', adjustable='box')
    plt.legend(loc=0, frameon=False)
    plt.savefig(filename)
    plt.clf() 


def performance1(filename):
   # Get the first performance value from the given file
   # This should be the Dw measure (second is Dw_hat)
   for line in open(filename):
    if "Mflops" in line:
      return float( line.split(' ')[3][1:] )
  

def get_performance_data(avx512dir, avx2dir, index, volumes):
    # Get the performance result for each volume with index 
    # from both avx512 and avx2 directories. Return as a
    # dataframe
    avx512 = []
    avx2 = []
    for volume in volumes:
      folder = "{}{}/".format(avx512dir,volume)
      filename = "time{}.log".format(index)
      avx512.append(performance1(folder+filename))
      
      folder = "{}{}/".format(avx2dir,volume)
      filename = "time{}.log".format(index)
      avx2.append(performance1(folder+filename))
    
    perf_dict = {'Volume': volumes, 'AVX-512': avx512, 'AVX2': avx2}
    return pd.DataFrame.from_dict(perf_dict)


def plot_dw(data, filename):
    # Plot the dirac operator performance values from a dataframe
    # Save to filename
    plt.xlabel('Volume')
    plt.ylabel('Performance (Gflops)')

    plt.plot(data['Volume'], data['AVX-512']/1000, '-o', label='AVX-512', color='black')
    plt.plot(data['Volume'], data['AVX2']/1000, '-x',    label='AVX2', color='red')

    plt.xscale('log', basex=2)
    plt.gca().set_aspect('equal', adjustable='box')
    plt.legend(loc=0, frameon=False)
    plt.savefig(filename)
    plt.clf()

# Read the strong scaling data, larger lattice
strong = get_time_data('bench_knl_avx512_','bench_knl_avx_',[1,2,4,8,16,32])
strong['Cores'] *= 64
strong.to_csv("strong", sep=' ', columns=['Cores','AVX-512','AVX2'],index=False)

# Plot
plot_scaling( data=[strong], labels=['$32^4'], filename='strong_knl.eps' )

# Read the weak scaling data and plot
weak = get_time_data('bench_knl_weak_avx512_','bench_knl_weak_avx_',[32,64,128,256])
weak['Cores'] *= 64
weak.to_csv("weak", sep=' ', columns=['Cores','AVX-512','AVX2'],index=False)
plot_scaling( data=[weak], labels=[''], filename='weak_knl.eps' )

# Read the single precision Dirac operator performance data and plot
dw = get_performance_data("bench_knl_avx512_dirac/dirac","bench_knl_avx_dirac/dirac",1,[256,512,1024,2048,4096,8192])
dw.to_csv("Dw", sep=' ', columns=['Volume','AVX-512','AVX2'],index=False)
plot_dw( data=dw, filename='Dw_knl.eps' )

# Same for double precision
dw_dble = get_performance_data("bench_knl_avx512_dirac/dirac","bench_knl_avx_dirac/dirac",2,[256,512,1024,2048,4096,8192])
dw_dble.to_csv("Dw_dble", sep=' ', columns=['Volume','AVX-512','AVX2'],index=False)
plot_dw( data=dw_dble, filename='Dw_knl_dble.eps' )


