import os 
import numpy as np
import pandas as pd
import geopandas as gpd
import rasterio
from esda.getisord import G_Local
from libpysal.weights import KNN
from shapely.geometry import Point
import matplotlib.pyplot as plt

# Definir las clases de cobertura del suelo
land_cover_classes = {
    0: 'Ocean',
    1: 'Terra Firma',
    2: 'Not used',
    3: 'Wetland',
    4: 'Open surface water',
    5: 'Snow/ice',
    6: 'Cropland',
    7: 'Built-up',
    8: 'Coastal',
    9: 'No data',
    10: 'Desert'
}

esv_values_per_area = {
    'Terra Firma': [55, 69, 3.5, 70.5, 1.5, 87, 5.5, 14, 34],
    'Not used': [0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00],
    'Wetland': [304, 106, 133, 0, 3800, 4177, 0, 574, 881],
    'Open surface water': [41, 0, 0, 0, 2117, 665, 0, 230, 0],
    'Snow/ice': [0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00],
    'Cropland': [54, 0, 0, 0, 0, 14, 0, 24, 0],
    'Built-up': [0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00],
    'Ocean': [15, 0, 38, 0, 0, 0, 0, 1.25, 0],
    'Desert': [0, 0, 0, 0, 0, 0, 0, 0, 0],
    'Coastal': [93, 4, 0, 0, 0, 0, 8, 33.5, 82]
}
# Servicios ecosistémicos
servicios_ecosistemicos = ['Food Production', 'Raw Material Supply', 'Gas Regulation', 'Climate Regulation', 'Water Supply', 'Waste Treatment', 'Soil Formation', 'Biodiversity Protection', 'Recreation']

# Ajustar los valores a 3 hectáreas
esv_values_per_area = {key: [value * 1 for value in values] for key, values in esv_values_per_area.items()}


 # Función para leer un raster y obtener la cobertura del suelo
def leer_raster(ruta):
    with rasterio.open(ruta) as src:
        cobertura = src.read(1)  # Leer la primera banda
        profile = src.profile
    return cobertura, profile

# Leer los rasters para diferentes años

rutas_rasters = {
    2000: "D:/Islas/simulacion/Atlanticas/reprojected/reclassified_land_cover_2000.tif",
    2005: "D:/Islas/simulacion/Atlanticas/reprojected/reclassified_land_cover_2005.tif",
    2010: "D:/Islas/simulacion/Atlanticas/reprojected/reclassified_land_cover_2010.tif",
    2015: "D:/Islas/simulacion/Atlanticas/reprojected/reclassified_land_cover_2015.tif",
    2020: "D:/Islas/simulacion/Atlanticas/reprojected/reclassified_land_cover_2015.tif",
    2040: "D:/Islas/simulacion/Atlanticas/cobertura_terrestre_simulada_Azores_2040_CA.tif"
}

coberturas = {}
perfiles = {}

for year, ruta in rutas_rasters.items():
    cobertura, profile = leer_raster(ruta)
    coberturas[year] = cobertura
    perfiles[year] = profile

# Obtener las clases discretas
clases_discretas = list(land_cover_classes.keys())

# Función para calcular el ESV raster utilizando valores específicos por unidad de área
def calcular_esv_raster(cobertura, esv_values_per_area, clases_discretas, servicios_ecosistemicos):
    esv_rasters = {servicio: np.zeros_like(cobertura, dtype=float) for servicio in servicios_ecosistemicos}
    for clase in clases_discretas:
        clase_nombre = land_cover_classes[clase]
        if clase_nombre in esv_values_per_area:
            mask = (cobertura == clase)
            for i, servicio in enumerate(servicios_ecosistemicos):
                esv_rasters[servicio][mask] = esv_values_per_area[clase_nombre][i]
    return esv_rasters

# Calcular el ESV raster para cada año
esv_rasters_anuales = {}
esv_totales_anuales = {}
esv_por_clase_anuales = {}

for year, cobertura in coberturas.items():
    esv_rasters = calcular_esv_raster(cobertura, esv_values_per_area, clases_discretas, servicios_ecosistemicos)
    esv_rasters_anuales[year] = esv_rasters
    
    esv_totales = {servicio: np.nansum(esv_rasters[servicio]) for servicio in servicios_ecosistemicos}
    esv_totales_anuales[year] = esv_totales
    
    esv_por_clase = {land_cover_classes[clase]: {servicio: np.nansum(esv_rasters[servicio][cobertura == clase]) for servicio in servicios_ecosistemicos} for clase in clases_discretas}
    esv_por_clase_anuales[year] = esv_por_clase

# Guardar los resultados en archivos raster
output_dir = "D:/Islas/simulacion/reprojected/ESV_Azores_BS"
if not os.path.exists(output_dir):
    os.makedirs(output_dir)

for year, esv_rasters in esv_rasters_anuales.items():
    for servicio, esv_raster in esv_rasters.items():
        output_path = f"{output_dir}/ESV_BS_{servicio}_{year}.tif"
        with rasterio.open(
            output_path,
            'w',
            driver='GTiff',
            height=esv_raster.shape[0],
            width=esv_raster.shape[1],
            count=1,
            dtype=esv_raster.dtype,
            crs=perfiles[year]['crs'],
            transform=perfiles[year]['transform']
        ) as dst:
            dst.write(esv_raster, 1)

# Calcular la tasa de cambio
esv_tasa_cambio = {}
for servicio in servicios_ecosistemicos:
    esv_tasa_cambio[servicio] = {}
    for year in sorted(esv_totales_anuales.keys()):
        if year > 2000:
            if esv_totales_anuales[2000][servicio] != 0:  # Evitar división por cero
                tasa_cambio = (esv_totales_anuales[year][servicio] - esv_totales_anuales[2000][servicio]) / esv_totales_anuales[2000][servicio] * 100
            else:
                tasa_cambio = np.nan
            esv_tasa_cambio[servicio][year] = tasa_cambio

# Crear un DataFrame para la tasa de cambio
df_tasa_cambio = pd.DataFrame.from_dict(esv_tasa_cambio, orient='index')
df_tasa_cambio = df_tasa_cambio.transpose()

# Crear DataFrame para los valores de ESV por clase y por año
esv_por_clase_df = pd.DataFrame.from_dict({(year, clase): esv_por_clase_anuales[year][clase] for year in esv_por_clase_anuales for clase in esv_por_clase_anuales[year]}, orient='index')

# Mostrar DataFrames
print("Tasa de cambio de los servicios ecosistémicos:")
print(df_tasa_cambio)
print("\nValores de ESV por clase y por año:")
print(esv_por_clase_df)

# Guardar los DataFrames en Excel
df_tasa_cambio.to_excel(f"{output_dir}/Tasa_Cambio_Servicios_Ecosistemicos_BS.xlsx")
esv_por_clase_df.to_excel(f"{output_dir}/ESV_Por_Clase.xlsx")

# Visualización de la tasa de cambio
fig, ax = plt.subplots(figsize=(10, 6))
df_tasa_cambio.plot(kind='bar', ax=ax)
plt.title('Rate of Change of Ecosystem Services in BS')
plt.xlabel('Year')
plt.ylabel('Rate of change (%)')
plt.legend(title='Ecosystem Services')
plt.tight_layout()
plt.show()

# Análisis de puntos calientes y fríos combinado
def calcular_puntos_calientes_frios_combinado(esv_rasters_anuales, perfiles, output_dir):
    for year, esv_rasters in esv_rasters_anuales.items():
        combined_esv_raster = np.nanmean(np.array(list(esv_rasters.values())), axis=0)
        
        # Crear GeoDataFrame con los valores combinados de ESV
        rows, cols = np.indices(combined_esv_raster.shape)
        valid_mask = ~np.isnan(combined_esv_raster)
        points = np.array([cols[valid_mask], rows[valid_mask]]).T
        values = combined_esv_raster[valid_mask]

        # Asegurarse de que el perfil para el año existe
        if year not in perfiles:
            print(f"No se encontró el perfil para el año {year}. Omitting year.")
            continue
        
        perfil = perfiles[year]
        
        gdf = gpd.GeoDataFrame({'value': values}, geometry=[Point(x, y) for x, y in points], crs=perfil['crs'])
        gdf = gdf.to_crs(epsg=3857)

        # Calcular los valores de Getis-Ord Gi*
        weights = KNN.from_dataframe(gdf, k=8)
        gdf['Gi*'] = G_Local(gdf['value'], weights).Zs

        # Clasificar los puntos calientes/fríos con intervalos de confianza
        def classify_hotspot(z):
            if z > 2.576:
                return 3  # Hot Spot - 99% Confidence
            elif z > 1.96:
                return 2  # Hot Spot - 95% Confidence
            elif z > 1.645:
                return 1  # Hot Spot - 90% Confidence
            elif z < -2.576:
                return -3  # Cold Spot - 99% Confidence
            elif z < -1.96:
                return -2  # Cold Spot - 95% Confidence
            elif z < -1.645:
                return -1  # Cold Spot - 90% Confidence
            else:
                return 0  # Not Significant

        gdf['hotspot'] = gdf['Gi*'].apply(classify_hotspot)

        # Crear un raster de puntos calientes/fríos
        hotspot_raster = np.full(combined_esv_raster.shape, np.nan)
        for (x, y), value in zip(points, gdf['hotspot']):
            hotspot_raster[y, x] = value

        # Guardar el raster
        output_path = f"{output_dir}/Hotspots_CombinedBS_{year}.tif"
        with rasterio.open(
            output_path,
            'w',
            driver='GTiff',
            height=hotspot_raster.shape[0],
            width=hotspot_raster.shape[1],
            count=1,
            dtype=hotspot_raster.dtype,
            crs=perfil['crs'],
            transform=perfil['transform']
        ) as dst:
            dst.write(hotspot_raster.astype(rasterio.float32), 1)

        # Mensaje de depuración para confirmar la escritura del archivo
        print(f"Archivo escrito: {output_path}")

# Calcular puntos calientes/fríos combinados para cada año
calcular_puntos_calientes_frios_combinado(esv_rasters_anuales, perfiles, output_dir)


### Visualización de los Rasters de Puntos Calientes y Fríos


def plot_hotspot_rasters(output_dir):
    years = [2000, 2005, 2010, 2015, 2020, 2040]
    for year in years:
        fig, ax = plt.subplots(figsize=(10, 10))
        hotspot_raster_path = f"{output_dir}/Hotspots_CombinedBS_{year}.tif"
        
        # Comprobar si el archivo existe antes de abrirlo
        if not os.path.exists(hotspot_raster_path):
            print(f"El archivo {hotspot_raster_path} no existe.")
            continue
        
        with rasterio.open(hotspot_raster_path) as src:
            hotspot_raster = src.read(1)
        cmap = plt.get_cmap('coolwarm', 7)
        img = ax.imshow(hotspot_raster, cmap=cmap, vmin=-3, vmax=3)
        cbar = plt.colorbar(img, ax=ax, ticks=[-3, -2, -1, 0, 1, 2, 3], orientation='vertical')
        cbar.ax.set_yticklabels(['Cold Spot - 99%', 'Cold Spot - 95%', 'Cold Spot - 90%', 'Not Significant', 'Hot Spot - 90%', 'Hot Spot - 95%', 'Hot Spot - 99%'])
        plt.title(f'Hotspot Analysis Combined BS in {year}')
        plt.savefig(f"{output_dir}/Hotspot_Rasters_CombinedBS_{year}.png")
        plt.show()

# Visualizar los rasters de puntos calientes y fríos
plot_hotspot_rasters(output_dir)



# Calcular puntos calientes/fríos combinados para cada año
calcular_puntos_calientes_frios_combinado(esv_rasters_anuales, perfiles[2000], output_dir)

# Crear los gráficos de los ESV y puntos calientes/fríos
def plot_esv_rasters(esv_rasters_anuales, servicios_ecosistemicos, output_dir):
    for year in esv_rasters_anuales.keys():
        fig, axs = plt.subplots(1, len(servicios_ecosistemicos), figsize=(20, 5))
        for i, servicio in enumerate(servicios_ecosistemicos):
            img = axs[i].imshow(esv_rasters_anuales[year][servicio], cmap='viridis')
            axs[i].set_title(f'{servicio} ({year})')
            plt.colorbar(img, ax=axs[i], orientation='horizontal')
        plt.suptitle(f'ESV Rasters BS for {year}')
        plt.tight_layout()
        plt.savefig(f"{output_dir}/ESV_Rasters_{year}.png")
        plt.show()


plot_esv_rasters(esv_rasters_anuales, servicios_ecosistemicos, output_dir)