# Copyright (C) 2008-2009 EDF R&D
# Author: Damien Garaud
#
# This file is part of the air quality modeling system Polyphemus. It is used
# to generate ensembles.
#
# Polyphemus is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or (at your option)
# any later version.
#
# Polyphemus is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
# more details.
#
# For more information, visit the Polyphemus web site:
#      http://cerea.enpc.fr/polyphemus/

"""\package ensemble_generation.ensemble_program

This module contains the class 'EnsembleProgram'.

It is used to launch automatically several Polyphemus programs.
"""

from atmopy import talos
from run.ensemble_generation import EnsembleParameter
from run.ensemble_generation.function import create_output_dir,\
                                             get_bc_init_date
from run.network import Network
from run.polyphemus import Configuration, Polyphemus, Program

import commands, os


class EnsembleProgram:
    """This class manages an ensemble of programs to launch several
    simulations.

    First, a program can depend on several choices of parameters. A
    configuration file is needed to describe these dependencies. Actually, a
    same program does not write the results in the same place for two
    different values of parameters. Then, a few generic configuration files
    exist for each program. A few variables between \%variable\% will be
    replaced by the right value of a parameter. Then subdirectories will be
    created to store the different results. Finally, a list of programs
    (preprocessing and simulations) will be able to be launched.
    """

    ###############
    # CONSTRUCTOR #
    ###############


    def __init__(self, config_parameter = None, config_program = None,
                 polyphemus_dir = None, ConfigReplacement = None,
                 only_preprocessing = True):
        """The construcor.

        Initializes the attributes and reads two configuration files.
        \param config_parameter the path to the configuration file which
        describes the different parameters.
        \param config_program the path to the configuration file which
        enumerates the programs and their dependencies.
        \param polyphemus_dir path to the polyphemus directory.
        \param ConfigReplacement an instance 'ConfigReplacement'.
        \param only_preprocessing just the preprocessing?
        """

        ## Only launch the preprocessing programs?
        self.only_preprocessing = only_preprocessing

        ## \brief A dictionary.
        ## \details The list of the programs for each section.
        self.program_name = {}

        ## \brief A dictionary.
        ## \details The names of the parameters on which depends each program.
        self.dependency = {}

        ## \brief A dictionary.
        ## \details The name of the binary files generated by each program.
        self.binary_file = {}

        ## \brief A dictionary.
        ## \details The index of group for each program.
        self.group = {}

        ## \brief A dictionary.
        ## \details The directories of the generic configuration files for
        ## each section.
        self.generic_config_dir = {}

        ## \brief A dictionary.
        ## \details Several variables which will be replaced in the generic
        ## configuration files.
        self.general_variable = {}

        ## \brief A dictionary.
        ## \details The default values for each parameter.
        self.default_dict = {}

        ## \brief A dictionary.
        ## \details The identity of each program in function to the
        ## dependencies and the model identity.
        self.id_program = {}

        ## List of all programs which are ready to be launched.
        self.program_instance_list = []

        # For directories.
        if polyphemus_dir != None:
            self.SetPolyphemusDirectory(polyphemus_dir)
        else:
            ## The Polyphemus directory.
            self.polyphemus_dir = ''

        ## The directory where the preprocessing data will be stored.
        self.data_dir = ""

        ## The directory where the simulations results will be stored.
        self.output_dir = ""

        if ConfigReplacement != None:
            self.SetConfigReplacement(ConfigReplacement)
        else:
            self.config_replacement = None

        self.Init(config_parameter, config_program)


    def Init(self, config_parameter = None, config_program = None):
        """Checks a few directories and loads configuration files.
        \param config_parameter the path to the configuration file which
        describes the different parameters.
        \param config_program the path to the configuration file which
        enumerates the programs and their dependencies.
        """

        # EnsembleParameter
        if config_parameter != None:
            ## An instance 'EnsembleParameter'.
            self.parameter = EnsembleParameter(config_parameter)

            ## The number of models.
            self.Nmodel = self.parameter.Nmodel
        else:
            self.parameter = EnsembleParameter()

        # Program configuration.
        if config_program != None:
            self.LoadConfiguration(config_program)

        # Identity program.
        self.InitIdProgram()


    def ReadIdEnsembleFile(self, filename = "id_ens.dat"):
        """Reads the identity of each model from a file.
        \param filename the name of the file.
        """

        self.parameter.ReadIdEnsembleFile(filename, copy = True)
        self.Nmodel = self.parameter.Nmodel


    def LoadConfiguration(self, config_program):
        """Checks the path of the configuraton file and loads it.
        \param config_program the path to the configuration file which
        enumerates the programs and their dependencies.
        """

        ## The program configuraton file.
        self.config_program = os.path.abspath(config_program)
        if not os.path.isfile(config_program):
            raise Exception, "Unable to find the configuration file : \"" \
                  + config_program + "\"."
        self.ReadConfigProgram()


    def ReadConfigProgram(self):
        """Reads the program configuraton file.
        It reads each section [section] in the file and fills the different
        dictionaries as 'generic_config_dir', 'dependency', 'group' or
        'program_name'.
        """

        ## List of all programs which appear in the configuration file which
        ## describes the programs.
        self.all_program = []

        content = [("generic_cfg", "[general]", "String"), \
                   ("[ground]", "", "ground", "StringSection"), \
                   ("[meteo]", "", "meteo", "StringSection"), \
                   ("[dep]", "", "dep", "StringSection"), \
                   ("[emissions]", "", "emissions", "StringSection"), \
                   ("[bio]", "", "bio", "StringSection"), \
                   ("[ic]", "", "ic", "StringSection"), \
                   ("[bc]", "", "bc", "StringSection"), \
                   ("[driver]", "", "driver", "StringSection")]

        ## A atmopy.talos.Config instance.
        self.config = talos.Config(self.config_program, new_content = content)

        ## The directory where generic configurations files are stored.
        self.generic_cfg = self.config.generic_cfg

        # Extracts the name of the programs and their dependencies from the
        # configuration file.
        self.ExtractSection('ground')
        self.ExtractSection('meteo')
        self.ExtractSection('dep')
        self.ExtractSection('emissions')
        self.ExtractSection('bio')
        self.ExtractSection('ic')
        self.ExtractSection('bc')
        self.ExtractSection('driver')


    def InitIdProgram(self):
        """Initializes the dictionary where there are the choices of
        parameters for each program.
        """

        self.id_program.clear()
        for section in self.program_name.values():
            for name in section:
                tmp = {name: []}
                self.id_program.update(tmp)


    def InitEnsemble(self):
        """Initializes an ensemble of program.
        """

        self.Check()
        self.InitIdProgram()
        self.program_instance_list.__init__()
        self.CopyConfigurationFile()


    ##################
    # ACCESS METHODS #
    ##################


    def GetProgramSection(self, program):
        """Returns the section to which the program belongs.
        \param program the name of a program.
        """

        if len(self.all_program) == 0:
            raise Exception, "The program list is empty."
        if program not in self.all_program:
            raise Exception, "The program: \"" + program + "\" is not in " \
                  + "the program list."
        key = []
        value = []
        it = self.program_name.iterkeys()
        for i in range(len(self.program_name)):
            key.append(it.next())
            value.append(self.program_name[key[i]])
            if program in value[i]:
                return key[i]


    def GetParameterDict(self, model_index):
        """Returns the description of a model.
        If a name of a parameter does not appear in the description of the
        model, the value of the parameter by default is added to the
        dictionary.
        \param model_index the index of a model.
        @return A dictionary.
        """

        # Checks if there is a 'ConfigReplacement' instance.
        if self.config_replacement == None:
            raise Exception, "The attribute 'config_replacement' does not "\
                + "have value. Please use the method 'SetConfigReplacement'."

        if len(self.default_dict) == 0:
            self.SetDefaultDict()
        param_dict = self.parameter.GetDict(model_index)
        # Checks and replaces parameters with default parameters.
        for k in self.default_dict.keys():
            if k not in self.parameter.name:
                param_dict[k] = self.default_dict[k]

        return param_dict


    def GetGeneralDict(self, model_index):
        """Returns the 'general_dict'.
        Takes the description of a model and adds it to the
        'general_dict'. This dictionary will be used to replace the right
        variables in the generic configurations files.
        \param model_index the index of a model.
        @return A dictionary.
        """

        result = {}

        # Directories
        preproc_dir = self.polyphemus_dir + "/preprocessing"
        self.data_dir = self.parameter.config.data_dir
        model_dir = self.parameter.config.model_dir
        self.output_dir = model_dir + "/" \
                          + self.parameter.GetStringModel(model_index)

        # General datas.
        date_min = self.parameter.config.date_min
        str_date_min = date_min.strftime("%Y%m%d")
        x_min = self.parameter.config.x_min
        y_min = self.parameter.config.y_min
        Delta_x = self.parameter.config.Delta_x
        Delta_y = self.parameter.config.Delta_y
        Nx = self.parameter.config.Nx
        Ny = self.parameter.config.Ny

        # General dictionary.
        base_dict = {"%polyphemus_preprocessing_dir%": preproc_dir,
                     "%data_dir%": self.data_dir,
                     "%output_config_dir%": self.output_dir + "/config",
                     "%output_dir%": self.output_dir,
                     "%Date%": str_date_min,
                     "%x_min%": x_min,
                     "%y_min%": y_min,
                     "%Delta_x%": Delta_x,
                     "%Delta_y%": Delta_y,
                     "%Nx%": Nx,
                     "%Ny%": Ny}

        variable_replacement = self.config_replacement\
                               .GetConfigVariable(model_index, self)
        directory_replacement = self.GetDirectoryReplacement(model_index)

        result.update(base_dict)
        result.update(variable_replacement)
        result.update(directory_replacement)

        # Binary files dictionary.
        self.binary_file = self.config_replacement.GetBinaryFile(model_index,
                                                                self)
        # Checks the programs in the binary files dictionary.
        for k in self.binary_file.keys():
            if k not in self.dependency.keys():
                raise Exception, "The program \"" + k + "\" is not " \
                    + "in the programs list. Please check " \
                    + "the method 'GetBinaryFile' from your '" \
                    + str(self.config_replacement.__class__).split('.')[-1] \
                    + "' object or your configuration file \"" \
                    + self.config_program + "\"."

        return result


    def SetConfigReplacement(self, ConfigReplacement):
        """Sets the attribute 'config_replacement'.
        It is used to return a few dictionaries which will be used in the
        replacement of several variables in the generic configuration files.

        \param ConfigReplacement an object with the methods:
        'GetConfigVariable', 'GetBinaryFile', 'GetDefaultDict',
        'GetPerturbedFielDict' and returns several useful dictionaries.
        """

        # Checks the method 'GetConfigVariable'.
        if not hasattr(ConfigReplacement, 'GetConfigVariable'):
            raise Exception, "There is not the method  'GetConfigVariable'" \
                + " in the '" \
                + str(ConfigReplacement.__class__).split('.')[-1] \
                + "' object."
        # Checks the method 'GetBinaryFile'.
        if not hasattr(ConfigReplacement, 'GetBinaryFile'):
            raise Exception, "There is not the method  'GetBinaryFile'" \
                + " in the '" \
                + str(ConfigReplacement.__class__).split('.')[-1] \
                + "' object."
        # Checks the method 'GetDefaultDict'.
        if not hasattr(ConfigReplacement, 'GetDefaultDict'):
            raise Exception, "There is not the method  'GetDefaultDict'" \
                + " in the '" \
                + str(ConfigReplacement.__class__).split('.')[-1] \
                + "' object."
        # Checks the method 'GetPerturbedFieldDict'.
        if not hasattr(ConfigReplacement, 'GetPerturbedFieldDict'):
            raise Exception, "There is not the method  'GetPerturbedFieldDict'" \
                + " in the '" \
                + str(ConfigReplacement.__class__).split('.')[-1] \
                + "' object."

        ## An instance 'ConfigReplacement'.
        self.config_replacement = ConfigReplacement



    def GetDirectoryReplacement(self, model_index):
        """Returns a dictionary about the path of the preprocessing binary files.

        \param model_index the index of a model.
        @return A dictionary.
        """

        result = {}
        for program in self.dependency.keys():
            key = '%' + program + '_dir%'
            result[key] = \
                create_output_dir(program, self.dependency[program],
                                  self.GetParameterDict(model_index))
        return result


    def GetPerturbationDict(self, model_index):
        """Returns the names of perturbed fields and the associated
        perturbation values.
        \param model_index the index of a model.
        @return A dictionary.
        """

        result = self.config_replacement.GetPerturbedFieldDict(model_index,
                                                               self)
        return result


    ##############
    # PROCESSING #
    ##############


    def Check(self):
        """Checks a few attributes, the directories, the parameters and the
        programs.
        """

        if self.parameter.Nmodel == 0:
            raise Exception, "You must load a parameter configuration file."
        if len(self.parameter.id_ensemble) != self.parameter.Nmodel:
            raise Exception, "The model number must be equal to \"" \
                  + str(self.parameter.Nmodel) + "\" instead of \"" \
                  + str(len(self.parameter.id_ensemble)) + "\"." \
                  + " You forget to read or generate an identity ensemble" \
                  + " from your " \
                  + str(self.parameter.__class__).split(".")[-1]
        if len(self.program_name) == 0:
            raise Exception, "There is no program. You must load a program "\
                  + "configuration file."

        self.CheckParameterAndProgram()
        self.CheckConfigDirectory()


    def CheckProgramExist(self, program):
        """Checks if a program exists.

        If the program does not exist in the subdirectories 'preprocessing' or
        'processing' in the Polyphemus directory, an Exception is thrown.
        """
        list_file = os.listdir(self.polyphemus_dir + "/processing/racm/")
        list_file += os.listdir(self.polyphemus_dir + "/processing/radm/")
        list_file += os.listdir(self.polyphemus_dir + "/preprocessing/bc/")
        list_file += os.listdir(self.polyphemus_dir + "/preprocessing/bio/")
        list_file += os.listdir(self.polyphemus_dir + "/preprocessing/dep/")
        list_file += os.listdir(self.polyphemus_dir + \
                                "/preprocessing/emissions/")
        list_file += os.listdir(self.polyphemus_dir
                                + "/preprocessing/ground/")
        list_file += os.listdir(self.polyphemus_dir + "/preprocessing/ic/")
        list_file += os.listdir(self.polyphemus_dir
                                + "/preprocessing/meteo/")
        if program not in list_file:
            raise Exception, "The program \"" + program \
                + "\" does not exist in " + self.polyphemus_dir \
                + "/driver or /preprocessing."


    def CheckParameterAndProgram(self):
        """Checks if the dependencies of all programs appear with the option
        'preprocessing' in the parameter configuration file.
        """

        for program in self.all_program:
            for dependency in self.dependency[program]:
                if dependency != "None" and dependency \
                       not in self.parameter.preprocessing:
                    raise Exception, "The dependency \"" + dependency \
                        + "\" from program \"" + program + "\" in \"" \
                        + self.config_program + "\" does not appear" \
                        + " in file \"" + self.parameter.config_filename \
                        + "\"" + " with \"preprocessing\" argument."


    def CheckConfigDirectory(self):
        """Checks the directories where the generic configuration files are
        stored.
        """

        count = 0
        for cfg in self.generic_config_dir.values():
            if not os.path.isdir(cfg):
                raise Exception, "Unable to find the directory: \"" \
                      + cfg + "\" in [" + self.generic_config_dir.keys()[count]  \
                      + "] in this file: \"" + self.config_program + "\"."
            count += 1


    def CheckDataDirectory(self):
        """Checks if the preprocessing directories exist.
        """

        data_dir = self.parameter.config.data_dir
        data_dir_list = os.listdir(data_dir)
        section_list = self.program_name.keys()
        section_list.remove("driver")
        for section in section_list:
            if section not in data_dir_list:
                raise Exception, "The section name \"" + section \
                      + "\" from the file: \"" + self.config_program \
                      + "\" must appear in your data directory: \"" \
                      + data_dir + "\"."


    def SetPolyphemusDirectory(self, polyphemus_dir):
        """Checks and sets the Polyphemus directory.

        \param polyphemus_dir the Polyphemus directory.
        """

        # Checks the polyphemus directory.
        if not os.path.isdir(polyphemus_dir):
            raise Exception, "Unable to find the Polyphemus directory: \"" \
                + polyphemus_dir + "\"."
        self.polyphemus_dir = polyphemus_dir


    def ExtractSection(self, section):
        """ Extracts each program name and their dependencies held
        in a section from the configuration file.

        \param section the name of the section which can be 'ground', 'meteo',
        'dep', 'emissions', 'bio', 'ic', 'bc' or 'driver'.
        """

        all_section = ['ground', 'meteo', 'dep', 'emissions', \
                           'bio', 'ic', 'bc', 'driver']
        if section not in all_section:
            raise Exception, "The section \"" + section \
                + "\" is not supported."
        # A list of strings.
        program_list = ['']
        if hasattr(self.config, section):
            program_list = getattr(self.config, section)
        # The name of the programs.
        program_name = []
        # If the section is not empty.
        if len(program_list[0]) != 0:
            # A loop for each program.
            for i in range(len(program_list)):
                # The first line must be a directory of the generic
                # configuration files in the section 'section'.
                if i == 0:
                    # Checks if this value corresponds to a directory.
                    tmp = program_list[i].split()[-1]
                    if not os.path.isdir(tmp):
                        raise Exception, "Unable to find the directory \"" \
                              + tmp + "\" in the section \"" + section \
                              + "\" from the configuration file \"" \
                              + self.config_program + "\"."
                    self.generic_config_dir[section] = tmp
                    continue
                tmp = program_list[i].split()
                program_name.append(tmp[0].strip(":=,;#"))
                dependency = []
                # A loop for each dependency of a program.
                for s in tmp[1:]:
                    dependency.append(s.strip(":=,;#"))
                self.dependency[program_name[i - 1]] = dependency[0:-1]
                self.group[program_name[i - 1]] = talos.to_num(dependency[-1])
            self.program_name[section] = program_name
            self.all_program += program_name


    def DataFileCopy(self, model_index):
        """Copies a few data files.
        These data files describe the vertical levels and the emissions
        vertical distribution of a model.
        \param model_index the index of a model.
        """

        general_dict = self.GetGeneralDict(model_index)
        levels = general_dict["%levels%"]
        vertical_distribution = general_dict["%vertical_distribution%"]
        # levels.dat
        if os.path.isfile(self.generic_cfg + "/levels/" + levels):
            command = "cp " + self.generic_cfg + "/levels/"  + levels + " " \
                      + self.output_dir + "/config/"
            commands.getstatusoutput(command)
        else:
            raise Exception, "The file \"" + self.generic_cfg + "/levels/" \
                  + levels + "\" does not exist."
        # vertical-distribution.dat
        if os.path.isfile(self.generic_config_dir["emissions"] + "/input/" \
                          + vertical_distribution):
            command = "cp " + self.generic_config_dir["emissions"] + "/input/" \
                      + vertical_distribution + " " + self.output_dir \
                      + "/config/"
            commands.getstatusoutput(command)
        else:
            raise Exception, "The file \"" + self.generic_config_dir["emissions"] \
                  + "/input/" + vertical_distribution + "\" does not exist."


    def ModelExist(self, model_index):
        """Checks if the directory of a model exist and creates it.
        \param model_index the index of a model.
        @return 0 or 1.
        """

        # Checks if imodel already exists and creates directories.
        imodel = self.parameter.GetStringModel(model_index)
        model_dir = self.parameter.config.model_dir
        model_list = os.listdir(model_dir)
        output_dir = model_dir + "/" + imodel
        if imodel not in model_list:
            # Creates directories.
            os.mkdir(output_dir)
            os.mkdir(output_dir + "/config")
            os.mkdir(output_dir + "/results")
            return 1
        else:
            if len(os.listdir(output_dir + "/results")) != 0:
                return 0
            else:
                return 1


    def SetDefaultDict(self):
        """Sets the default dictionary.

        Sets the value of each parameter by default in the case where a few
        parameters do not appear in the configuration file which describes the
        parameters.
        """

        self.default_dict = self.config_replacement.GetDefaultDict()
        Nphysics_parameter = len(self.parameter.config.physics)
        Nnumerics_parameter = len(self.parameter.config.numerics)
        if len(self.default_dict) < Nphysics_parameter + Nnumerics_parameter:
            raise Exception, "A parameter name is missing in the default" \
                + " dictionary in your " \
                + str(self.config_replacement.__class__) + "."


    def GroundProgram(self, model_index):
        """Adds the programs related to the section 'ground'.
        \param model_index the index of a model.
        """

        preproc_dir = self.polyphemus_dir + "/preprocessing/ground/"
        param_dict = self.GetParameterDict(model_index)
        general_dict = self.GetGeneralDict(model_index)
        imodel = self.parameter.GetStringModel(model_index)
        luc = param_dict["luc"]
        dep = param_dict["deposition_velocity"]
        for prg in self.program_name["ground"]:
            file_dir = create_output_dir(prg, self.dependency[prg],
                                         param_dict)
            tmp_dir = self.data_dir + "/ground/" + file_dir
            if not os.path.isdir(tmp_dir):
                os.makedirs(tmp_dir)
            if os.path.isfile(tmp_dir + "/" + self.binary_file[prg]):
                continue
            id_prg = self.parameter.GetIdProgram(imodel,
                                                 self.dependency[prg])
            if id_prg in self.id_program[prg]:
                continue
            else:
                self.id_program[prg].append(id_prg)
            if prg == "luc":
                prg = prg + "-" + luc
                self.group[prg] = self.group["luc"]
            tmp_config = "/" + prg + ".cfg"
            if prg == "luc-convert":
                dep = "zhang"
                tmp_config = "/" + luc + "_to_" + dep + ".cfg"
            config_list = [self.generic_cfg + "/general.cfg",
                           self.generic_config_dir["ground"] + tmp_config]
            self.CheckProgramExist(prg)
            program = Program(preproc_dir + prg,
                              format = " %a ",
                              group = self.group[prg])
            program.SetConfiguration(config = config_list, mode = "raw",
                                     replacement = general_dict,
                                     path = self.output_dir + "/config/")
            self.program_instance_list.append(program)


    def MeteoProgram(self, model_index):
        """Adds the programs related to the section 'meteo'.
        \param model_index the index of a model.
        """

        preproc_dir = self.polyphemus_dir + "/preprocessing/meteo/"
        str_date_min = self.parameter.config.date_min.strftime("%Y%m%d")
        str_date_max = self.parameter.config.date_max.strftime("%Y%m%d")
        general_dict = self.GetGeneralDict(model_index)
        param_dict = self.GetParameterDict(model_index)
        imodel = self.parameter.GetStringModel(model_index)
        for prg in self.program_name["meteo"]:
            file_dir = create_output_dir(prg, self.dependency[prg],
                                         param_dict)
            tmp_dir = self.data_dir + "/meteo/" + file_dir
            if not os.path.isdir(tmp_dir):
                os.makedirs(tmp_dir)
            if os.path.isfile(tmp_dir + "/" + self.binary_file[prg]):
                continue
            id_prg = self.parameter.GetIdProgram(imodel,
                                                 self.dependency[prg])
            if id_prg in self.id_program[prg]:
                continue
            else:
                self.id_program[prg].append(id_prg)
            config_list = [self.generic_cfg + "/general.cfg",
                           self.generic_config_dir["meteo"] + "/meteo.cfg"]
            self.CheckProgramExist(prg)
            program = Program(self.polyphemus_dir + "/utils/call_dates " \
                                  + preproc_dir + prg,
                              format = " %a " + str_date_min + " " \
                              + str_date_max,
                              group = self.group[prg])
            program.SetConfiguration(config = config_list, mode = "raw",
                                     replacement = general_dict,
                                     path = self.output_dir + "/config/")
            self.program_instance_list.append(program)


    def DepositionProgram(self, model_index):
        """Adds the programs related to the section 'dep'.
        \param model_index the index of a model.
        """

        preproc_dir = self.polyphemus_dir + "/preprocessing/dep/"
        str_date_min = self.parameter.config.date_min.strftime("%Y%m%d")
        str_date_max = self.parameter.config.date_max.strftime("%Y%m%d")
        general_dict = self.GetGeneralDict(model_index)
        param_dict = self.GetParameterDict(model_index)
        imodel = self.parameter.GetStringModel(model_index)
        for prg in self.program_name["dep"]:
            file_dir = create_output_dir(prg, self.dependency[prg],
                                         param_dict)
            tmp_dir  = self.data_dir + "/dep/" + file_dir
            if not os.path.isdir(tmp_dir):
                os.makedirs(tmp_dir)
            if os.path.isfile(tmp_dir + "/" + self.binary_file[prg]):
                continue
            id_prg = self.parameter.GetIdProgram(imodel,
                                                 self.dependency[prg])
            if id_prg in self.id_program[prg]:
                continue
            else:
                self.id_program[prg].append(id_prg)
            config_list = [self.generic_cfg + "/general.cfg",
                           self.generic_config_dir["dep"] + "/dep.cfg"]
            add_file = [self.generic_config_dir["dep"] + "/zhang.cfg"]
            self.CheckProgramExist(prg)
            program = Program(self.polyphemus_dir + "/utils/call_dates " \
                                  + preproc_dir + prg,
                              format = " %a " + str_date_min + " " \
                              + str_date_max,
                              group = self.group[prg])
            program.SetConfiguration(config = config_list, mode = "raw",
                                     replacement = general_dict,
                                     path = self.output_dir + "/config/",
                                     additional_file_list = add_file)
            self.program_instance_list.append(program)


    def EmissionProgram(self, model_index):
        """Adds the programs related to the section 'emissions'.
        \param model_index the index of a model.
        """

        preproc_dir = self.polyphemus_dir + "/preprocessing/emissions/"
        str_date_min = self.parameter.config.date_min.strftime("%Y%m%d")
        str_date_max = self.parameter.config.date_max.strftime("%Y%m%d")
        general_dict = self.GetGeneralDict(model_index)
        param_dict = self.GetParameterDict(model_index)
        imodel = self.parameter.GetStringModel(model_index)
        for prg in self.program_name["emissions"]:
            file_dir = create_output_dir(prg, self.dependency[prg],
                                         param_dict)
            tmp_dir = self.data_dir + "/emissions/"
            if not os.path.isdir(tmp_dir + "surface-emissions/" + file_dir):
                os.makedirs(tmp_dir + "surface-emissions/" + file_dir)
            if not os.path.isdir(tmp_dir + "volume-emissions/" + file_dir):
                os.makedirs(tmp_dir + "volume-emissions/" + file_dir)
            if os.path.isfile(tmp_dir + "/surface-emissions/" \
                              + file_dir + "/" + self.binary_file[prg]):
                continue
            id_prg = self.parameter.GetIdProgram(imodel,
                                                 self.dependency[prg])
            if id_prg in self.id_program[prg]:
                continue
            else:
                self.id_program[prg].append(id_prg)
            config_list = [self.generic_cfg + "/general.cfg",
                           self.generic_config_dir["emissions"] + "/emissions.cfg"]
            self.CheckProgramExist(prg)
            program = Program(self.polyphemus_dir + "/utils/call_dates " \
                                  + preproc_dir + prg,
                              format = " %a " + str_date_min + " " \
                              + str_date_max,
                              group = self.group[prg])
            program.SetConfiguration(config = config_list, mode = "raw",
                                     replacement = general_dict,
                                     path = self.output_dir + "/config/")
            self.program_instance_list.append(program)


    def BioProgram(self, model_index):
        """Adds the programs related to the section 'bio'.
        \param model_index the index of a model.
        """

        preproc_dir = self.polyphemus_dir + "/preprocessing/bio/"
        str_date_min = self.parameter.config.date_min.strftime("%Y%m%d")
        str_date_max = self.parameter.config.date_max.strftime("%Y%m%d")
        general_dict = self.GetGeneralDict(model_index)
        param_dict = self.GetParameterDict(model_index)
        imodel = self.parameter.GetStringModel(model_index)
        for prg in self.program_name["bio"]:
            file_dir = create_output_dir(prg, self.dependency[prg],
                                         param_dict)
            tmp_dir = self.data_dir + "/bio/" + file_dir
            if not os.path.isdir(tmp_dir):
                os.makedirs(tmp_dir)
            if os.path.isfile(tmp_dir + "/" + self.binary_file[prg]):
                continue
            id_prg = self.parameter.GetIdProgram(imodel,
                                                 self.dependency[prg])
            if id_prg in self.id_program[prg]:
                continue
            else:
                self.id_program[prg].append(id_prg)
            config_list = [self.generic_cfg + "/general.cfg",
                           self.generic_config_dir["bio"] + "/bio.cfg"]
            self.CheckProgramExist(prg)
            program = Program(self.polyphemus_dir + "/utils/call_dates " \
                                  + preproc_dir + prg,
                              format = " %a " + str_date_min + " " \
                              + str_date_max,
                              group = self.group[prg])
            program.SetConfiguration(config = config_list, mode = "raw",
                                     replacement = general_dict,
                                     path = self.output_dir + "/config/")
            self.program_instance_list.append(program)



    def IcProgram(self, model_index):
        """Adds the programs related to the section 'ic'.
        \param model_index the index of a model.
        """

        preproc_dir = self.polyphemus_dir + "/preprocessing/ic/"
        general_dict = self.GetGeneralDict(model_index)
        param_dict = self.GetParameterDict(model_index)
        imodel = self.parameter.GetStringModel(model_index)
        for prg in self.program_name["ic"]:
            file_dir = create_output_dir(prg, self.dependency[prg],
                                         param_dict)
            tmp_dir = self.data_dir + "/ic/" + file_dir
            if not os.path.isdir(tmp_dir):
                os.makedirs(tmp_dir)
            if os.path.isfile(tmp_dir + "/" + self.binary_file[prg]):
                continue
            id_prg = self.parameter.GetIdProgram(imodel,
                                                 self.dependency[prg])
            if id_prg in self.id_program[prg]:
                continue
            else:
                self.id_program[prg].append(id_prg)
            config_list = [self.generic_cfg + "/general.cfg",
                           self.generic_config_dir["ic"] + "/ic.cfg"]
            self.CheckProgramExist(prg)
            program = Program(preproc_dir + prg, format = " %a ",
                              group = self.group[prg])
            program.SetConfiguration(config = config_list, mode = "raw",
                                     replacement = general_dict,
                                     path = self.output_dir + "/config/")
            self.program_instance_list.append(program)


    def BcProgram(self, model_index):
        """Adds the programs related to the section 'bc'.
        \param model_index the index of a model.
        """

        preproc_dir = self.polyphemus_dir + "/preprocessing/bc/"
        str_date_min = self.parameter.config.date_min.strftime("%Y%m%d")
        str_date_max = self.parameter.config.date_max.strftime("%Y%m%d")
        general_dict = self.GetGeneralDict(model_index)
        param_dict = self.GetParameterDict(model_index)
        imodel = self.parameter.GetStringModel(model_index)
        for prg in self.program_name["bc"]:
            file_dir = create_output_dir(prg, self.dependency[prg],
                                         param_dict)
            tmp_dir = self.data_dir + "/bc/" + file_dir
            if not os.path.isdir(tmp_dir):
                os.makedirs(tmp_dir)
            if os.path.isfile(tmp_dir + "/" + self.binary_file[prg]):
                continue
            id_prg = self.parameter.GetIdProgram(imodel,
                                                 self.dependency[prg])
            if id_prg in self.id_program[prg]:
                continue
            else:
                self.id_program[prg].append(id_prg)
            config_list = [self.generic_cfg + "/general.cfg",
                           self.generic_config_dir["bc"] + "/bc-dates.cfg"]
            self.CheckProgramExist(prg)
            program = Program(preproc_dir + prg,
                              format = " %a " + str_date_min + " " \
                              + str_date_max,
                              group = self.group[prg])
            program.SetConfiguration(config = config_list, mode = "raw",
                                     replacement = general_dict,
                                     path = self.output_dir + "/config/")
            self.program_instance_list.append(program)


    def DriverProgram(self, model_index):
        """Adds the programs related to the section 'driver'.
        \param model_index the index of a model.
        """

        date_min = self.parameter.config.date_min
        date_max = self.parameter.config.date_max
        str_date_min = date_min.strftime("%Y%m%d")
        general_dict = self.GetGeneralDict(model_index)
        param_dict = self.GetParameterDict(model_index)
        # Time step
        time_step = param_dict["time_step"]
        delta_t = float(talos.to_num(time_step))
        Nday = (date_max - date_min).days
        Nt = Nday * 24. * 3600. / delta_t
        Nt = int(Nt)
        if 3600. % delta_t != 0.:
            raise Exception, "The time step = \"" + time_step + "\". " \
                  + "The division " + "\"3600. / time_step\" must be" \
                  + " equal to an integer."
        interval_length = 3600. / delta_t
        bc_date_min = get_bc_init_date(date_min)

        # Photolytic constant.
        photolytic = param_dict["photolytic_constant"]

        # Other numerical datas.
        with_air_density = param_dict["with_air_density"]
        with_source_splitting = param_dict["with_source_splitting"]
        with_forced_concentration = param_dict["with_forced_concentration"]
        splitting_method = param_dict["splitting_method"]

        # Program choice.
        chemistry = param_dict["chemistry"]
        prg_dir = "/processing/"
        if chemistry == "racm":
            polair_data_config = "polair3d-data.cfg"
            species = "species-" + chemistry + ".dat"
            prg = "polair3d"
            prg_dir += "racm/"
        elif chemistry == "radm":
            polair_data_config = "polair3d-data-radm.cfg"
            species = "species-" + chemistry + ".dat"
            prg = "polair3d-radm"
            prg_dir += "radm/"
        if len(self.parameter.config.input_data[0]) != 0:
            prg += "-perturbation"
        if photolytic == "zenith_angle":
            prg += "-zenith_angle_photolysis"
        if prg not in self.all_program:
            raise Exception, "The program name: \"" + prg + "\" does not" \
                  + " appear in file \"" + self.config_program + "\"."

        # species.dat
        if os.path.isfile(self.generic_cfg + "/species/" + species):
            command = "cp " + self.generic_cfg + "/species/" + species + " " \
                      + self.output_dir + "/config/"
            commands.getstatusoutput(command)
        else:
            raise Exception, "The file \"" + self.generic_cfg + "/species/" \
                  + species + "\" does not exist."

        # The other dictionaries.
        second_dict = {"%time_step%": time_step,
                       "%date_min%": str_date_min,
                       "%bc_date_min%":  str(bc_date_min.date()) + "_12",
                       "%Nt%": str(Nt),
                       "%polair-data%": polair_data_config,
                       "%species%": species,
                       "%saved_interval_length%": str(int(interval_length)),
                       "%with_air_density%": with_air_density,
                       "%with_source_splitting%": with_source_splitting,
                       "%with_forced_concentration%": \
                           with_forced_concentration,
                       "%splitting_method%": splitting_method}
        perturbation_dict = self.GetPerturbationDict(model_index)
        general_dict.update(second_dict)
        general_dict.update(perturbation_dict)

        config_list = [self.generic_config_dir["driver"] + "/polair3d.cfg"]
        add_file_list = [self.generic_config_dir["driver"] \
                             + "/polair3d-saver.cfg",
                         self.generic_config_dir["driver"] \
                             + "/" + polair_data_config,
                         self.generic_config_dir["driver"] \
                             + "/perturbation.cfg"]

        self.CheckProgramExist(prg)
        program = Program(self.polyphemus_dir + prg_dir + prg,
                          format = " %a ", group = self.group[prg])
        program.SetConfiguration(config = config_list, mode = "raw",
                                 replacement = general_dict,
                                 path = self.output_dir + "/",
                                 additional_file_list = add_file_list)
        self.program_instance_list.append(program)



    def CopyConfigurationFile(self):
        """Copies the configurations files in the model directory.
        """

        model_dir = self.parameter.config.model_dir
        # Copies program configuration file.
        if os.path.isfile(self.config_program):
            command = "cp " + self.config_program  + " " \
                      + model_dir + "/"
            commands.getstatusoutput(command)
        else:
            raise Exception, "The file \"" + self.config_program \
                  + "\" does not exist."
        # Copies parameter configuration file.
        config_parameter = self.parameter.config_filename
        if os.path.isfile(config_parameter):
            command = "cp " + config_parameter  + " " \
                      + model_dir + "/"
            commands.getstatusoutput(command)
        else:
            raise Exception, "The file \"" + config_parameter \
                  + "\" does not exist."


    def GeneratePerturbationFile(self, model_index, directory):
        """Replaces several variables and copies the generic perturbation
        file in the directory of the model.

        \param model_index the index of a model.
        \param directory the directory where the perturbation file will be
        stored.
        """

        if not os.path.isdir(directory):
            raise Exception, "Unable to find the directory: \"" \
                  + directory + "\"."
        perturbation_dict = self.GetPerturbationDict(model_index)
        generic_file = self.generic_config_dir["driver"] + "/perturbation.cfg"
        config = Configuration(file_list = generic_file, mode = "raw",
                               path = directory)
        config.SetConfiguration(config = perturbation_dict, mode = "raw",
                                path = directory)



    def GetEnsemble(self, model_index = None,
                    group = "polyphemus"):
        """Returns an ensemble of programs ready to launch.
        \param model_index a list of index of models.
        \param group the name of the file where there is the hosts list.
        @return An instance 'Polyphemus'.
        """

        self.InitEnsemble()

        # Checks polyphemus_dir.
        if len(self.polyphemus_dir) == 0:
            raise Exception, "The polyphemus directory is not set." \
                + " Please use the method SetPolyphemusDirectory."

        if model_index == None:
            model_index = range(self.Nmodel)
        ensemble = Polyphemus(Network(group))
        for m in model_index:
            if self.ModelExist(m):
                self.DataFileCopy(m)
                if len(self.config.ground[0]) != 0:
                    self.GroundProgram(m)
                if len(self.config.meteo[0]) != 0:
                    self.MeteoProgram(m)
                if len(self.config.dep[0]) != 0:
                    self.DepositionProgram(m)
                if len(self.config.emissions[0]) != 0:
                    self.EmissionProgram(m)
                if len(self.config.bio[0]) != 0:
                    self.BioProgram(m)
                if len(self.config.ic[0]) != 0:
                    self.IcProgram(m)
                if len(self.config.bc[0]) != 0:
                    self.BcProgram(m)
                if not self.only_preprocessing and \
                       len(self.config.driver[0]) != 0:
                    self.DriverProgram(m)

        for prg in self.program_instance_list:
            ensemble.AddProgram(prg)
        if len(self.program_instance_list) == 0:
            print "No program to launch in your \"" \
                  + str(self.__class__).split(".")[-1] + "\"."

        return ensemble


    def GetEnsembleProgramSection(self, section, model_index = None,
                                  group = "polyphemus"):
        """Returns an ensemble of programs related to a list of sections ready to
        launch.

        \param section the name of the section or a list of names of the
        section.
        \param model_index a list of index of models.
        \param group the name of the file where there is the hosts list.
        @return An instance 'Polyphemus'.
        """

        self.InitEnsemble()

        # Checks polyphemus_dir.
        if len(self.polyphemus_dir) == 0:
            raise Exception, "The polyphemus directory is not set." \
                + " Please use the method SetPolyphemusDirectory."

        if model_index == None:
            model_index = range(self.Nmodel)
        ensemble = Polyphemus(Network(group))
        # Checks if section exists.
        section_list = [x[2] for x in self.config.content]
        if isinstance(section, str):
            section = [section]
        for s in section:
            if s not in section_list:
                raise Exception, "Unable to find section name: \"" \
                      + s + "\" in file: \"" + self.config.filename + "\"."
        # Gets the ensemble.
        for m in model_index:
            if self.ModelExist(m):
                self.DataFileCopy(m)
                if "ground" in section:
                    self.GroundProgram(m)
                if "meteo" in section:
                    self.MeteoProgram(m)
                if "dep" in section:
                    self.DepositionProgram(m)
                if "emissions" in section:
                    self.EmissionProgram(m)
                if "bio" in section:
                    self.BioProgram(m)
                if "ic" in section:
                    self.IcProgram(m)
                if "bc" in section:
                    self.BcProgram(m)
                if "driver" in section:
                    self.DriverProgram(m)

        for prg in self.program_instance_list:
            ensemble.AddProgram(prg)
        if len(self.program_instance_list) == 0:
            print "No program to launch in your \"" \
                  + str(self.__class__).split(".")[-1] + "\"."

        return ensemble


    def AddParameter(self, new_config_parameter, new_config_program):
        """Adds parameters.
        \param new_config_parameter the path to the configuration file which
        describes the different parameters.
        \param new_config_program the path to the configuration file which
        enumerates the programs and their dependencies.

        \note In construction.
        \warning This method has not been tested.
        """

        import shutil
        new_program = EnsembleProgram(new_config_parameter,
                                      new_config_program,
                                      self.polyphemus_dir)
        # Exceptions
        self.Check()
        self.CheckDataDirectory()
        if self.parameter.Nparameter >= new_program.parameter.Nparameter:
            raise Exception, "The new " \
                  + str(new_program.parameter.__class__).split(".")[-1] \
                  + "what you want to add has a parameter number " \
                  + "equal or smaller than the old one. " \
                  + "Please, check your file: \"" \
                  + new_config_parameter + "\"."
        for name in self.parameter.name:
            if name not in new_program.parameter.name:
                raise Exception, "The parameter name: \"" + name \
                      + "\" must appear in file: \"" \
                      + new_config_parameter + "\"."
        for value in self.parameter.value:
            if value not in new_program.parameter.value:
                raise Exception, "The value: \"" + value \
                      + "\" must appear in file: \"" \
                      + new_config_parameter + "\"."
        if self.program_name != new_program.program_name:
            raise Exception, "Program name is different in \"" \
                  + new_program.config_program + "\"."
        # Directories
        self.data_dir = self.parameter.config.data_dir
        model_dir = self.parameter.config.model_dir
        new_model_dir = new_program.parameter.config.model_dir
        # Looks for the new parameter index.
        index = []
        for name in new_program.parameter.name:
            if name not in self.parameter.name:
                index.append(new_program.parameter.name.index(name))
        # Old model identities which are in model directory.
        old_imodel = []
        index_old_imodel = []
        for m in range(self.Nmodel):
            imodel = self.parameter.GetStringModel(m)
            if imodel in os.listdir(model_dir):
                old_imodel.append(imodel)
                index_old_imodel.append(m)
        # Inserts the new index and writes the new identity ensemble file.
        for m in range(self.Nmodel):
            for i in index:
                self.parameter.id_ensemble[m].insert(i, 0)
        self.parameter.Nparameter = new_program.parameter.Nparameter
        new_id_ens_file = "son_of-" + self.parameter.id_ens_file
        self.parameter.WriteIdEnsembleInFile(new_id_ens_file)
        self.parameter = EnsembleParameter(new_config_parameter,
                                           new_id_ens_file)
        self.CopyConfigurationFile()
        # Moves the model directories.
        for m in range(len(old_imodel)):
            shutil.move(model_dir + "/" + old_imodel[m],
                        new_model_dir + "/" \
                        + self.parameter.GetStringModel(index_old_imodel[m]))

        new_program.ReadIdEnsembleFile(self.parameter.id_ens_file)
        new_program.Check()
        new_program.CheckDataDirectory()

        # Which program of which dependency has changed ?
        prg_dependency_change = []
        for prg in self.program_name:
            if len(self.dependency[prg]) != len(new_program.dependacy[prg]):
                prg_dependency_change.append(prg)

        # Changes some data directories.
        for m in range(self.Nmodel):
            param_dict = self.GetParameterDict(m)
            new_param_dict = new_program.GetParameterDict(m)
            for prg in prg_dependency_change:
                file_dir = create_output_dir(prg, self.dependency[prg],
                                             param_dict)
                new_file_dir = create_output_dir(new_program.dependency[prg],
                                                 new_param_dict)
                section = self.GetProgramSection(prg)
                if section == "emissions":
                    tmp = ["surface-emissions", "volume-emissions"]
                    for i in range(2):
                        section = section + "/" + tmp[i]
                        tmp_dir = self.data_dir + "/" + section + "/" \
                                  + file_dir
                        new_tmp_dir = self.data_dir + "/" + section + "/" \
                                      + new_file_dir
                        shutil.move(tmp_dir, new_tmp_dir)
                else:
                    tmp_dir = self.data_dir + "/" + section + "/" \
                              + file_dir
                    new_tmp_dir = self.data_dir + "/" + section + "/" \
                                  + new_file_dir
                    shutil.move(tmp_dir, new_tmp_dir)
