import argparse
import configparser
import os
import requests


def _as_token_from_env(env_name="AS_TOKEN") -> str:
    """Read token value from env (default name AS_TOKEN)"""
    return os.environ[env_name]


def _as_token_from_file(token_file=None, server="www") -> str:
    """Read token from ~/.amplisim/token.ini or in specified ini file"""
    # default token file
    if token_file is None:
        token_file = f"{os.path.expanduser('~')}/.amplisim/token.ini"

    # check input
    if not os.path.exists(token_file):
        raise FileNotFoundError(f"token_file: {token_file} not found.")

    # read & return token
    config = configparser.ConfigParser()
    config.read(token_file)
    return config.get(server, "token")


def as_token(env_name="AS_TOKEN", token_file=None, server="www") -> str:
    """Read token from ~/.amplisim.token file or ENV var (AS_TOKEN)"""
    # try to read from env & fallback to file if not raise
    try:
        token = _as_token_from_env(env_name)
    except KeyError:
        token = _as_token_from_file(token_file, server)
    except Exception as exc:
        msg = f"Couldn't read token ({env_name}) from env or {token_file}"
        raise Exception(msg) from exc

    return token


def as_auth(client, token, server="www") -> dict:
    """Authentication to AmpliSIM API service"""
    # auth get request
    url_csrf = f"https://{server}.amplisim.com/user/login/"
    response = client.get(url_csrf)

    # check response
    if response.status_code != 200:
        raise IOError(f"Error, failed auth on AmpliSIM API {response.status_code}")
    print("Success, connected to AmpliSIM API")

    # handle header names
    if server == "www":
        csrfname = "csrftoken"
    else:
        csrfname = f"csrftoken_{server}"

    # return headers
    csrftoken = client.cookies[csrfname]
    headers = {"ASSERVICETOKEN": token, "X-CSRFToken": csrftoken, "Referer": url_csrf}

    return headers


def as_post(url, token=None, server="www", data=None, json=None, files=None) -> None:
    """Perform post request on url using token

    Args:
        url (str): _description_
        token (string, optional): token to access API. Defaults to None.
        server (string,  optional): which server you request (www or edu). Defaults to www)
        data (dict, optional): data. Defaults to None.
        json (dict, optional): json data. Defaults to None.
        files (dict, optional): file encoded, i.e {'file': ('arch.zip', open('arch.zip', 'rb'))} . Defaults to None.
    """
    # Creating session
    print(f"Connecting to {url} using POST")
    client = requests.session()

    # Get token
    token = token
    if token is None:
        token = as_token(server=server)

    # Authentication
    headers = as_auth(client, token, server=server)

    # Send POST request
    try:
        response = client.post(
            url,
            timeout=(10, 10),
            headers=headers,
            data=data,
            json=json,
            files=files,
        )
        print(f"{url} responded to POST request")
    except Exception as exc:
        raise IOError("Error, connection failed using post on {url}") from exc

    if response.status_code != 200:
        msg = (
            f"Error, connection failed using post on {url} service answered with "
            f"http status {response.status_code}"
        )
        raise IOError(msg)
    print(
        f"Using post on {url} service answered with http status {response.status_code}"
    )


def as_params_check(model_name, zipfile, file_ext, encoding="utf-8", server="www") -> None:
    """as_params_check _summary_

    Args:
        model_name (str): model name
        zipfile (str): path to zipfile
        file_ext (str, list[str]): file extension to read in zipfile
        encoding (str, optional): encoding of file in zipfile. Defaults to "utf-8".
        server (string,  optional): which server you request (www or edu). Defaults to www)
    """
    # check input
    if not os.path.exists(zipfile):
        raise FileExistsError(f"Zipfile {zipfile} doesn't exist.")

    # prepare data to post (file & data)
    file = {"file": (os.path.basename(zipfile), open(zipfile, "rb"))}
    data = {
        "model_name": model_name,
        "input_file": os.path.basename(zipfile),
        "file_ext": file_ext,
        "encoding": encoding,
    }

    # make post request
    url_check = f"https://{server}.amplisim.com/airquality/params/check/"
    as_post(url_check, token=None, data=data, files=file, server=server)


def as_params_update(model_name, git_kwargs, encoding="utf-8", server="www") -> None:
    """Launch request to update params to AmpliSIM API

    Args:
        model_name (str): model name
        git_kwargs (dict): proj_id, ref & list of file to fetch from repo
        encoding (str, optional): files encoding. Defaults to "utf-8".
        server (string,  optional): which server you request (www or edu). Defaults to www)
    """
    # check input
    mandatory_key = {"proj_id", "ref", "list_file"}
    if not set(git_kwargs).issubset(mandatory_key):
        raise ValueError(f"git_kwargs should contain {mandatory_key}.")

    # prepare data to post
    data = {
        "model_name": model_name,
        "encoding": encoding,
        "proj_id": git_kwargs.get("proj_id"),
        "ref": git_kwargs.get("ref"),
        "list_file": git_kwargs.get("list_file"),
    }

    # make post request
    url_update = f"https://{server}.amplisim.com/airquality/params/update/"
    as_post(url_update, token=None, json=data, server=server)


if __name__ == "__main__":
    # ------------------------------------------
    # Parse args: define variables & parameters
    # ------------------------------------------
    parser = argparse.ArgumentParser(
        prog="Req_API_ASParams", description="Module to launch simulation "
    )

    # which api call
    parser.add_argument(
        "action", help="Req. AS API to check or update", choices=["check", "update"]
    )
    # common args to both API (model_name & encoding)
    parser.add_argument(
        "model_name",
        help="Which model",
        choices=["pepp", "ping", "sing", "munich", "polair3d"],
    )
    parser.add_argument("encoding", help="Which encoding for file", default="utf-8")
    parser.add_argument(
        "-server",
        help="Which server request (prod = www, pre-prod = edu)",
        default="www",
        choices=["www", "edu"],
        required=False
    )

    # opt args depending whether you call for check or update
    # arg requested by check API
    parser.add_argument("-zipfile", help="Path to zipfile to scan", required=False)
    parser.add_argument(
        "-file_ext",
        help="File extension to scan in zipfile",
        required=False,
    )

    # arg requested by update API
    parser.add_argument(
        "-proj_id", help="Proj_id code as in gitlab repo", type=int, required=False
    )
    parser.add_argument(
        "-ref", help="Ref (i.e. branch or sha in gitlab...", type=str, required=False
    )
    parser.add_argument(
        "-list_file", nargs="*", help="Filelist to parse", required=False
    )

    # parse the params & check
    args = parser.parse_args()

    # API Params Check ==> parse cfg in archive & return params by mail (or error)
    if args.action == "check":
        # check if the right opt args are specified
        if args.zipfile is None:
            raise ValueError("Zipfile is mandatory to call ASParams check API!")
        if args.file_ext is None:
            raise ValueError("File_ext is mandatory to call ASParams check API!")
        # submit to API
        as_params_check(
            model_name=args.model_name,
            zipfile=args.zipfile,
            file_ext=args.file_ext,
            encoding=args.encoding,
            server=args.server
        )

    # API Params Update ==> parse cfg in gitlab & return params by mail (or error)
    if args.action == "update":
        # check if the right opt args are specified
        if args.proj_id is None:
            raise ValueError("proj_id is mandatory to call ASParams check API!")
        if args.ref is None:
            raise ValueError("ref is mandatory to call ASParams check API!")
        if args.list_file is None:
            raise ValueError("list_file is mandatory to call ASParams check API!")
        # submit to api
        as_params_update(
            model_name=args.model_name,
            git_kwargs={
                "proj_id": args.proj_id,
                "ref": args.ref,
                "list_file": args.list_file,
            },
            encoding=args.encoding,
            server=args.server
        )



# if __name__ == "__main__":
#     # API Params Check ==> parse cfg in archive & return params by mail (or error)

#     # Gaussien Plume
#     # zipfile = "plume.zip"
#     # model_name = "pepp"

#     # Polair3D
#     # zipfile = "polair3d.zip"
#     # model_name = "polair3d"    

#     zipfile = "polair3d.zip"
#     model_name = "polair3d"        
#     #"/Users/vincent/AmpliSIM Dropbox/_Technique/_Projet/_Vincent/CEREA/params_cfg/data/plume.zip"
#     as_params_check(
#         model_name=model_name, zipfile=zipfile, file_ext="cfg", encoding="utf-8"
#     )

#     # # API Params Update ==> parse cfg in gitlab & return params by mail (or error)
#     # git_kwargs = {
#     #     "proj_id": 330,
#     #     "ref": "c7301ed3a987a6612c5d92190cfb0601ed06d7c5",  # pushed from mf with error
#     #     "list_file": ["processing/gaussian/plume.cfg"],
#     # }
#     # as_params_update(model_name="pepp", git_kwargs=git_kwargs, encoding="utf-8")
