// Copyright (C) 2005-2007, ENPC - INRIA - EDF R&D
// Author(s): Vivien Mallet
//
// This file is part of the air quality modeling system Polyphemus.
//
// Polyphemus is developed in the INRIA - ENPC joint project-team CLIME and in
// the ENPC - EDF R&D joint laboratory CEREA.
//
// Polyphemus is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 2 of the License, or (at your option)
// any later version.
//
// Polyphemus is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// For more information, visit the Polyphemus web site:
//      http://cerea.enpc.fr/polyphemus/


#ifndef POLYPHEMUS_FILE_MODULES_CHEMISTRY_CHEMISTRYRADM_HXX


#include <vector>
#include "AtmoData.hxx"
#include "BaseModuleParallel.hxx"


namespace Polyphemus
{


  using namespace std;
  using namespace AtmoData;


  //////////////////////
  // FORTRAN FUNCTION //
  //////////////////////


#define _chem_radm chem_radm_
#define _dimensions_radm dimensions_radm_

  extern "C"
  {
    void _chem_radm(int *, int *, int*, int*, int*, int*, int*, int*,
                    int*, int*, int*,
                    int*, double*, double*, double*, double*, double*,
                    double*, double*, double*, double*, double*,
                    double*, double*, double*, double*, double*, int*,
                    double *, double*, double*, int*);

    void _dimensions_radm(int*, int*, int*);
  }


  ///////////////////
  // CHEMISTRYRADM //
  ///////////////////


  //! This class is a numerical solver for the chemical mechanism RADM.
  /*! It uses a second-order Rosenbrock method.
   */
  template<class T>
  class ChemistryRADM: public BaseModuleParallel
  {

  protected:

    //! Number of species.
    int Ns;
    //! Number of reactions.
    int Nr;
    //! Number of photolysis reactions.
    int Nr_photolysis;
    //! Number of species with volume sources.
    int Ns_source;
    //! Number of levels with volume sources.
    int Nz_source;

    //! Number of sub-cycles.
    int Ncycle;

    //! Sorted list of species names.
    vector<string> species_list;

    //! Molecular weights of species.
    Array<T, 1> molecular_weight;
    /*! \brief Map between photolysis reactions names and their indices in
      photolysis reactions.
    */
    map<string, int> photolysis_reaction_name;
    //! Indices of photolysis reactions among other reactions.
    Array<int, 1> photolysis_reaction_index;
    //! Indices of species with volume sources.
    Array<int, 1> source_index;
    /*! \brief photolysis rates obtained by the tabulation generated by SPACK
      or by binary files from FastJ.
    */
    int option_photolysis_tabulation;

  public:

    /*** Constructor ***/

    ChemistryRADM();

    /*** Other methods ***/

    template<class ClassModel>
    void Init(ClassModel& Model);

    template<class ClassModel>
    void Forward(ClassModel& Model);

    void Forward(T current_time,
                 Data<T, 3>& Attenuation_i,
                 Data<T, 3>& SpecificHumidity_i,
                 Data<T, 3>& Temperature_i,
                 Data<T, 3>& Pressure_i,
                 Data<T, 4>& VolumeEmission_i,
                 Data<T, 4>& PhotolysisRate_i,
                 T next_time,
                 Data<T, 3>& Attenuation_f,
                 Data<T, 3>& SpecificHumidity_f,
                 Data<T, 3>& Temperature_f,
                 Data<T, 3>& Pressure_f,
                 Data<T, 4>& VolumeEmission_f,
                 Data<T, 4>& PhotolysisRate_f,
                 Array<T, 1>& Longitude,
                 Array<T, 1>& Latitude,
                 Data<T, 4>& Concentration);

  };


} // namespace Polyphemus.


#define POLYPHEMUS_FILE_MODULES_CHEMISTRY_CHEMISTRYRADM_HXX
#endif
