#### Helper functions ####

#' Predicts average remaining masses and remaining masses for new samples.
#'
#' @noRd
#' @keywords Internal
hpmd_predict_fit_4_helper_1 <- function(newdata, m_config) {

  # predict remaining mass
  newdata <-
    newdata %>%
    dplyr::mutate(
      l0 =
        dplyr::case_when(
          incubation_duration <= 0 ~ posterior::as_rvar(0),
          TRUE ~ .data$l0
        ),
      mass_relative_mass_mu =
        (m0 - .data$l0) / (1.0 + (.data$alpha_2 - 1.0) * .data$k0 * .data$incubation_duration)^(1.0 / (.data$alpha_2 - 1.0)),
      mass_relative_mass_mu =
        posterior::rvar_ifelse(mass_relative_mass_mu >= 1.0, 1.0 - m_config$s, mass_relative_mass_mu),
      mass_relative_mass =
        posterior::rvar_rng(
          stats::rbeta,
          n = length(.data$mass_relative_mass_mu),
          shape1 = .data$mass_relative_mass_mu * .data$phi_2,
          shape2 = (1 - .data$mass_relative_mass_mu) * .data$phi_2
        ),
      mass_relative_mass_mu = mass_relative_mass_mu,
      mass_relative_mass = mass_relative_mass
    )

  newdata

}


#### Exported ####

#' Predicts decomposition rates and initial leaching losses for model HPM-leaching in Teickner et al. (2024) for constant decomposition conditions
#'
#' This function takes the posterior of  model HPM-leaching in
#' \insertCite{Teickner.2024a;textual}{hpmdpredict} and a data frame
#' specifying prediction controls and optionally other parameter values and
#' predicts Sphagnum decomposition rates, initial leaching losses, and remaining
#' masses during a litterbag experiment assuming constant conditions.
#'
#' For parameters `phi_2_p2` (measurement errors for remaining masses) and
#' `alpha_2` (factor that controls how fast decomposition rates decrease with
#' increasing mass loss), the functions assumes by default average values
#' estimated for species across the studies analyzed in
#' \insertCite{Teickner.2024a;textual}{hpmdpredict}. If other measurement errors
#' for remaining masses or values for `alpha_2` should to be assumed, this has
#' to be specified via `newdata`.
#'
#' @param newdata A data frame with a row for each prediction to generate. The
#' data frame must contain the following columns:
#' \describe{
#'   \item{`incubation_duration`}{A numeric vector with the incubation durations.}
#'   \item{`m0`}{A numeric vector with the initial mass of the litter.}
#'   \item{`layer_degree_of_saturation_1`}{A numeric vector with the degree of
#'   saturation during the incubation.}
#'   \item{`layer_water_table_depth_to_surface_1`}{A numeric vector with the
#'   water table depth to the surface (cm) during the incubation.}
#'   \item{`sample_depth_lower`}{A numeric vector with the depth of the sample
#'   below the peat surface during the incubation.}
#'   \item{`hpm_taxon_rank_value`}{A character value defining the Sphagnum
#'   species. Allowed species names are defined in `hpm_taxon_rank_value_allowed`}
#' }
#' The data frame may contain additional column, in particular the parameter
#' values estimated by model HPM-leaching may be changed to other values. See
#' the readme file for a list of parameters whose values can be changed.
#'
#' @return `newdata` with additional columns with predicted initial leaching
#' losses, decomposition rates, and remaining masses. Additional columns are:
#' \describe{
#'   \item{m69_p1}{See readme.}
#'   \item{m69_p2}{See readme.}
#'   \item{m68_p1}{See readme.}
#'   \item{m68_p2}{See readme.}
#'   \item{m68_p3_2}{See readme.}
#'   \item{hpm_l_2_p1}{See readme.}
#'   \item{hpm_l_2_p3}{See readme.}
#'   \item{hpm_l_2_p4}{See readme.}
#'   \item{hpm_k_2_p1}{See readme.}
#'   \item{alpha_2}{See readme.}
#'   \item{phi_2}{See readme.}
#'   \item{hpm_k_2}{Average decomposition rates predicted by the HPM
#'   decomposition module \insertCite{Frolking.2010}{hpmdpredict} (yr$^{-1}$).}
#'   \item{k0}{Decomposition rate of an individual litterbag experiment predicted
#'   by the HPM decomposition module \insertCite{Frolking.2010}{hpmdpredict}
#'   (yr$^{-1}$).}
#'   \item{l0}{Initial leaching loss of an individual litterbag experiment predicted
#'   by HPM-leaching (g/g).}
#'   \item{mass_relative_mass_mu}{Average fraction of the initial mass remaining
#'   predicted by the HPM decomposition module
#'   \insertCite{Frolking.2010}{hpmdpredict} (g/g).}
#'   \item{mass_relative_mass}{Fraction of the initial mass remaining of an
#'   individual litterbag as predicted by the HPM decomposition module
#'   \insertCite{Frolking.2010}{hpmdpredict} (g/g).}
#' }
#'
#' @examples
#' set.seed(4334)
#' newdata <-
#'   tibble::tibble(
#'     incubation_duration = seq(0, 100, 1),
#'     m0 = 1,
#'     layer_degree_of_saturation_1 = 0.4,
#'     layer_water_table_depth_to_surface_1 = 20,
#'     sample_depth_lower = 10,
#'     hpm_taxon_rank_value = "Sphagnum fuscum"
#'  )
#'
#' res <- hpmd_predict_fit_4(newdata)
#'
#' # Example for modifying model parameters: Here, a different optimum degree of
#' # saturation for decomposition
#' newdata$m69_p1 <- 0.4
#' res1 <- hpmd_predict_fit_4(newdata)
#'
#' plot(mean(res$mass_relative_mass) ~ res$incubation_duration, type = "l")
#' lines(mean(res1$mass_relative_mass) ~ res1$incubation_duration, col = "red")
#'
#' @references
#'    \insertAllCited{}
#'
#' @export
hpmd_predict_fit_4 <- function(newdata) {

  # hpmd_stan_fit_4 <- NULL
  #hpmd_data_stan_4 <- NULL
  #hpmd_data_hpm_microhabitat2 <- NULL
  # utils::data(hpmd_stan_fit_4, envir = environment())
  # utils::data(hpmd_data_stan_4, envir = environment())
  # utils::data(hpmd_data_hpm_microhabitat2, envir = environment())

  m_parameters <- hpmd_stan_draws_4
  m_config <- hpmd_data_stan_4

  rvar_inv_logit <- posterior::rfun(stats::binomial()$linkinv)

  # transformed data
  newdata <-
    newdata %>%
    dplyr::mutate(
      layer_water_table_depth_1 = .data$layer_water_table_depth_to_surface_1 - .data$sample_depth_lower
    )

  # predict decomposition rate
  newdata <-
    newdata %>%
    dplyr::mutate(
      hpm_microhabitat2 =
        factor(.data$hpm_taxon_rank_value, levels = levels(m_config$index_hpm_microhabitats_to_hpm))
    ) %>%
    dplyr::left_join(
      m_parameters %>%
        dplyr::select(! dplyr::all_of(setdiff(intersect(colnames(newdata), colnames(m_parameters)), "hpm_microhabitat2"))),
      by = "hpm_microhabitat2"
    ) %>%
    dplyr::mutate(
      hpm_k_2 =
        m68(
          layer_degree_of_saturation_1 = .data$layer_degree_of_saturation_1,
          layer_water_table_depth_1 = .data$layer_water_table_depth_1,
          m69_p1 = .data$m69_p1,
          m69_p2 = .data$m69_p2,
          m68_p1 = .data$m68_p1,
          m68_p2 = .data$m68_p2,
          m68_p3 = .data$m68_p3_2,
          m68_p4 = 1,
          m68_p5 = 1
        ),
      k0 =
        posterior::rvar_rng(
          stats::rgamma,
          n = length(.data$hpm_k_2_p1),
          shape = .data$hpm_k_2_p1,
          rate = .data$hpm_k_2_p1 / .data$hpm_k_2
        )
    )

  # predict initial leaching loss
  newdata <-
    newdata %>%
    dplyr::mutate(
      l0 =
        if("l0" %in% colnames(.)) {
          posterior::as_rvar(.data$l0)
        } else {
          mu <- rvar_inv_logit(.data$hpm_l_2_p1 + .data$hpm_l_2_p3 * .data$layer_degree_of_saturation_1)
          posterior::rvar_rng(
            stats::rbeta,
            n = length(mu),
            shape1 = mu * .data$hpm_l_2_p4,
            shape2 = (1 - mu) * .data$hpm_l_2_p4
          )
        }
    )

  # remaining masses
  newdata <- hpmd_predict_fit_4_helper_1(newdata = newdata, m_config = m_config)

  newdata |>
    dplyr::select(! dplyr::all_of(c("hpm_microhabitat2", "layer_water_table_depth_1")))

}


#' Predicts decomposition rates and initial leaching losses for model HPM-leaching in Teickner et al. (2024) for one litter sample with variable environmental conditions
#'
#' Here, it is assumed that one litter sample decomposes where conditions are
#' constant during specified periods of time, but change from one period to the
#' next period. The function then predicts decomposition from one time step to the
#' next. This function may be used to simulate decomposition of *Sphagnum* litter
#' when environmental conditions change.
#'
#' For parameters `phi_2_p2` (measurement errors for remaining masses) and
#' `alpha_2` (factor that controls how fast decomposition rates decrease with
#' increasing mass loss), the functions assumes by default average values
#' estimated for species across the studies analyzed in
#' \insertCite{Teickner.2024a;textual}{hpmdpredict}. If other measurement errors
#' for remaining masses or values for `alpha_2` should to be assumed, this has
#' to be specified via `newdata`.
#'
#' @param newdata The same as for `hpmd_predict_fit_4` with an additional column
#' `id_time_step`, an integer specifying the step number. Column
#' `incubation_duration` now specifies the duration of the time step and the total
#' incubation duration up to the current step is the sum of the `incubation_duration`
#' values of all previous steps and the current step. `m0` is only considered for
#' the first time step and dynamically computed for all subsequent time steps.
#'
#' @return `newdata` with additional columns with predicted initial leaching
#' losses, decomposition rates, and remaining masses.
#'
#' @examples
#' # Comparison between changing and constant environmental conditions. Here, in
#' # the first example both layer_degree_of_saturation_1 and
#' # sample_depth_lower change over time.
#' set.seed(8484)
#' newdata <-
#'   tibble::tibble(
#'     id_time_step = 1:6,
#'     incubation_duration = c(5, 5, 5, 5, 5, 100),
#'     m0 = 1,
#'     l0 = 0,
#'     layer_degree_of_saturation_1 = c(0.2, 0.6, 0.8, 1, 1, 1),
#'     layer_water_table_depth_to_surface_1 = 30,
#'     sample_depth_lower = c(10, 20, 30, 35, 40, 50),
#'     hpm_taxon_rank_value = "Sphagnum angustifolium"
#'  )
#'
#' res <- hpmd_predict_fit_4_variable(newdata)
#'
#' newdata <-
#'   tibble::tibble(
#'     incubation_duration = cumsum(c(5, 5, 5, 5, 5, 100)),
#'     m0 = 1,
#'     l0 = 0,
#'     layer_degree_of_saturation_1 = c(0.4),
#'     layer_water_table_depth_to_surface_1 = 30,
#'     sample_depth_lower = c(10),
#'     hpm_taxon_rank_value = "Sphagnum angustifolium"
#'  )
#'
#' res1 <- hpmd_predict_fit_4(newdata)
#'
#' plot(mean(res$mass_relative_mass) ~ cumsum(res$incubation_duration))
#' points(mean(res1$mass_relative_mass) ~ res1$incubation_duration, col = "red")
#'
#' set.seed(8484)
#' newdata <-
#'   tibble::tibble(
#'     id_time_step = 1:6,
#'     incubation_duration = cumsum(c(5, 5, 5, 5, 5, 100)),
#'     m0 = 1,
#'     l0 = 0,
#'     layer_degree_of_saturation_1 = c(0.4),
#'     layer_water_table_depth_to_surface_1 = 30,
#'     sample_depth_lower = c(10),
#'     hpm_taxon_rank_value = "Sphagnum angustifolium"
#'  )
#'
#' res1 <- hpmd_predict_fit_4(newdata)
#'
#' newdata$incubation_duration <- c(5, 5, 5, 5, 5, 100)
#'
#' set.seed(8484)
#' res <- hpmd_predict_fit_4_variable(newdata)
#'
#' plot(mean(res$mass_relative_mass) ~ cumsum(res$incubation_duration))
#' points(mean(res1$mass_relative_mass) ~ res1$incubation_duration, col = "red")
#'
#' @export
hpmd_predict_fit_4_variable <- function(newdata) {

  stopifnot("id_time_step" %in% colnames(newdata))
  stopifnot(length(unique(newdata$id_time_step)) == nrow(newdata))

  m_config <- hpmd_data_stan_4

  newdata <- newdata[order(newdata$id_time_step), ]

  # make predictions as if conditions were constant, just to get the initial k0 under the respective conditions
  res_constant <- hpmd_predict_fit_4(newdata)

  # make predictions
  res <- tibble::tibble()

  for(i in seq_len(nrow(newdata))) {

    cur_res <- res_constant[i, ]

    if(i > 1L) {

      # update the decomposition rate (to consider that it is lower because some mass has already been lost)
      cur_res$k0 = cur_res$k0 * res$mass_relative_mass_mu[[i - 1]]^(cur_res$alpha_2 - 1.0) / res$m0[[1]]^(cur_res$alpha_2 - 1.0)

      # make correct predictions for the remaining mass
      cur_res <- hpmd_predict_fit_4_helper_1(newdata = cur_res |> dplyr::mutate(m0 = res$mass_relative_mass_mu[[i - 1]]), m_config = m_config)

    }

    # combine
    res <- dplyr::bind_rows(res, cur_res)

  }

  res

}





