#!/usr/bin/python3

"""
Run a benchmark with the ROSA backdoor detector.

A benchmark consists of one or more runs of the ROSA backdoor detector given a target
program-under-test.
"""

from __future__ import annotations

import argparse
import json
import os
import shutil
import signal
import subprocess
import sys

CONFIGS_DIR = os.path.join("/root", "artifact", "configs")
TARGET_FILE_PATH = os.path.join("/root", "artifact", "targets.json")
JANITOR_PATH = os.path.join("/root", "artifact", "tools", "janitor.py")


def run(
    target_name: str, minutes_per_run: int, runs: int, output_dir: str, verbose: bool
) -> None:
    """Run a benchmark."""
    assert minutes_per_run > 0
    assert runs > 0

    rosa_dir = os.path.join(output_dir, f"rosa-out-{target_name}")
    fuzzer_dir = os.path.join(output_dir, f"fuzzer-out-{target_name}")
    rosa_cmd = [
        "rosa",
        os.path.join(CONFIGS_DIR, f"{target_name}.toml"),
    ]
    if verbose:
        rosa_cmd = [*rosa_cmd, "--verbose"]

    for run in range(runs):
        print(
            f"  Performing backdoor detection run {run + 1}/{runs}...", file=sys.stderr
        )
        rosa_process = subprocess.Popen(
            rosa_cmd,
            cwd=output_dir,
        )
        try:
            _ = rosa_process.communicate(timeout=minutes_per_run * 60)
        except subprocess.TimeoutExpired:
            rosa_process.send_signal(signal.SIGINT)
            rosa_process.wait()

        # Move the output directories to a separate directory to prepare for the
        # next run.
        run_dir = os.path.join(output_dir, f"run-{run:02d}")
        os.makedirs(run_dir)
        shutil.move(fuzzer_dir, run_dir)
        shutil.move(rosa_dir, run_dir)


def main() -> None:
    """Parse arguments and run a benchmark."""
    targets = {}
    with open(TARGET_FILE_PATH, "r") as targets_file:
        targets = json.load(targets_file)

    parser = argparse.ArgumentParser(
        description="Evaluate backdoor benchmarks with the ROSA backdoor detector."
    )
    parser.add_argument(
        "target", help="The target to run.", choices=list(targets.keys())
    )
    parser.add_argument(
        "minutes_per_run",
        help="The amount of minutes to allocate to each run.",
        type=int,
    )
    parser.add_argument("runs", help="The number of runs to perform.", type=int)
    parser.add_argument(
        "output_dir", help="The output directory for the fuzzer & ROSA."
    )
    parser.add_argument(
        "-J",
        "--no-janitor",
        dest="use_janitor",
        help="Disable the janitor process, which cleans up files automatically.",
        action="store_false",
    )
    parser.add_argument(
        "-v",
        "--verbose",
        help="Display more detailed output.",
        action="store_true",
    )

    args = parser.parse_args()
    assert args is not None

    if args.use_janitor:
        # Run the janitor in parallel to avoid filling up the disk space with garbage.
        janitor_process = subprocess.Popen(
            [JANITOR_PATH],
            cwd=args.output_dir,
            stdout=None if args.verbose else subprocess.DEVNULL,
            stderr=None if args.verbose else subprocess.STDOUT,
        )

    # Run target setup code.
    target_root_dir = targets[args.target]["root_dir"]
    assert type(target_root_dir) is str
    subprocess.call(
        ["make", "-C", target_root_dir, "setup"],
        stdout=None if args.verbose else subprocess.DEVNULL,
        stderr=None if args.verbose else subprocess.STDOUT,
    )

    try:
        # Run the target.
        run(
            target_name=args.target,
            minutes_per_run=args.minutes_per_run,
            runs=args.runs,
            output_dir=args.output_dir,
            verbose=args.verbose,
        )
    except KeyboardInterrupt:
        pass

    # Run target teardown code.
    subprocess.call(
        ["make", "-C", target_root_dir, "teardown"],
        stdout=None if args.verbose else subprocess.DEVNULL,
        stderr=None if args.verbose else subprocess.STDOUT,
    )

    if args.use_janitor:
        # Stop the janitor.
        janitor_process.send_signal(signal.SIGINT)
        janitor_process.wait()


if __name__ == "__main__":
    main()
