#!/usr/bin/python3

"""
Clean up unwanted files.

Sometimes, the fuzzer will cause the target program to generate new files on disk. This
becomes a problem as these files accumulate, taking up precious space and sometimes
leading fuzzing campaigns to crash. This program's goal is to clean up any unwanted
files to prevent that from happening.
"""

import argparse
import os
import shutil
import threading
from enum import Enum, unique
from typing import Optional


@unique
class NameKind(Enum):
    """Describe the kind of the filesystem element name."""

    """The name must match the element exactly."""
    ExactMatch = 0
    """The element starts with this name."""
    StartsWith = 1


KNOWN_GOOD_FILES_AND_DIRECTORIES = [
    # Scripts.
    ("clean.sh", NameKind.ExactMatch),
    ("run_benchmark.py", NameKind.ExactMatch),
    # Directories.
    ("configs", NameKind.ExactMatch),
    ("dictionaries", NameKind.ExactMatch),
    ("seeds", NameKind.ExactMatch),
    ("tools", NameKind.ExactMatch),
    # Fuzzer directories.
    ("fuzzer-out-", NameKind.StartsWith),
    # Rosa directories.
    ("rosa-out-", NameKind.StartsWith),
    # Run result directories (containing raw experiment data).
    ("run-", NameKind.StartsWith),
    # Benchmark result directories.
    ("benchmark_", NameKind.StartsWith),
    # Some targets need this stuff in /tmp/.
    ("nvram", NameKind.ExactMatch),
    ("nvram_config", NameKind.ExactMatch),
    ("www", NameKind.ExactMatch),
    # Some tmux stuff that may exist in /tmp/.
    ("tmux", NameKind.StartsWith),
]


def should_be_deleted(
    element: str, ignore: Optional[list[tuple[str, NameKind]]] = None
) -> bool:
    """Check if a filesystem element should be deleted."""
    ignore = ignore or []
    element_name = os.path.basename(element)

    for ignored_name, ignore_kind in ignore:
        if (ignore_kind == NameKind.ExactMatch and element_name == ignored_name) or (
            ignore_kind == NameKind.StartsWith and element_name.startswith(ignored_name)
        ):
            return False

    return True


def delete(element: str) -> None:
    """Delete a filesystem element."""
    try:
        if os.path.isfile(element):
            os.unlink(element)
        else:
            shutil.rmtree(element)
    except OSError:
        pass


def cleanup(
    dirs: Optional[list[str]] = None,
    ignore: Optional[list[tuple[str, NameKind]]] = None,
) -> None:
    """Clean up any unknown files in a list of directories."""
    dirs = dirs or []
    ignore = ignore or []

    for directory in dirs:
        for element in os.listdir(directory):
            full_element_path = os.path.join(directory, element)
            if should_be_deleted(element=full_element_path, ignore=ignore):
                delete(full_element_path)


def cleanup_loop(stop_event: threading.Event) -> None:
    """Perform a cleanup in a loop (until a stop event happens)."""
    while not stop_event.is_set():
        cleanup(dirs=[os.getcwd(), "/tmp"], ignore=KNOWN_GOOD_FILES_AND_DIRECTORIES)


def main() -> None:
    """Parse arguments and run a cleanup."""
    parser = argparse.ArgumentParser(
        description="Periodically clean up unwanted files."
    )

    args = parser.parse_args()
    assert args is not None

    stop_event = threading.Event()
    stop_event.clear()
    cleanup_thread = threading.Thread(target=cleanup_loop, args=(stop_event,))

    try:
        print("Starting janitor.")
        cleanup_thread.start()
        while True:
            pass
    except KeyboardInterrupt:
        print("Stopping janitor.")
        stop_event.set()
        cleanup_thread.join()


if __name__ == "__main__":
    main()
