#!/usr/bin/env bash

## Run a backdoor detection campaign and evaluate it for a single target.
##
## The campaign will run for a number of minutes, and will be repeated for an amount of runs. These
## parameters are given by the user. At the end, a limited version of Table II and Figure 2 from
## the paper will be reproduced.


set -e


if [ "$#" -ne "3" ]
then
    echo "Usage: $0 TARGET MINUTES_PER_RUN RUNS" 1>&2
    exit 1
fi

target=$1
minutes_per_run=$2
runs=$3
phase_1_durations=(30 300 600 900 1200)
verbose=""
if [ "${RUN_TARGET_VERBOSE:-0}" -eq "1" ]
then
    verbose="--verbose"
fi

date=$(printf '%(%Y%m%d)T' -1)
output_dir=/root/evaluation/${target}__${minutes_per_run}mpr_${runs}r_${date}

# First, run the actual backdoor detection campaign, for a phase 1 duration of 60 seconds.
base_campaign_dir=$output_dir/phase_1_60s
# Here, we use a single instance.
base_instance_dir=$base_campaign_dir/0

mkdir -p $base_instance_dir/
echo "Running base backdoor detection campaign for '$target'..."
# Run the backdoor detection campaign.
/root/artifact/tools/run-benchmark.py $verbose $target $minutes_per_run $runs $base_instance_dir
# Evaluate the backdoor detection campaign.
/root/artifact/tools/evaluate-benchmark.py $verbose $target $base_campaign_dir \
    $base_campaign_dir/results

echo "Running parameter sweep study for '$target'..."
# Now, run simulations for all other phase 1 durations.
# We do not need to run fresh detection campaigns, as we can simulate what would have happened with
# the same fuzzer data.
for duration_index in "${!phase_1_durations[@]}"
do
    phase_1_duration="${phase_1_durations[$duration_index]}"
    campaign_dir=$output_dir/phase_1_${phase_1_duration}s
    echo "  Running study for phase 1 duration of $phase_1_duration seconds" \
        "($((duration_index+1))/${#phase_1_durations[@]})..."
    mkdir -p $campaign_dir/
    # Copy over the "base" campaign of 60 seconds.
    cp -r $base_instance_dir/ $campaign_dir/
    # We now need to iterate over the runs, assuming a single instance.
    for run_dir in $(ls $campaign_dir/0)
    do
        pushd . >/dev/null 2>&1
            cd $campaign_dir/0/$run_dir
            # Rename the ROSA output directory to not lose any data.
            mv rosa-out-$target rosa-out-$target.original
            # Replace the number of seconds in the phase 1 duration.
            cp rosa-out-$target.original/config.toml .
            sed -i "s/seconds = [0-9]\+/seconds = $phase_1_duration/" ./config.toml
            # Run the simulation, copying the original fuzzer inputs.
            if [ "${RUN_TARGET_VERBOSE:-0}" -eq "1" ]
            then
                rosa-simulate --copy-inputs rosa-out-$target.original ./config.toml
            else
                rosa-simulate --copy-inputs rosa-out-$target.original ./config.toml 2>/dev/null
            fi
            # Evaluate the results of the simulation.
            /root/artifact/tools/evaluate-benchmark.py $verbose $target $campaign_dir \
                $campaign_dir/results
            # Remove the original ROSA output directory and the temporary configuration file since
            # they are no longer needed.
            rm -rf rosa-out-$target.original/ config.toml
        popd >/dev/null 2>&1
    done
done

if [ "${RUN_TARGET_NO_PDF:-0}" -ne "1" ]
then
    # Generate the PDF with the slice of Table II and partial Figure 2.
    echo "Generating PDF..."
    /root/artifact/tools/results-to-pdf.py $output_dir --output /root/evaluation/$target.pdf
fi
