#!/usr/bin/env bash

## Run the reduced evaluation from the "ROSA: Finding Backdoors with Fuzzing" paper.
##
## This script offers a reduced evaluation of the main results from the paper (i.e., Table II and
## Figure 2), in order to allow artifact reviewers to complete the evaluation in a reasonable
## amount of time.
##
## Concretely, it performs the following:
## 1. A "from-scratch" evaluation of ROSA on Sudo. A single run of 30 minutes is performed (as
##    opposed to 10 runs of 8 hours in the paper), with a phase 1 duration of 1 minute.
## 2. A "simulated" evaluation of ROSA on D-Link/thttpd. In order to avoid the time-consuming
##    fuzzing step, base fuzzer inputs are given (with their timings), and `rosa-simulate` is used
##    to evaluate ROSA without rerunning the fuzzing step. In this case, the full 10 run, 8 hours
##    per run settings from the paper apply.


set -e


phase_1_durations=(30 60 300 600 900 1200)
time_limit_seconds=28800  # 8 hours

# Run Sudo once, for 30 minutes.
RUN_TARGET_NO_PDF=1 /root/artifact/run-target.sh sudo-backdoored 30 1

# Run simulations for D-Link.
output_dir=/root/evaluation/dlink-backdoored__480mpr_10r_20241031
mkdir -p output_dir/
mkdir -p $output_dir/
echo "Running backdoor detection campaign for D-Link/thttpd based on existing data..."
for duration_index in "${!phase_1_durations[@]}"
do
    phase_1_duration="${phase_1_durations[$duration_index]}"
    campaign_dir=$output_dir/phase_1_${phase_1_duration}s
    run_index=1
    total_runs=10
    echo "  Running for phase 1 duration of $phase_1_duration seconds" \
        "($((duration_index+1))/${#phase_1_durations[@]})..."
    # Copy base data.
    cp -r /root/artifact/data/dlink-backdoored__480mpr_10r_20241031/ $campaign_dir/

    for instance_dir in $(ls $campaign_dir)
    do
        for run_dir in $(ls $campaign_dir/$instance_dir)
        do
            echo "    Executing run $run_index/$total_runs..."
            run_index=$((run_index+1))
            pushd . >/dev/null 2>&1
                cd $campaign_dir/$instance_dir/$run_dir
                # Rename the ROSA output directory to not lose any data.
                mv rosa-out-dlink-backdoored rosa-out-dlink-backdoored.original
                # Replace the number of seconds in the phase 1 duration.
                cp rosa-out-dlink-backdoored.original/config.toml .
                sed -i "s/seconds = [0-9]\+/seconds = $phase_1_duration/" ./config.toml
                # Run the simulation, copying the original fuzzer inputs.
                rosa-simulate --copy-inputs rosa-out-dlink-backdoored.original ./config.toml \
                    2>/dev/null
                # Remove the original ROSA output directory and the temporary configuration file
                # since they are no longer needed.
                rm -rf rosa-out-dlink-backdoored.original/ config.toml
            popd >/dev/null 2>&1
        done
    done

    echo "    Evaluating the set of $total_runs runs..."
    # Evaluate the results of the simulation.
    /root/artifact/tools/evaluate-benchmark.py --time-limit $time_limit_seconds \
        dlink-backdoored $campaign_dir $campaign_dir/results
    # Remove the fuzzer directories, since they are no longer needed.
    rm -rf */**/fuzzer-out-dlink-backdoored/
done


# Generate the PDF with the partial Table II and Figure 2.
echo "Generating PDF..."
/root/artifact/tools/results-to-pdf.py /root/evaluation/sudo-backdoored__30mpr_1r_*/ \
    /root/evaluation/dlink-backdoored__480mpr_10r_20241031/ \
    --output /root/evaluation/reduced-evaluation.pdf
