#' Plot heatmap of residuals
#'
#' Plot heatmap of residuals (observed - predicted counts).
#'
#' @param pred.df A data frame containing the predicted counts, as generated by
#' [predict_counts()]
#' @param data Data list (from `prepare_data()`)
#' @param ... Further arguments for [network.tools::plot_web_heatmap()].
#'
#' @return A ggplot object
#' @export
#'
#' @examplesIf interactive()
#' data(web)
#' dt <- prepare_data(mat = web, sampl.eff = rep(20, nrow(web)))
#' fit <- fit_model(dt, refresh = 0)
#' pred.df <- predict_counts(fit, dt)
#' plot_residuals(pred.df, dt)
#' plot_residuals(pred.df, dt, sort = FALSE)

plot_residuals <- function(pred.df = NULL, data = NULL, ...) {

  obs <- data$M |>
    network.tools::wide2long()

  pred <- pred.df |>
    dplyr::select(Plant, Animal, count) |>
    tidybayes::mean_qi() |>
    dplyr::select(Plant, Animal, count) |>
    dplyr::left_join(obs, by = c("Plant", "Animal")) |>
    dplyr::mutate(resid = Visits - count)

  max.resid <- max(abs(pred$resid))

  suppressMessages(
    pred |>
      network.tools::plot_web_heatmap(int.var = "resid", ...) +
      ggplot2::scale_fill_distiller(palette = "RdBu", limits = c(-max.resid, max.resid), direction = 1) +
      ggplot2::labs(title = "Residuals", fill = "")
  )

}
