/*******************************************************************************************
Authors: Carlos Segura (Programmer and Designer), Lazaro Lugo (Programmer and Designer),
Gara Miranda (Designer)

Description: 
*Implementation of a stochatic hill-climber with the move operation. The whole
neighborhood is traversed in O(n^2). (local_search())
*Implementation of iterated local search ils using four type of perturbations operator and stochatic hill-climber
local serach. (intensify())
*Perturbation operators that can be used:
-mutationScramble()
-mutationSwap()
-mutationInvertion()
-mutationInsert()
********************************************************************************************/

#include "Individual.h"
#include "utils.h"
#include <algorithm>
#include <sys/time.h>
#include <iostream>
#include <fstream>
#include <string>

void Individual::local_search(){
	long int i, j, t, k, v, best_j;
	long int maxgain;
	best_j = 0;
	maxgain = 0;
	
	//Sort the indexes randomly to make it stochastic
	vector < int > vs;
	vs.reserve(problem->problemDimension);
	for (int i = 0; i < problem->problemDimension; i++) {
		vs.push_back(i);
	}
	random_shuffle(vs.begin(), vs.end());
    bool changed = true;

	while (changed) {
		changed = false;
		for (int indexv = 0; indexv < problem->problemDimension; indexv++) {
			//Select and value to move, and look for its best new position
			v = vs[indexv];
			long gain = 0;
			maxgain = 0;
			i = v;
			for (j = i - 1; j >= 0; j--) {//Move to the left
				gain += ((problem->instance[S[i]][S[j]]) - (problem->instance[S[j]][S[i]]));
				if (gain > maxgain) {
					maxgain = gain;
					best_j = j;
				}
			}

			gain = 0;
			for (j = i + 1; j < problem->problemDimension; j++) {//Move to the right
				gain += ((problem->instance[S[j]][S[i]]) - (problem->instance[S[i]][S[j]]));
				if (gain > maxgain) {
					maxgain = gain;
					best_j = j;
				}
			}

			if (maxgain > 0) {//If there is a gain, move it
				j = best_j;
				t = S[i];
				if (i < j)
					for (k = i + 1; k <= j; k++) {
						S[k - 1] = S[k];
					}
				else
					for (k = i - 1; k >= j; k--) {
						S[k + 1] = S[k];
					}
				S[j] = t;
				changed = true;
				cost += maxgain;
			}
		}
	}
}

int Individual::next(int node){
	return (node + 1) % problem->problemDimension;
}

int Individual::prev(int node){
	return (node - 1 + problem->problemDimension) % problem->problemDimension;
}

void Individual::mutationScramble(){
    int max,min,tmp;
    int r1 = getRandomInteger0_N(problem->problemDimension - 1);
	int r2 = getRandomInteger0_N(problem->problemDimension - 1);
    min = (r1 > r2)?r2:r1;
	max = (r1 > r2)?r1:r2;

	if (max == next(min)){//R1 y R2 son consecutivos
		tmp=S[max];
		S[max]=S[min];
        S[min]=tmp;
	} else {
		//Destroy current edges
		int length = 1;
		for (int i = min; i != max; i = next(i)){
			length++;
		}
		//Scramble
		for (int i = 0; i < length - 1; i++){
			int r = getRandomInteger0_N(length - i - 1);
			int tmp = S[(min + length - i - 1) % problem->problemDimension];
			S[(min + length - i - 1) % problem->problemDimension]=S[(min + r) % problem->problemDimension];
			S[(min + r) % problem->problemDimension]=tmp;
		}
	  }
	evaluate();
}
void Individual::mutationSwap(){
    int w, wp, v, j, n1, n2, min, max, g;
    
	//Valores por defecto 
    int MutationsRepeat = 1;
	int MaxMutLength = problem->problemDimension-1;
	
    for (v = 0; v < MutationsRepeat; v++){
		n1 = getRandomInteger0_N(problem->problemDimension);
		w = ((problem->problemDimension-n1) < MaxMutLength)?(problem->problemDimension-n1):MaxMutLength;
		wp = (n1 < MaxMutLength)?n1:MaxMutLength;
		
		do {
			n2 = n1 - wp + (int)getRandomInteger0_N(w+wp);
			if ( n2 >= problem->problemDimension ){
				n2 = problem->problemDimension - 1;	
			}
		} while ( n1 == n2 );
		min = (n1 > n2)?n2:n1;
		max = (n1 > n2)?n1:n2;
		
		max = (max >= problem->problemDimension)?max-1:max;

		for ( g = 0, j = min + 1; j <= max; j++){
			g += problem->instance[S[j]][S[min]] - problem->instance[S[min]][S[j]];
		}
		for ( j = min + 1; j < max; j++){
			g += problem->instance[S[max]][S[j]] - problem->instance[S[j]][S[max]];
		}
		int auxSwap = S[min];
		S[min]=S[max];
		S[max]=auxSwap;
		cost = cost + g;
    }
}
void Individual::mutationInvertion(){
	int r1 = getRandomInteger0_N(problem->problemDimension - 1);
	int r2 = getRandomInteger0_N(problem->problemDimension - 1);
	if (r1 != r2){	
		//Reverse
		int left = r1;
		int right = r2;
		int times = (right > left)?((right - left + 1) / 2):(((right+1) + (problem->problemDimension - left)) / 2);
		for (int i = 0; i < times; i++){
			int tmp = S[left];
			S[left]=S[right];
			S[right]=tmp;
			left = next(left);
			right = prev(right);
		}
	}
	evaluate();
}
void Individual::mutationInsert(){
    int r1 = getRandomInteger0_N(problem->problemDimension - 1);
	int r2 = getRandomInteger0_N(problem->problemDimension - 1);
	if ((r1 != r2) && (r1 != next(r2))){
		int movedNode = S[r1];
		int first = prev(r1);
		int notMove = r2;
		for (int i = first; i != notMove; i = prev(i)){
			S[next(i)]=S[i];
		}
		int newPosition = next(r2);
		S[newPosition]=movedNode;
	}
	evaluate();
}

void Individual::intensify(string &perturbationType, double &ils_time) {
	//Time structures
	struct timeval start_bl,end_bl;
	double stop=0.0;
	gettimeofday(&start_bl,NULL);
	/*P_1 Initial intensification process*/
	local_search();
	/*Save solution after apply local search*/
	Individual newInd=internalClone();
	/*Loop principal*/
	int count=0;
	while(stop<ils_time){
	/*P_2 Perturb the newSolution*/
	if(perturbationType=="SM"){
		newInd.mutationScramble();
	}else if(perturbationType=="SWM"){
		newInd.mutationSwap();
		newInd.mutationSwap();
		newInd.mutationSwap();
	}else if(perturbationType=="INM"){
		newInd.mutationInvertion();
	}else if(perturbationType=="ISM"){
		newInd.mutationInsert();
	}else{
		cout<<"The provided perturbation operator is not recognized"<< endl;
		exit(1);
	}
	/*P_3 Local Search after perturb*/
	newInd.local_search();

	/*P_4 Check cost in the pertutbate solution*/
	if(newInd.cost>=cost){
		S=newInd.S;
		cost=newInd.cost;
	}else{
		newInd=internalClone();
	}
	count++;
	//Check ils_time
	gettimeofday(&end_bl, NULL);
	stop=(end_bl.tv_sec - start_bl.tv_sec)+(end_bl.tv_usec - start_bl.tv_usec)/1.0e6;
	}
}
