/*-
 * #%L
 * BroadleafCommerce Framework
 * %%
 * Copyright (C) 2009 - 2024 Broadleaf Commerce
 * %%
 * Licensed under the Broadleaf Fair Use License Agreement, Version 1.0
 * (the "Fair Use License" located  at http://license.broadleafcommerce.org/fair_use_license-1.0.txt)
 * unless the restrictions on use therein are violated and require payment to Broadleaf in which case
 * the Broadleaf End User License Agreement (EULA), Version 1.1
 * (the "Commercial License" located at http://license.broadleafcommerce.org/commercial_license-1.1.txt)
 * shall apply.
 *
 * Alternatively, the Commercial License may be replaced with a mutually agreed upon license (the "Custom License")
 * between you and Broadleaf Commerce. You may not use this file except in compliance with the applicable license.
 * #L%
 */
package org.broadleafcommerce.core.workflow;

import org.broadleafcommerce.core.order.domain.Order;
import org.broadleafcommerce.core.workflow.state.RollbackHandler;
import org.springframework.beans.factory.BeanNameAware;
import org.springframework.core.Ordered;

import java.util.Map;

/**
 * <p>
 * Interface to be used for workflows in Broadleaf. Usually implementations will subclass {@link BaseActivity}.
 * </p>
 * <p>
 * Important note: if you are writing a 3rd-party integration module or adding a module outside of the Broadleaf core, your
 * activity should implement the {@link ModuleActivity} interface as well. This ensures that there is proper logging
 * for users that are using your module so that they know exactly what their final workflow configuration looks like.
 *
 * @param <T>
 * @author Phillip Verheyden (phillipuniverse)
 * @see {@link BaseActivity}
 * @see {@link ModuleActivity}
 * @see {@link BaseProcessor}
 * @see {@link SequenceProcessor}
 */
public interface Activity<T extends ProcessContext<?>> extends BeanNameAware, Ordered {

    /**
     * Called by the encompassing processor to activate
     * the execution of the Activity
     *
     * @param context - process context for this workflow
     * @return resulting process context
     * @throws Exception
     */
    T execute(T context) throws Exception;

    /**
     * Determines if an activity should execute based on the current values in the {@link ProcessContext}. For example, a
     * context might have both an {@link Order} as well as a String 'status' of what the order should be changed to. It is
     * possible that an activity in a workflow could only deal with a particular status change, and thus could return false
     * from this method.
     *
     * @param context
     * @return
     */
    boolean shouldExecute(T context);

    /**
     * Get the fine-grained error handler wired up for this Activity
     *
     * @return
     */
    ErrorHandler getErrorHandler();

    void setErrorHandler(final ErrorHandler errorHandler);

    String getBeanName();

    /**
     * Retrieve the RollbackHandler instance that should be called by the ActivityStateManager in the
     * event of a workflow execution problem. This RollbackHandler will presumably perform some
     * compensating operation to revert state for the activity.
     *
     * @return the handler responsible for reverting state for the activity
     */
    RollbackHandler<T> getRollbackHandler();

    /**
     * Set the RollbackHandler instance that should be called by the ActivityStateManager in the
     * event of a workflow execution problem. This RollbackHandler will presumably perform some
     * compensating operation to revert state for the activity.
     *
     * @param rollbackHandler the handler responsible for reverting state for the activity
     */
    void setRollbackHandler(RollbackHandler<T> rollbackHandler);

    /**
     * Retrieve the optional region label for the RollbackHandler. Setting a region allows
     * partitioning of groups of RollbackHandlers for more fine grained control of rollback behavior.
     * Explicit calls to the ActivityStateManager API in an ErrorHandler instance allows explicit rollback
     * of specific rollback handler regions. Note, to disable automatic rollback behavior and enable explicit
     * rollbacks via the API, the workflow.auto.rollback.on.error property should be set to false in your implementation's
     * runtime property configuration.
     *
     * @return the rollback region label for the RollbackHandler instance
     */
    String getRollbackRegion();

    /**
     * Set the optional region label for the RollbackHandler. Setting a region allows
     * partitioning of groups of RollbackHandlers for more fine grained control of rollback behavior.
     * Explicit calls to the ActivityStateManager API in an ErrorHandler instance allows explicit rollback
     * of specific rollback handler regions. Note, to disable automatic rollback behavior and enable explicit
     * rollbacks via the API, the workflow.auto.rollback.on.error property should be set to false in your implementation's
     * runtime property configuration.
     *
     * @param rollbackRegion the rollback region label for the RollbackHandler instance
     */
    void setRollbackRegion(String rollbackRegion);

    /**
     * Retrieve any user-defined state that should accompany the RollbackHandler. This configuration will be passed to
     * the RollbackHandler implementation at runtime.
     *
     * @return any user-defined state configuratio necessary for the execution of the RollbackHandler
     */
    Map<String, Object> getStateConfiguration();

    /**
     * Set any user-defined state that should accompany the RollbackHandler. This configuration will be passed to
     * the RollbackHandler implementation at runtime.
     *
     * @param stateConfiguration any user-defined state configuration necessary for the execution of the RollbackHandler
     */
    void setStateConfiguration(Map<String, Object> stateConfiguration);

    /**
     * Whether or not this activity should automatically register a configured RollbackHandler with the ActivityStateManager.
     * It is useful to adjust this value if you plan on using the ActivityStateManager API to register RollbackHandlers
     * explicitly in your code. The default value is false.
     *
     * @return Whether or not to automatically register a RollbackHandler with the ActivityStateManager
     */
    boolean getAutomaticallyRegisterRollbackHandler();

    /**
     * Whether or not this activity should automatically register a configured RollbackHandler with the ActivityStateManager.
     * It is useful to adjust this value if you plan on using the ActivityStateManager API to register RollbackHandlers
     * explicitly in your code. The default value is false.
     *
     * @param automaticallyRegisterRollbackHandler Whether or not to automatically register a RollbackHandler with the ActivityStateManager
     */
    void setAutomaticallyRegisterRollbackHandler(boolean automaticallyRegisterRollbackHandler);

}
