/*-
 * #%L
 * BroadleafCommerce Framework
 * %%
 * Copyright (C) 2009 - 2024 Broadleaf Commerce
 * %%
 * Licensed under the Broadleaf Fair Use License Agreement, Version 1.0
 * (the "Fair Use License" located  at http://license.broadleafcommerce.org/fair_use_license-1.0.txt)
 * unless the restrictions on use therein are violated and require payment to Broadleaf in which case
 * the Broadleaf End User License Agreement (EULA), Version 1.1
 * (the "Commercial License" located at http://license.broadleafcommerce.org/commercial_license-1.1.txt)
 * shall apply.
 *
 * Alternatively, the Commercial License may be replaced with a mutually agreed upon license (the "Custom License")
 * between you and Broadleaf Commerce. You may not use this file except in compliance with the applicable license.
 * #L%
 */
package org.broadleafcommerce.core.search.redirect.domain;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.broadleafcommerce.common.admin.domain.AdminMainEntity;
import org.broadleafcommerce.common.extensibility.jpa.copy.DirectCopyTransform;
import org.broadleafcommerce.common.extensibility.jpa.copy.DirectCopyTransformMember;
import org.broadleafcommerce.common.extensibility.jpa.copy.DirectCopyTransformTypes;
import org.broadleafcommerce.common.persistence.IdOverrideTableGenerator;
import org.broadleafcommerce.common.presentation.AdminPresentation;
import org.broadleafcommerce.common.presentation.ConfigurationItem;
import org.broadleafcommerce.common.presentation.RequiredOverride;
import org.broadleafcommerce.common.presentation.ValidationConfiguration;
import org.broadleafcommerce.common.presentation.client.VisibilityEnum;
import org.broadleafcommerce.common.time.SystemTime;
import org.broadleafcommerce.common.util.BLCSystemProperty;
import org.hibernate.annotations.Cache;
import org.hibernate.annotations.CacheConcurrencyStrategy;
import org.hibernate.annotations.GenericGenerator;
import org.hibernate.annotations.Parameter;

import java.io.Serial;
import java.util.Date;

import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.GeneratedValue;
import jakarta.persistence.Id;
import jakarta.persistence.Index;
import jakarta.persistence.Inheritance;
import jakarta.persistence.InheritanceType;
import jakarta.persistence.Table;
import jakarta.persistence.Transient;

/**
 * @author priyeshpatel
 */
@Entity
@Inheritance(strategy = InheritanceType.JOINED)
@Table(name = "BLC_SEARCH_INTERCEPT", indexes = {
        @Index(name = "SEARCH_ACTIVE_INDEX", columnList = "ACTIVE_START_DATE, ACTIVE_END_DATE")
})
@Cache(usage = CacheConcurrencyStrategy.READ_WRITE, region = "blSearchElements")
@DirectCopyTransform({
        @DirectCopyTransformMember(templateTokens = DirectCopyTransformTypes.MULTITENANT_SITE),
        @DirectCopyTransformMember(templateTokens = DirectCopyTransformTypes.AUDITABLE_ONLY)
})
public class SearchRedirectImpl implements SearchRedirect, java.io.Serializable, AdminMainEntity,
        SearchRedirectAdminPresentation {

    @Serial
    private static final long serialVersionUID = 1L;

    @Transient
    private static final Log LOG = LogFactory.getLog(SearchRedirectImpl.class);

    @Id
    @GeneratedValue(generator = "SearchRedirectID")
    @GenericGenerator(
            name = "SearchRedirectID",
            type = IdOverrideTableGenerator.class,
            parameters = {
                    @Parameter(name = "segment_value", value = "SearchRedirectImpl"),
                    @Parameter(name = "entity_name",
                            value = "org.broadleafcommerce.core.search.redirect.domain.SearchRedirectImpl")
            }
    )
    @Column(name = "SEARCH_REDIRECT_ID")
    @AdminPresentation(visibility = VisibilityEnum.HIDDEN_ALL)
    protected Long id;

    @Column(name = "PRIORITY")
    @AdminPresentation(excluded = true)
    protected Integer searchPriority;

    @AdminPresentation(friendlyName = "SearchRedirectImpl_searchTerm", order = 1000,
            group = GroupName.General, prominent = true, groupOrder = 1, defaultValue = "")
    @Column(name = "SEARCH_TERM", nullable = false)
    protected String searchTerm;

    @Column(name = "URL", nullable = false)
    @AdminPresentation(friendlyName = "SearchRedirectImpl_url", order = 2000,
            group = GroupName.General, prominent = true, groupOrder = 1, defaultValue = "")
    protected String url;

    /**
     * The active start date.
     */
    @Column(name = "ACTIVE_START_DATE")
    @AdminPresentation(friendlyName = "SkuImpl_Sku_Start_Date",
            order = 3000, group = GroupName.Dates,
            tooltip = "skuStartDateTooltip", groupOrder = 1,
            requiredOverride = RequiredOverride.REQUIRED,
            defaultValue = "today")
    protected Date activeStartDate;

    /**
     * The active end date.
     */
    @Column(name = "ACTIVE_END_DATE")
    @AdminPresentation(friendlyName = "SkuImpl_Sku_End_Date", order = 4000, group = GroupName.Dates,
            tooltip = "skuEndDateTooltip",
            groupOrder = 1,
            validationConfigurations = {
                    @ValidationConfiguration(
                            validationImplementation = "blAfterStartDateValidator",
                            configurationItems = {
                                    @ConfigurationItem(itemName = "otherField", itemValue = "activeStartDate")
                            })
            })
    protected Date activeEndDate;

    @Override
    public Date getActiveStartDate() {
        return activeStartDate;
    }

    @Override
    public void setActiveStartDate(Date activeStartDate) {
        this.activeStartDate = activeStartDate;
    }

    @Override
    public Date getActiveEndDate() {
        return activeEndDate;
    }

    @Override
    public void setActiveEndDate(Date activeEndDate) {
        this.activeEndDate = activeEndDate;
    }

    @Override
    public Long getId() {
        return id;
    }

    @Override
    public void setId(Long id) {
        this.id = id;
    }

    @Override
    public String getSearchTerm() {
        return searchTerm;
    }

    @Override
    public void setSearchTerm(String searchTerm) {
        this.searchTerm = searchTerm;
    }

    @Override
    public String getUrl() {
        return url;
    }

    @Override
    public void setUrl(String url) {
        this.url = url;
    }

    @Override
    public Integer getSearchPriority() {
        return searchPriority;
    }

    @Override
    public void setSearchPriority(Integer searchPriority) {
        this.searchPriority = searchPriority;
    }

    @Override
    public boolean isActive() {
        Long date = SystemTime.asMillis(true);
        boolean isNullActiveStartDateActive = BLCSystemProperty.resolveBooleanSystemProperty(
                "searchRedirect.is.null.activeStartDate.active");

        boolean isActive;
        if (isNullActiveStartDateActive) {
            isActive = (getActiveStartDate() == null || getActiveStartDate().getTime() <= date)
                    && (getActiveEndDate() == null || getActiveEndDate().getTime() > date);
        } else {
            isActive = (getActiveStartDate() != null && getActiveStartDate().getTime() <= date)
                    && (getActiveEndDate() == null || getActiveEndDate().getTime() > date);
        }

        if (LOG.isDebugEnabled() && !isActive) {
            LOG.debug("product, " + id + ", inactive due to date");
        }
        return isActive;
    }

    @Override
    public String getMainEntityName() {
        return getSearchTerm();
    }

}
