/*-
 * #%L
 * BroadleafCommerce Framework
 * %%
 * Copyright (C) 2009 - 2024 Broadleaf Commerce
 * %%
 * Licensed under the Broadleaf Fair Use License Agreement, Version 1.0
 * (the "Fair Use License" located  at http://license.broadleafcommerce.org/fair_use_license-1.0.txt)
 * unless the restrictions on use therein are violated and require payment to Broadleaf in which case
 * the Broadleaf End User License Agreement (EULA), Version 1.1
 * (the "Commercial License" located at http://license.broadleafcommerce.org/commercial_license-1.1.txt)
 * shall apply.
 *
 * Alternatively, the Commercial License may be replaced with a mutually agreed upon license (the "Custom License")
 * between you and Broadleaf Commerce. You may not use this file except in compliance with the applicable license.
 * #L%
 */
package org.broadleafcommerce.core.payment.domain.secure;

import org.broadleafcommerce.common.encryption.EncryptionModule;
import org.broadleafcommerce.common.persistence.IdOverrideTableGenerator;
import org.broadleafcommerce.core.payment.service.SecureOrderPaymentService;
import org.hibernate.annotations.GenericGenerator;
import org.hibernate.annotations.Parameter;

import java.io.Serial;

import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.GeneratedValue;
import jakarta.persistence.Id;
import jakarta.persistence.Index;
import jakarta.persistence.Inheritance;
import jakarta.persistence.InheritanceType;
import jakarta.persistence.Table;
import jakarta.persistence.Transient;

@Entity
@Inheritance(strategy = InheritanceType.JOINED)
@Table(name = "BLC_GIFT_CARD_PAYMENT", indexes = {@Index(name = "GIFTCARD_INDEX", columnList = "REFERENCE_NUMBER")})
public class GiftCardPaymentImpl implements GiftCardPayment {

    @Serial
    private static final long serialVersionUID = 1L;
    @Transient
    protected EncryptionModule encryptionModule;
    @Id
    @GeneratedValue(generator = "GiftCardPaymentId")
    @GenericGenerator(
            name = "GiftCardPaymentId",
            type = IdOverrideTableGenerator.class,
            parameters = {
                    @Parameter(name = "segment_value", value = "GiftCardPaymentImpl"),
                    @Parameter(name = "entity_name",
                            value = "org.broadleafcommerce.core.payment.domain.GiftCardPaymentInfoImpl")
            }
    )
    @Column(name = "PAYMENT_ID")
    protected Long id;
    @Column(name = "REFERENCE_NUMBER", nullable = false)
    protected String referenceNumber;
    @Column(name = "PAN", nullable = false)
    protected String pan;
    @Column(name = "PIN")
    protected String pin;

    /**
     * Rather than constructing directly, use {@link SecureOrderPaymentService#create(org.broadleafcommerce.core.payment.service.type.PaymentType)}
     * so that the appropriate {@link EncryptionModule} can be hooked up to this entity
     */
    protected GiftCardPaymentImpl() {
        // do not allow direct instantiation -- must at least be package private
        // for bytecode instrumentation
        // this complies with JPA specification requirements for entity
        // construction
    }

    @Override
    public Long getId() {
        return id;
    }

    @Override
    public void setId(Long id) {
        this.id = id;
    }

    @Override
    public String getPan() {
        return encryptionModule.decrypt(pan);
    }

    @Override
    public void setPan(String pan) {
        this.pan = encryptionModule.encrypt(pan);
    }

    @Override
    public String getPin() {
        return encryptionModule.decrypt(pin);
    }

    @Override
    public void setPin(String pin) {
        this.pin = encryptionModule.encrypt(pin);
    }

    @Override
    public String getReferenceNumber() {
        return referenceNumber;
    }

    @Override
    public void setReferenceNumber(String referenceNumber) {
        this.referenceNumber = referenceNumber;
    }

    @Override
    public EncryptionModule getEncryptionModule() {
        return encryptionModule;
    }

    @Override
    public void setEncryptionModule(EncryptionModule encryptionModule) {
        this.encryptionModule = encryptionModule;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((pan == null) ? 0 : pan.hashCode());
        result = prime * result + ((pin == null) ? 0 : pin.hashCode());
        result = prime * result + ((referenceNumber == null) ? 0 : referenceNumber.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (!getClass().isAssignableFrom(obj.getClass()))
            return false;
        GiftCardPaymentImpl other = (GiftCardPaymentImpl) obj;

        if (id != null && other.id != null) {
            return id.equals(other.id);
        }

        if (pan == null) {
            if (other.pan != null)
                return false;
        } else if (!pan.equals(other.pan))
            return false;
        if (pin == null) {
            if (other.pin != null)
                return false;
        } else if (!pin.equals(other.pin))
            return false;
        if (referenceNumber == null) {
            return other.referenceNumber == null;
        } else
            return referenceNumber.equals(other.referenceNumber);
    }

}
