/*-
 * #%L
 * BroadleafCommerce Framework
 * %%
 * Copyright (C) 2009 - 2024 Broadleaf Commerce
 * %%
 * Licensed under the Broadleaf Fair Use License Agreement, Version 1.0
 * (the "Fair Use License" located  at http://license.broadleafcommerce.org/fair_use_license-1.0.txt)
 * unless the restrictions on use therein are violated and require payment to Broadleaf in which case
 * the Broadleaf End User License Agreement (EULA), Version 1.1
 * (the "Commercial License" located at http://license.broadleafcommerce.org/commercial_license-1.1.txt)
 * shall apply.
 *
 * Alternatively, the Commercial License may be replaced with a mutually agreed upon license (the "Custom License")
 * between you and Broadleaf Commerce. You may not use this file except in compliance with the applicable license.
 * #L%
 */
package org.broadleafcommerce.core.order.service.workflow;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.broadleafcommerce.core.catalog.domain.Sku;
import org.broadleafcommerce.core.catalog.service.CatalogService;
import org.broadleafcommerce.core.order.domain.BundleOrderItem;
import org.broadleafcommerce.core.order.domain.DiscreteOrderItem;
import org.broadleafcommerce.core.order.domain.Order;
import org.broadleafcommerce.core.order.domain.OrderItem;
import org.broadleafcommerce.core.order.service.OrderItemService;
import org.broadleafcommerce.core.order.service.call.NonDiscreteOrderItemRequestDTO;
import org.broadleafcommerce.core.order.service.call.OrderItemRequestDTO;
import org.broadleafcommerce.core.workflow.ProcessContext;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;

import jakarta.annotation.Resource;

/**
 * Very similar to the {@link CheckAddAvailabilityActivity} but in the blUpdateItemWorkflow instead
 *
 * @author Phillip Verheyden (phillipuniverse)
 */
@Component("blCheckUpdateAvailabilityActivity")
public class CheckUpdateAvailabilityActivity extends AbstractCheckAvailabilityActivity {

    public static final int ORDER = 2000;
    private static final Log LOG = LogFactory.getLog(CheckUpdateAvailabilityActivity.class);
    @Resource(name = "blCatalogService")
    protected CatalogService catalogService;

    @Resource(name = "blOrderItemService")
    protected OrderItemService orderItemService;

    public CheckUpdateAvailabilityActivity() {
        setOrder(ORDER);
    }

    @Override
    public ProcessContext<CartOperationRequest> execute(ProcessContext<CartOperationRequest> context) throws Exception {
        CartOperationRequest request = context.getSeedData();
        OrderItemRequestDTO orderItemRequestDTO = request.getItemRequest();
        if (orderItemRequestDTO instanceof NonDiscreteOrderItemRequestDTO) {
            return context;
        }

        Sku sku;
        Long orderItemId = request.getItemRequest().getOrderItemId();
        OrderItem orderItem = orderItemService.readOrderItemById(orderItemId);
        if (orderItem instanceof DiscreteOrderItem) {
            sku = ((DiscreteOrderItem) orderItem).getSku();
        } else if (orderItem instanceof BundleOrderItem) {
            sku = ((BundleOrderItem) orderItem).getSku();
        } else {
            LOG.warn("Could not check availability; did not recognize passed-in item " + orderItem.getClass().getName());
            return context;
        }

        if (sku.getProduct().getEnableDefaultSkuInInventory()) {
            sku = sku.getProduct().getDefaultSku();
        }

        Order order = context.getSeedData().getOrder();
        Integer requestedQuantity = request.getItemRequest().getQuantity();
        Map<Sku, Integer> skuItems = new HashMap<>();
        for (OrderItem orderItemFromOrder : order.getOrderItems()) {
            Sku skuFromOrder = null;
            if (orderItemFromOrder instanceof DiscreteOrderItem) {
                skuFromOrder = ((DiscreteOrderItem) orderItemFromOrder).getSku();
            } else if (orderItemFromOrder instanceof BundleOrderItem) {
                skuFromOrder = ((BundleOrderItem) orderItemFromOrder).getSku();
            }
            if (skuFromOrder != null && skuFromOrder.getProduct().getEnableDefaultSkuInInventory()) {
                skuFromOrder = skuFromOrder.getProduct().getDefaultSku();
            }
            if (skuFromOrder != null && skuFromOrder.equals(sku) && !orderItemFromOrder.equals(orderItem)) {
                skuItems.merge(sku, orderItemFromOrder.getQuantity(), (oldVal, newVal) -> oldVal + newVal);
            }
        }
        skuItems.merge(sku, requestedQuantity, (oldVal, newVal) -> oldVal + newVal);
        for (Map.Entry<Sku, Integer> entry : skuItems.entrySet()) {
            checkSkuAvailability(order, entry.getKey(), entry.getValue());
        }

        Integer previousQty = orderItem.getQuantity();
        for (OrderItem child : orderItem.getChildOrderItems()) {
            Sku childSku = ((DiscreteOrderItem) child).getSku();
            if (childSku.getProduct().getEnableDefaultSkuInInventory()) {
                childSku = childSku.getProduct().getDefaultSku();
            }
            Integer childQuantity = child.getQuantity();
            childQuantity = childQuantity / previousQty;
            checkSkuAvailability(order, childSku, childQuantity * requestedQuantity);
        }

        return context;
    }

}
