/*-
 * #%L
 * BroadleafCommerce Framework
 * %%
 * Copyright (C) 2009 - 2024 Broadleaf Commerce
 * %%
 * Licensed under the Broadleaf Fair Use License Agreement, Version 1.0
 * (the "Fair Use License" located  at http://license.broadleafcommerce.org/fair_use_license-1.0.txt)
 * unless the restrictions on use therein are violated and require payment to Broadleaf in which case
 * the Broadleaf End User License Agreement (EULA), Version 1.1
 * (the "Commercial License" located at http://license.broadleafcommerce.org/commercial_license-1.1.txt)
 * shall apply.
 *
 * Alternatively, the Commercial License may be replaced with a mutually agreed upon license (the "Custom License")
 * between you and Broadleaf Commerce. You may not use this file except in compliance with the applicable license.
 * #L%
 */
package org.broadleafcommerce.core.order.domain;

import org.broadleafcommerce.common.copy.MultiTenantCloneable;
import org.broadleafcommerce.core.catalog.domain.Sku;
import org.broadleafcommerce.core.order.service.type.FulfillmentType;
import org.broadleafcommerce.core.pricing.service.fulfillment.provider.FulfillmentPricingProvider;

import java.io.Serializable;

/**
 * A FulfillmentOption is used to hold information about a particular type of Fulfillment implementation.
 * Third-party fulfillment implementations should extend this to provide their own configuration options
 * particular to that implementation. For instance, a UPS shipping calculator might want an admin user to be
 * able to specify which type of UPS shipping this FulfillmentOption represents.
 * <br />
 * <br />
 * This entity will be presented to the user to allow them to specify which shipping they want. A possible
 * scenario is that say a site can ship with both UPS and Fedex. They will import both the Fedex and UPS
 * third-party modules, each of which will have a unique definition of FulfillmentOption (for instance,
 * FedexFulfillmentOption and UPSFulfillmentOption). Let's say the site can do 2-day shipping with UPS,
 * and next-day shipping with Fedex. What they would do in the admin is create an instance of FedexFulfillmentOption
 * entity and give it the name "Overnight" (along with any needed Fedex configuration properties), then create an instance of
 * UPSFulfillmentOption and give it the name "2 Day". When the user goes to check out, they will then see a list
 * with "Overnight" and "2 day" in it. A FulfillmentPricingProvider can then be used to estimate the fulfillment cost
 * (and calculate the fulfillment cost) for that particular option.
 * <br />
 * <br />
 * FulfillmentOptions are also inherently related to FulfillmentProcessors, in that specific types of FulfillmentOption
 * implementations should also have a FulfillmentPricingProvider that can handle operations (estimation and calculation) for
 * pricing a FulfillmentGroup. Typical third-party implementations of this paradigm would have a 1 FulfillmentOption
 * entity implementation and 1 FulfillmentPricingProvider implementation for that particular service.
 *
 * <p>
 * <b>Note: even though the default Broadleaf implementation of this is non-abstract ({@link FulfillmentOptionImpl}),
 * it is very rare that you would actually want to instantiate a raw {@link FulfillmentOptionImpl} rather than using
 * a subclass of this.
 * </p>
 *
 * @author Phillip Verheyden
 * @see {@link FulfillmentPricingProvider}, {@link FulfillmentGroup}
 */
public interface FulfillmentOption extends Serializable, MultiTenantCloneable<FulfillmentOption> {

    Long getId();

    void setId(Long id);

    /**
     * Gets the name displayed to the user when they selected the FulfillmentOption for
     * their order. This might be "2-day" or "Super-saver shipping"
     *
     * @return the display name for this option
     */
    String getName();

    /**
     * Set the display name for this option that will be shown to the user to select from
     * such as "2-day" or "Express" or "Super-saver shipping"
     *
     * @param name - the display name for this option
     */
    void setName(String name);

    /**
     * Gets the long description for this option which can be shown to the user
     * to provide more information about the option they are selecting. An example
     * might be that this is shipped the next business day or that it requires additional
     * processing time
     *
     * @return the description to display to the user
     */
    String getLongDescription();

    /**
     * Sets the long description for this option to show to the user when they select an option
     * for fulfilling their order
     *
     * @param longDescription - the description to show to the user
     */
    void setLongDescription(String longDescription);

    /**
     * Tells the {@link FulfillmentPricingProvider} whether it should try to use the
     * flat rate cost for a {@link Sku} rather than try to factor that {@link Sku}
     * into its shipping calculation. This defaults to <b>true</b>
     *
     * @return <b>true</b> if the {@link FulfillmentPricingProvider} should use
     * the flat rate on a {@link Sku} when it's available rather than try to calculate
     * shipping for it, <b>false</b> otherwise. Default value is <b>true</b>
     */
    Boolean getUseFlatRates();

    /**
     * Tells the {@link FulfillmentPricingProvider} whether it should try to use the
     * flat rate cost for a {@link Sku} rather than try to factor that {@link Sku}
     * into its shipping calculation
     *
     * @param useFlatRates
     */
    void setUseFlatRates(Boolean useFlatRates);

    /**
     * Gets the type of fulfillment that this option supports
     *
     * @return the type of this option
     */
    FulfillmentType getFulfillmentType();

    /**
     * Sets the type of fulfillment that this option supports
     *
     * @param fulfillmentType
     */
    void setFulfillmentType(FulfillmentType fulfillmentType);

    /**
     * Returns taxCode of option
     *
     * @return
     */
    String getTaxCode();

    /**
     * Sets the taxCode of the particular option
     *
     * @param taxCode
     */
    void setTaxCode(String taxCode);

    /**
     * Returns if option is taxable
     *
     * @return
     */
    Boolean getTaxable();

    /**
     * Sets taxability of option
     *
     * @param taxable
     */
    void setTaxable(Boolean taxable);

}
