/*-
 * #%L
 * BroadleafCommerce Framework
 * %%
 * Copyright (C) 2009 - 2024 Broadleaf Commerce
 * %%
 * Licensed under the Broadleaf Fair Use License Agreement, Version 1.0
 * (the "Fair Use License" located  at http://license.broadleafcommerce.org/fair_use_license-1.0.txt)
 * unless the restrictions on use therein are violated and require payment to Broadleaf in which case
 * the Broadleaf End User License Agreement (EULA), Version 1.1
 * (the "Commercial License" located at http://license.broadleafcommerce.org/commercial_license-1.1.txt)
 * shall apply.
 *
 * Alternatively, the Commercial License may be replaced with a mutually agreed upon license (the "Custom License")
 * between you and Broadleaf Commerce. You may not use this file except in compliance with the applicable license.
 * #L%
 */
package org.broadleafcommerce.core.order.domain;

import org.broadleafcommerce.common.copy.CreateResponse;
import org.broadleafcommerce.common.copy.MultiTenantCopyContext;
import org.broadleafcommerce.common.currency.util.BroadleafCurrencyUtils;
import org.broadleafcommerce.common.currency.util.CurrencyCodeIdentifiable;
import org.broadleafcommerce.common.extensibility.jpa.copy.DirectCopyTransform;
import org.broadleafcommerce.common.extensibility.jpa.copy.DirectCopyTransformMember;
import org.broadleafcommerce.common.extensibility.jpa.copy.DirectCopyTransformTypes;
import org.broadleafcommerce.common.money.Money;
import org.broadleafcommerce.common.persistence.IdOverrideTableGenerator;
import org.broadleafcommerce.common.presentation.AdminPresentation;
import org.broadleafcommerce.common.presentation.AdminPresentationClass;
import org.broadleafcommerce.common.presentation.AdminPresentationCollection;
import org.broadleafcommerce.common.presentation.PopulateToOneFieldsEnum;
import org.broadleafcommerce.common.presentation.client.SupportedFieldType;
import org.broadleafcommerce.common.presentation.override.AdminPresentationMergeEntry;
import org.broadleafcommerce.common.presentation.override.AdminPresentationMergeOverride;
import org.broadleafcommerce.common.presentation.override.AdminPresentationMergeOverrides;
import org.broadleafcommerce.common.presentation.override.PropertyType;
import org.broadleafcommerce.core.offer.domain.CandidateFulfillmentGroupOffer;
import org.broadleafcommerce.core.offer.domain.CandidateFulfillmentGroupOfferImpl;
import org.broadleafcommerce.core.offer.domain.FulfillmentGroupAdjustment;
import org.broadleafcommerce.core.offer.domain.FulfillmentGroupAdjustmentImpl;
import org.broadleafcommerce.core.order.service.type.FulfillmentGroupStatusType;
import org.broadleafcommerce.core.order.service.type.FulfillmentType;
import org.broadleafcommerce.profile.core.domain.Address;
import org.broadleafcommerce.profile.core.domain.AddressImpl;
import org.broadleafcommerce.profile.core.domain.Phone;
import org.broadleafcommerce.profile.core.domain.PhoneImpl;
import org.hibernate.annotations.Cache;
import org.hibernate.annotations.CacheConcurrencyStrategy;
import org.hibernate.annotations.GenericGenerator;
import org.hibernate.annotations.Parameter;

import java.io.Serial;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import jakarta.persistence.CascadeType;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.FetchType;
import jakarta.persistence.GeneratedValue;
import jakarta.persistence.Id;
import jakarta.persistence.Index;
import jakarta.persistence.Inheritance;
import jakarta.persistence.InheritanceType;
import jakarta.persistence.JoinColumn;
import jakarta.persistence.JoinTable;
import jakarta.persistence.ManyToOne;
import jakarta.persistence.OneToMany;
import jakarta.persistence.Table;

@Entity
@Inheritance(strategy = InheritanceType.JOINED)
@Table(name = "BLC_FULFILLMENT_GROUP", indexes = {
        @Index(name = "FG_REFERENCE_INDEX", columnList = "REFERENCE_NUMBER"),
        @Index(name = "FG_METHOD_INDEX", columnList = "METHOD"),
        @Index(name = "FG_SERVICE_INDEX", columnList = "SERVICE"),
        @Index(name = "FG_PRIMARY_INDEX", columnList = "IS_PRIMARY"),
        @Index(name = "FG_STATUS_INDEX", columnList = "STATUS"),
        @Index(name = "FG_ORDER_INDEX", columnList = "ORDER_ID"),
        @Index(name = "FG_ADDRESS_INDEX", columnList = "ADDRESS_ID"),
        @Index(name = "FG_PHONE_INDEX", columnList = "PHONE_ID"),
        @Index(name = "FG_MESSAGE_INDEX", columnList = "PERSONAL_MESSAGE_ID")
})
@Cache(usage = CacheConcurrencyStrategy.READ_WRITE, region = "blOrderElements")
@AdminPresentationMergeOverrides({
        @AdminPresentationMergeOverride(name = "", mergeEntries =
        @AdminPresentationMergeEntry(propertyType = PropertyType.AdminPresentation.READONLY,
                booleanOverrideValue = true)),
        @AdminPresentationMergeOverride(name = "currency", mergeEntries =
        @AdminPresentationMergeEntry(propertyType = PropertyType.AdminPresentation.PROMINENT,
                booleanOverrideValue = false)),
        @AdminPresentationMergeOverride(name = "personalMessage", mergeEntries = {
                @AdminPresentationMergeEntry(propertyType = PropertyType.AdminPresentation.TAB,
                        overrideValue = FulfillmentGroupImpl.Presentation.Tab.Name.Advanced),
                @AdminPresentationMergeEntry(propertyType = PropertyType.AdminPresentation.TABORDER,
                        intOverrideValue = FulfillmentGroupImpl.Presentation.Tab.Order.Advanced)
        }),
        @AdminPresentationMergeOverride(name = "address", mergeEntries = {
                @AdminPresentationMergeEntry(propertyType = PropertyType.AdminPresentation.TAB,
                        overrideValue = FulfillmentGroupImpl.Presentation.Tab.Name.Address),
                @AdminPresentationMergeEntry(propertyType = PropertyType.AdminPresentation.TABORDER,
                        intOverrideValue = FulfillmentGroupImpl.Presentation.Tab.Order.Address)
        }),
        @AdminPresentationMergeOverride(name = "address.isDefault", mergeEntries = {
                @AdminPresentationMergeEntry(propertyType = PropertyType.AdminPresentation.EXCLUDED,
                        booleanOverrideValue = true)
        }),
        @AdminPresentationMergeOverride(name = "address.isActive", mergeEntries = {
                @AdminPresentationMergeEntry(propertyType = PropertyType.AdminPresentation.EXCLUDED,
                        booleanOverrideValue = true)
        }),
        @AdminPresentationMergeOverride(name = "address.isBusiness", mergeEntries = {
                @AdminPresentationMergeEntry(propertyType = PropertyType.AdminPresentation.EXCLUDED,
                        booleanOverrideValue = true)
        }),
        @AdminPresentationMergeOverride(name = "phone", mergeEntries = {
                @AdminPresentationMergeEntry(propertyType = PropertyType.AdminPresentation.EXCLUDED,
                        booleanOverrideValue = true)
        }),
        @AdminPresentationMergeOverride(name = "phone.phoneNumber", mergeEntries = {
                @AdminPresentationMergeEntry(propertyType = PropertyType.AdminPresentation.EXCLUDED,
                        booleanOverrideValue = false),
                @AdminPresentationMergeEntry(propertyType = PropertyType.AdminPresentation.ORDER,
                        intOverrideValue = FulfillmentGroupImpl.Presentation.FieldOrder.PHONE),
                @AdminPresentationMergeEntry(propertyType = PropertyType.AdminPresentation.GROUP,
                        overrideValue = "General"),
                @AdminPresentationMergeEntry(propertyType = PropertyType.AdminPresentation.REQUIREDOVERRIDE,
                        overrideValue = "NOT_REQUIRED")
        })
})
@AdminPresentationClass(populateToOneFields = PopulateToOneFieldsEnum.TRUE, friendlyName = "FulfillmentGroupImpl_baseFulfillmentGroup")
@DirectCopyTransform({
        @DirectCopyTransformMember(templateTokens = DirectCopyTransformTypes.MULTITENANT_SITE)
})
public class FulfillmentGroupImpl implements FulfillmentGroup, CurrencyCodeIdentifiable {

    @Serial
    private static final long serialVersionUID = 1L;

    @Id
    @GeneratedValue(generator = "FulfillmentGroupId")
    @GenericGenerator(
            name = "FulfillmentGroupId",
            type = IdOverrideTableGenerator.class,
            parameters = {
                    @Parameter(name = "segment_value", value = "FulfillmentGroupImpl"),
                    @Parameter(name = "entity_name",
                            value = "org.broadleafcommerce.core.order.domain.FulfillmentGroupImpl")
            }
    )
    @Column(name = "FULFILLMENT_GROUP_ID")
    protected Long id;

    @Column(name = "REFERENCE_NUMBER")
    @AdminPresentation(friendlyName = "FulfillmentGroupImpl_FG_Reference_Number", order = Presentation.FieldOrder.REFNUMBER,
            groupOrder = Presentation.Group.Order.General)
    protected String referenceNumber;

    @Column(name = "METHOD")
    @AdminPresentation(excluded = true)
    @Deprecated
    protected String method;

    @Column(name = "SERVICE")
    @AdminPresentation(excluded = true)
    @Deprecated
    protected String service;

    @Column(name = "RETAIL_PRICE", precision = 19, scale = 5)
    @AdminPresentation(friendlyName = "FulfillmentGroupImpl_Retail_Shipping_Price", order = Presentation.FieldOrder.RETAIL,
            group = Presentation.Group.Name.Pricing, groupOrder = Presentation.Group.Order.Pricing,
            tab = Presentation.Tab.Name.Pricing, tabOrder = Presentation.Tab.Order.Pricing,
            fieldType = SupportedFieldType.MONEY)
    protected BigDecimal retailFulfillmentPrice;

    @Column(name = "SALE_PRICE", precision = 19, scale = 5)
    @AdminPresentation(friendlyName = "FulfillmentGroupImpl_Sale_Shipping_Price", order = Presentation.FieldOrder.SALE,
            group = Presentation.Group.Name.Pricing, groupOrder = Presentation.Group.Order.Pricing,
            tab = Presentation.Tab.Name.Pricing, tabOrder = Presentation.Tab.Order.Pricing,
            fieldType = SupportedFieldType.MONEY)
    protected BigDecimal saleFulfillmentPrice;

    @Column(name = "PRICE", precision = 19, scale = 5)
    @AdminPresentation(friendlyName = "FulfillmentGroupImpl_Shipping_Price", order = Presentation.FieldOrder.PRICE,
            group = Presentation.Group.Name.Pricing, groupOrder = Presentation.Group.Order.Pricing,
            tab = Presentation.Tab.Name.Pricing, tabOrder = Presentation.Tab.Order.Pricing,
            fieldType = SupportedFieldType.MONEY)
    protected BigDecimal fulfillmentPrice;

    @Column(name = "TYPE")
    @AdminPresentation(friendlyName = "FulfillmentGroupImpl_FG_Type", order = Presentation.FieldOrder.TYPE,
            fieldType = SupportedFieldType.BROADLEAF_ENUMERATION,
            broadleafEnumeration = "org.broadleafcommerce.core.order.service.type.FulfillmentType",
            prominent = true, gridOrder = 3000)
    protected String type;

    @Column(name = "TOTAL_TAX", precision = 19, scale = 5)
    @AdminPresentation(friendlyName = "FulfillmentGroupImpl_FG_Total_Tax", order = Presentation.FieldOrder.TOTALTAX,
            group = Presentation.Group.Name.Pricing, groupOrder = Presentation.Group.Order.Pricing,
            tab = Presentation.Tab.Name.Pricing, tabOrder = Presentation.Tab.Order.Pricing,
            fieldType = SupportedFieldType.MONEY)
    protected BigDecimal totalTax;

    @Column(name = "TOTAL_ITEM_TAX", precision = 19, scale = 5)
    @AdminPresentation(friendlyName = "FulfillmentGroupImpl_FG_Total_Item_Tax", order = Presentation.FieldOrder.ITEMTAX,
            group = Presentation.Group.Name.Pricing, groupOrder = Presentation.Group.Order.Pricing,
            tab = Presentation.Tab.Name.Pricing, tabOrder = Presentation.Tab.Order.Pricing,
            fieldType = SupportedFieldType.MONEY)
    protected BigDecimal totalItemTax;

    @Column(name = "TOTAL_FEE_TAX", precision = 19, scale = 5)
    @AdminPresentation(friendlyName = "FulfillmentGroupImpl_FG_Total_Fee_Tax", order = Presentation.FieldOrder.FEETAX,
            group = Presentation.Group.Name.Pricing, groupOrder = Presentation.Group.Order.Pricing,
            tab = Presentation.Tab.Name.Pricing, tabOrder = Presentation.Tab.Order.Pricing,
            fieldType = SupportedFieldType.MONEY)
    protected BigDecimal totalFeeTax;

    @Column(name = "TOTAL_FG_TAX", precision = 19, scale = 5)
    @AdminPresentation(friendlyName = "FulfillmentGroupImpl_FG_Total_FG_Tax", order = Presentation.FieldOrder.FGTAX,
            group = Presentation.Group.Name.Pricing, groupOrder = Presentation.Group.Order.Pricing,
            tab = Presentation.Tab.Name.Pricing, tabOrder = Presentation.Tab.Order.Pricing,
            fieldType = SupportedFieldType.MONEY)
    protected BigDecimal totalFulfillmentGroupTax;

    @Column(name = "DELIVERY_INSTRUCTION")
    @AdminPresentation(friendlyName = "FulfillmentGroupImpl_FG_Delivery_Instruction", order = Presentation.FieldOrder.DELIVERINSTRUCTION)
    protected String deliveryInstruction;

    @Column(name = "IS_PRIMARY")
    @AdminPresentation(friendlyName = "FulfillmentGroupImpl_Primary_FG", order = Presentation.FieldOrder.PRIMARY)
    protected boolean primary = false;

    @Column(name = "MERCHANDISE_TOTAL", precision = 19, scale = 5)
    @AdminPresentation(friendlyName = "FulfillmentGroupImpl_FG_Merchandise_Total", order = Presentation.FieldOrder.MERCHANDISETOTAL,
            group = Presentation.Group.Name.Pricing, groupOrder = Presentation.Group.Order.Pricing,
            tab = Presentation.Tab.Name.Pricing, tabOrder = Presentation.Tab.Order.Pricing,
            fieldType = SupportedFieldType.MONEY)
    protected BigDecimal merchandiseTotal;

    @Column(name = "TOTAL", precision = 19, scale = 5)
    @AdminPresentation(friendlyName = "FulfillmentGroupImpl_FG_Total", order = Presentation.FieldOrder.TOTAL,
            group = Presentation.Group.Name.Pricing, groupOrder = Presentation.Group.Order.Pricing,
            tab = Presentation.Tab.Name.Pricing, tabOrder = Presentation.Tab.Order.Pricing,
            fieldType = SupportedFieldType.MONEY, prominent = true, gridOrder = 2000)
    protected BigDecimal total;

    @Column(name = "STATUS")
    @AdminPresentation(friendlyName = "FulfillmentGroupImpl_FG_Status", order = Presentation.FieldOrder.STATUS,
            fieldType = SupportedFieldType.BROADLEAF_ENUMERATION,
            broadleafEnumeration = "org.broadleafcommerce.core.order.service.type.FulfillmentGroupStatusType",
            prominent = true, gridOrder = 4000)
    protected String status;

    @Column(name = "SHIPPING_PRICE_TAXABLE")
    @AdminPresentation(friendlyName = "FulfillmentGroupImpl_Shipping_Price_Taxable", order = Presentation.FieldOrder.TAXABLE,
            group = Presentation.Group.Name.Pricing, groupOrder = Presentation.Group.Order.Pricing,
            tab = Presentation.Tab.Name.Pricing, tabOrder = Presentation.Tab.Order.Pricing)
    protected Boolean isShippingPriceTaxable = Boolean.FALSE;

    @ManyToOne(targetEntity = FulfillmentOptionImpl.class, cascade = {CascadeType.PERSIST, CascadeType.MERGE, CascadeType.REFRESH})
    @JoinColumn(name = "FULFILLMENT_OPTION_ID")
    protected FulfillmentOption fulfillmentOption;

    @ManyToOne(targetEntity = OrderImpl.class, optional = false)
    @JoinColumn(name = "ORDER_ID")
    @AdminPresentation(excluded = true)
    protected Order order;

    @Column(name = "FULFILLMENT_GROUP_SEQUNCE")
    protected Integer sequence;

    @ManyToOne(targetEntity = AddressImpl.class, cascade = {CascadeType.PERSIST, CascadeType.MERGE, CascadeType.REFRESH})
    @JoinColumn(name = "ADDRESS_ID")
    protected Address address;

    /**
     * @deprecated uses the phonePrimary property on AddressImpl instead
     */
    @ManyToOne(targetEntity = PhoneImpl.class, cascade = {CascadeType.PERSIST, CascadeType.MERGE, CascadeType.REFRESH})
    @JoinColumn(name = "PHONE_ID")
    @Deprecated
    protected Phone phone;

    @ManyToOne(targetEntity = PersonalMessageImpl.class, cascade = {CascadeType.ALL})
    @JoinColumn(name = "PERSONAL_MESSAGE_ID")
    protected PersonalMessage personalMessage;

    @OneToMany(mappedBy = "fulfillmentGroup", targetEntity = FulfillmentGroupItemImpl.class, cascade = CascadeType.ALL,
            orphanRemoval = true)
    @Cache(usage = CacheConcurrencyStrategy.READ_WRITE, region = "blOrderElements")
    @AdminPresentationCollection(friendlyName = "FulfillmentGroupImpl_Items",
            tab = Presentation.Tab.Name.Items, tabOrder = Presentation.Tab.Order.Items)
    protected List<FulfillmentGroupItem> fulfillmentGroupItems = new ArrayList<FulfillmentGroupItem>();

    @OneToMany(mappedBy = "fulfillmentGroup", targetEntity = FulfillmentGroupFeeImpl.class, cascade = {CascadeType.ALL},
            orphanRemoval = true)
    @Cache(usage = CacheConcurrencyStrategy.READ_WRITE, region = "blOrderElements")
    @AdminPresentationCollection(friendlyName = "FulfillmentGroupImpl_Fees",
            tab = Presentation.Tab.Name.Pricing, tabOrder = Presentation.Tab.Order.Pricing)
    protected List<FulfillmentGroupFee> fulfillmentGroupFees = new ArrayList<FulfillmentGroupFee>();

    @OneToMany(mappedBy = "fulfillmentGroup", targetEntity = CandidateFulfillmentGroupOfferImpl.class, cascade = {CascadeType.ALL},
            orphanRemoval = true)
    @Cache(usage = CacheConcurrencyStrategy.READ_WRITE, region = "blOrderElements")
    protected List<CandidateFulfillmentGroupOffer> candidateOffers = new ArrayList<CandidateFulfillmentGroupOffer>();

    @OneToMany(mappedBy = "fulfillmentGroup", targetEntity = FulfillmentGroupAdjustmentImpl.class, cascade = {CascadeType.ALL},
            orphanRemoval = true)
    @Cache(usage = CacheConcurrencyStrategy.READ_WRITE, region = "blOrderElements")
    @AdminPresentationCollection(friendlyName = "FulfillmentGroupImpl_Adjustments",
            tab = Presentation.Tab.Name.Advanced, tabOrder = Presentation.Tab.Order.Advanced)
    protected List<FulfillmentGroupAdjustment> fulfillmentGroupAdjustments = new ArrayList<FulfillmentGroupAdjustment>();

    @OneToMany(fetch = FetchType.LAZY, targetEntity = TaxDetailImpl.class, cascade = {CascadeType.ALL}, orphanRemoval = true)
    @JoinTable(name = "BLC_FG_FG_TAX_XREF", joinColumns = @JoinColumn(name = "FULFILLMENT_GROUP_ID"),
            inverseJoinColumns = @JoinColumn(name = "TAX_DETAIL_ID"))
    @Cache(usage = CacheConcurrencyStrategy.READ_WRITE, region = "blOrderElements")
    protected List<TaxDetail> taxes = new ArrayList<TaxDetail>();

    @Column(name = "SHIPPING_OVERRIDE")
    protected Boolean shippingOverride;

    @Override
    public Long getId() {
        return id;
    }

    @Override
    public void setId(Long id) {
        this.id = id;
    }

    @Override
    public Order getOrder() {
        return order;
    }

    @Override
    public void setOrder(Order order) {
        this.order = order;
    }

    @Override
    public FulfillmentOption getFulfillmentOption() {
        return fulfillmentOption;
    }

    @Override
    public void setFulfillmentOption(FulfillmentOption fulfillmentOption) {
        this.fulfillmentOption = fulfillmentOption;
    }

    @Override
    public String getReferenceNumber() {
        return referenceNumber;
    }

    @Override
    public void setReferenceNumber(String referenceNumber) {
        this.referenceNumber = referenceNumber;
    }

    @Override
    public List<FulfillmentGroupItem> getFulfillmentGroupItems() {
        return fulfillmentGroupItems;
    }

    @Override
    public void setFulfillmentGroupItems(List<FulfillmentGroupItem> fulfillmentGroupItems) {
        this.fulfillmentGroupItems = fulfillmentGroupItems;
    }

    @Override
    public List<DiscreteOrderItem> getDiscreteOrderItems() {
        List<DiscreteOrderItem> discreteOrderItems = new ArrayList<DiscreteOrderItem>();
        for (FulfillmentGroupItem fgItem : fulfillmentGroupItems) {
            OrderItem orderItem = fgItem.getOrderItem();
            if (orderItem instanceof BundleOrderItem) {
                BundleOrderItemImpl bundleOrderItem = (BundleOrderItemImpl) orderItem;
                for (DiscreteOrderItem discreteOrderItem : bundleOrderItem.getDiscreteOrderItems()) {
                    discreteOrderItems.add(discreteOrderItem);
                }
            } else if (orderItem instanceof DiscreteOrderItem) {
                DiscreteOrderItem discreteOrderItem = (DiscreteOrderItem) orderItem;
                discreteOrderItems.add(discreteOrderItem);
            }
        }
        return discreteOrderItems;
    }

    @Override
    public void addFulfillmentGroupItem(FulfillmentGroupItem fulfillmentGroupItem) {
        if (this.fulfillmentGroupItems == null) {
            this.fulfillmentGroupItems = new Vector<FulfillmentGroupItem>();
        }
        this.fulfillmentGroupItems.add(fulfillmentGroupItem);

    }

    @Override
    public Address getAddress() {
        return address;
    }

    @Override
    public void setAddress(Address address) {
        this.address = address;
    }

    /**
     * @deprecated use the phonePrimary property on the related Address instead
     */
    @Deprecated
    @Override
    public Phone getPhone() {
        return phone;
    }

    /**
     * @deprecated use the phonePrimary property on the related Address instead
     */
    @Deprecated
    @Override
    public void setPhone(Phone phone) {
        this.phone = phone;
    }

    @Override
    @Deprecated
    public String getMethod() {
        return method;
    }

    @Override
    @Deprecated
    public void setMethod(String fulfillmentMethod) {
        this.method = fulfillmentMethod;
    }

    @Override
    public Money getRetailFulfillmentPrice() {
        return retailFulfillmentPrice == null
                ? null
                : BroadleafCurrencyUtils.getMoney(retailFulfillmentPrice, getOrder().getCurrency());
    }

    @Override
    public void setRetailFulfillmentPrice(Money retailFulfillmentPrice) {
        this.retailFulfillmentPrice = Money.toAmount(retailFulfillmentPrice);
    }

    @Override
    public Money getRetailShippingPrice() {
        return getRetailFulfillmentPrice();
    }

    @Override
    public void setRetailShippingPrice(Money retailShippingPrice) {
        setRetailFulfillmentPrice(retailShippingPrice);
    }

    @Override
    public FulfillmentType getType() {
        return FulfillmentType.getInstance(type);
    }

    @Override
    public void setType(FulfillmentType type) {
        this.type = type == null ? null : type.getType();
    }

    @Override
    public void addCandidateFulfillmentGroupOffer(CandidateFulfillmentGroupOffer candidateOffer) {
        candidateOffers.add(candidateOffer);
    }

    @Override
    public List<CandidateFulfillmentGroupOffer> getCandidateFulfillmentGroupOffers() {
        return candidateOffers;
    }

    @Override
    public void setCandidateFulfillmentGroupOffer(List<CandidateFulfillmentGroupOffer> candidateOffers) {
        this.candidateOffers = candidateOffers;

    }

    @Override
    public void removeAllCandidateOffers() {
        if (candidateOffers != null) {
            for (CandidateFulfillmentGroupOffer offer : candidateOffers) {
                offer.setFulfillmentGroup(null);
            }
            candidateOffers.clear();
        }
    }

    @Override
    public List<FulfillmentGroupAdjustment> getFulfillmentGroupAdjustments() {
        return this.fulfillmentGroupAdjustments;
    }

    @Override
    public void setFulfillmentGroupAdjustments(List<FulfillmentGroupAdjustment> fulfillmentGroupAdjustments) {
        this.fulfillmentGroupAdjustments = fulfillmentGroupAdjustments;
    }

    @Override
    public List<FulfillmentGroupAdjustment> getFutureCreditFulfillmentGroupAdjustments() {
        List<FulfillmentGroupAdjustment> futureCreditAdjustments = new ArrayList<>();
        for (FulfillmentGroupAdjustment adjustment : fulfillmentGroupAdjustments) {
            if (adjustment.isFutureCredit()) {
                futureCreditAdjustments.add(adjustment);
            }

        }
        return futureCreditAdjustments;
    }

    @Override
    public Money getFulfillmentGroupAdjustmentsValue() {
        Money adjustmentsValue = BroadleafCurrencyUtils.getMoney(BigDecimal.ZERO, getOrder().getCurrency());
        for (FulfillmentGroupAdjustment adjustment : fulfillmentGroupAdjustments) {
            if (!adjustment.isFutureCredit()) {
                adjustmentsValue = adjustmentsValue.add(adjustment.getValue());
            }
        }
        return adjustmentsValue;
    }

    @Override
    public Money getFutureCreditFulfillmentGroupAdjustmentsValue() {
        Money adjustmentsValue = BroadleafCurrencyUtils.getMoney(BigDecimal.ZERO, getOrder().getCurrency());
        for (FulfillmentGroupAdjustment adjustment : getFutureCreditFulfillmentGroupAdjustments()) {
            adjustmentsValue = adjustmentsValue.add(adjustment.getValue());
        }
        return adjustmentsValue;
    }

    @Override
    public void removeAllAdjustments() {
        if (fulfillmentGroupAdjustments != null) {
            for (FulfillmentGroupAdjustment adjustment : fulfillmentGroupAdjustments) {
                adjustment.setFulfillmentGroup(null);
            }
            fulfillmentGroupAdjustments.clear();
        }
    }

    @Override
    public Money getSaleFulfillmentPrice() {
        return saleFulfillmentPrice == null ? null : BroadleafCurrencyUtils.getMoney(saleFulfillmentPrice,
                getOrder().getCurrency());
    }

    @Override
    public void setSaleFulfillmentPrice(Money saleFulfillmentPrice) {
        this.saleFulfillmentPrice = Money.toAmount(saleFulfillmentPrice);
    }

    @Override
    public Money getSaleShippingPrice() {
        return getSaleFulfillmentPrice();
    }

    @Override
    public void setSaleShippingPrice(Money saleShippingPrice) {
        setSaleFulfillmentPrice(saleShippingPrice);
    }

    @Override
    public Money getFulfillmentPrice() {
        return fulfillmentPrice == null ? null : BroadleafCurrencyUtils.getMoney(fulfillmentPrice,
                getOrder().getCurrency());
    }

    @Override
    public void setFulfillmentPrice(Money fulfillmentPrice) {
        this.fulfillmentPrice = Money.toAmount(fulfillmentPrice);
    }

    @Override
    public Money getShippingPrice() {
        return getFulfillmentPrice();
    }

    @Override
    public void setShippingPrice(Money shippingPrice) {
        setFulfillmentPrice(shippingPrice);
    }

    @Override
    public List<TaxDetail> getTaxes() {
        return taxes;
    }

    @Override
    public void setTaxes(List<TaxDetail> taxes) {
        this.taxes = taxes;
    }

    @Override
    public Money getTotalTax() {
        return totalTax == null ? null : BroadleafCurrencyUtils.getMoney(totalTax, getOrder().getCurrency());
    }

    @Override
    public void setTotalTax(Money totalTax) {
        this.totalTax = Money.toAmount(totalTax);
    }

    @Override
    public Money getTotalItemTax() {
        return totalItemTax == null ? null : BroadleafCurrencyUtils.getMoney(totalItemTax, getOrder().getCurrency());
    }

    @Override
    public void setTotalItemTax(Money totalItemTax) {
        this.totalItemTax = Money.toAmount(totalItemTax);
    }

    @Override
    public Money getTotalFeeTax() {
        return totalFeeTax == null ? null : BroadleafCurrencyUtils.getMoney(totalFeeTax, getOrder().getCurrency());
    }

    @Override
    public void setTotalFeeTax(Money totalFeeTax) {
        this.totalFeeTax = Money.toAmount(totalFeeTax);
    }

    @Override
    public Money getTotalFulfillmentGroupTax() {
        return totalFulfillmentGroupTax == null
                ? null
                : BroadleafCurrencyUtils.getMoney(totalFulfillmentGroupTax, getOrder().getCurrency());
    }

    @Override
    public void setTotalFulfillmentGroupTax(Money totalFulfillmentGroupTax) {
        this.totalFulfillmentGroupTax = Money.toAmount(totalFulfillmentGroupTax);
    }

    @Override
    public String getDeliveryInstruction() {
        return deliveryInstruction;
    }

    @Override
    public void setDeliveryInstruction(String deliveryInstruction) {
        this.deliveryInstruction = deliveryInstruction;
    }

    @Override
    public PersonalMessage getPersonalMessage() {
        return personalMessage;
    }

    @Override
    public void setPersonalMessage(PersonalMessage personalMessage) {
        this.personalMessage = personalMessage;
    }

    @Override
    public boolean isPrimary() {
        return primary;
    }

    @Override
    public void setPrimary(boolean primary) {
        this.primary = primary;
    }

    @Override
    public Money getMerchandiseTotal() {
        return merchandiseTotal == null
                ? null
                : BroadleafCurrencyUtils.getMoney(merchandiseTotal, getOrder().getCurrency());
    }

    @Override
    public void setMerchandiseTotal(Money merchandiseTotal) {
        this.merchandiseTotal = Money.toAmount(merchandiseTotal);
    }

    @Override
    public Money getTotal() {
        return total == null ? null : BroadleafCurrencyUtils.getMoney(total, getOrder().getCurrency());
    }

    @Override
    public void setTotal(Money orderTotal) {
        this.total = Money.toAmount(orderTotal);
    }

    @Override
    public FulfillmentGroupStatusType getStatus() {
        return FulfillmentGroupStatusType.getInstance(status);
    }

    @Override
    public void setStatus(FulfillmentGroupStatusType status) {
        this.status = status.getType();
    }

    @Override
    public List<FulfillmentGroupFee> getFulfillmentGroupFees() {
        return fulfillmentGroupFees;
    }

    @Override
    public void setFulfillmentGroupFees(List<FulfillmentGroupFee> fulfillmentGroupFees) {
        this.fulfillmentGroupFees = fulfillmentGroupFees;
    }

    @Override
    public void addFulfillmentGroupFee(FulfillmentGroupFee fulfillmentGroupFee) {
        if (fulfillmentGroupFees == null) {
            fulfillmentGroupFees = new ArrayList<FulfillmentGroupFee>();
        }
        fulfillmentGroupFees.add(fulfillmentGroupFee);
    }

    @Override
    public void removeAllFulfillmentGroupFees() {
        if (fulfillmentGroupFees != null) {
            fulfillmentGroupFees.clear();
        }
    }

    @Override
    public Boolean isShippingPriceTaxable() {
        return isShippingPriceTaxable;
    }

    @Override
    public void setIsShippingPriceTaxable(Boolean isShippingPriceTaxable) {
        this.isShippingPriceTaxable = isShippingPriceTaxable;
    }

    @Override
    public Integer getSequence() {
        return this.sequence;
    }

    @Override
    public void setSequence(Integer sequence) {
        this.sequence = sequence;
    }

    @Override
    @Deprecated
    public String getService() {
        return service;
    }

    @Override
    @Deprecated
    public void setService(String service) {
        this.service = service;
    }

    @Override
    public String getCurrencyCode() {
        if (getOrder().getCurrency() != null) {
            return getOrder().getCurrency().getCurrencyCode();
        }
        return null;
    }

    @Override
    public Boolean getShippingOverride() {
        return shippingOverride != null && shippingOverride;
    }

    @Override
    public void setShippingOverride(Boolean shippingOverride) {
        this.shippingOverride = shippingOverride;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((address == null) ? 0 : address.hashCode());
        result = prime * result + ((fulfillmentGroupItems == null) ? 0 : fulfillmentGroupItems.hashCode());
        return result;
    }

    @Override
    public <G extends FulfillmentGroup> CreateResponse<G> createOrRetrieveCopyInstance(
            MultiTenantCopyContext context
    ) throws CloneNotSupportedException {
        CreateResponse<G> createResponse = context.createOrRetrieveCopyInstance(this);
        if (createResponse.isAlreadyPopulated()) {
            return createResponse;
        }
        FulfillmentGroup cloned = createResponse.getClone();
        cloned.setAddress(address == null ? null : address.createOrRetrieveCopyInstance(context).getClone());
        cloned.setDeliveryInstruction(deliveryInstruction);
        cloned.setFulfillmentOption(fulfillmentOption);
        cloned.setFulfillmentPrice(fulfillmentPrice == null ? null : new Money(fulfillmentPrice));
        cloned.setIsShippingPriceTaxable(isShippingPriceTaxable == null ? null : isShippingPriceTaxable);
        cloned.setMerchandiseTotal(merchandiseTotal == null ? null : new Money(merchandiseTotal));
        cloned.setRetailFulfillmentPrice(fulfillmentPrice == null ? null : new Money(fulfillmentPrice));
        cloneTaxDetails(context, cloned);
        cloned.setTotalItemTax(totalItemTax == null ? null : new Money(totalItemTax));
        cloned.setTotalFulfillmentGroupTax(totalFulfillmentGroupTax == null ? null : new Money(totalFulfillmentGroupTax));
        cloned.setTotalFeeTax(totalFeeTax == null ? null : new Money(totalFeeTax));
        cloned.setTotalTax(totalTax == null ? null : new Money(totalTax));
        cloned.setRetailFulfillmentPrice(retailFulfillmentPrice == null ? null : new Money(retailFulfillmentPrice));
        cloned.setSaleFulfillmentPrice(saleFulfillmentPrice == null ? null : new Money(saleFulfillmentPrice));
        cloned.setTotal(total == null ? null : new Money(total));
        cloned.setOrder(order);
        cloned.setPrimary(primary);
        cloned.setType(getType());
        for (FulfillmentGroupItem fgi : fulfillmentGroupItems) {
            FulfillmentGroupItem fulfillmentGroupItem = fgi.createOrRetrieveCopyInstance(context).getClone();
            fulfillmentGroupItem.setFulfillmentGroup(cloned);
            cloned.getFulfillmentGroupItems().add(fulfillmentGroupItem);
        }
        return createResponse;
    }

    protected void cloneTaxDetails(MultiTenantCopyContext context, FulfillmentGroup cloned) throws CloneNotSupportedException {
        for (TaxDetail taxDetail : getTaxes()) {
            TaxDetail clonedTaxDetail = taxDetail.createOrRetrieveCopyInstance(context).getClone();
            cloned.getTaxes().add(clonedTaxDetail);
        }
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!getClass().isAssignableFrom(obj.getClass())) {
            return false;
        }
        FulfillmentGroupImpl other = (FulfillmentGroupImpl) obj;

        if (id != null && other.id != null) {
            return id.equals(other.id);
        }

        if (address == null) {
            if (other.address != null) {
                return false;
            }
        } else if (!address.equals(other.address)) {
            return false;
        }
        if (fulfillmentGroupItems == null) {
            return other.fulfillmentGroupItems == null;
        } else
            return fulfillmentGroupItems.equals(other.fulfillmentGroupItems);
    }

    public static class Presentation {
        public static class Tab {
            public static class Name {
                public static final String Items = "FulfillmentGroupImpl_Items_Tab";
                public static final String Pricing = "FulfillmentGroupImpl_Pricing_Tab";
                public static final String Address = "FulfillmentGroupImpl_Address_Tab";
                public static final String Advanced = "FulfillmentGroupImpl_Advanced_Tab";
            }

            public static class Order {
                public static final int Items = 2000;
                public static final int Pricing = 3000;
                public static final int Address = 4000;
                public static final int Advanced = 5000;
            }
        }

        public static class Group {
            public static class Name {
                public static final String Pricing = "FulfillmentGroupImpl_Pricing";
            }

            public static class Order {
                public static final int General = 1000;
                public static final int Pricing = 2000;
            }
        }

        public static class FieldOrder {
            public static final int REFNUMBER = 3000;
            public static final int STATUS = 4000;
            public static final int TYPE = 5000;
            public static final int DELIVERINSTRUCTION = 6000;
            public static final int PRIMARY = 7000;
            public static final int PHONE = 8000;

            public static final int RETAIL = 1000;
            public static final int SALE = 2000;
            public static final int PRICE = 3000;
            public static final int ITEMTAX = 4000;
            public static final int FEETAX = 5000;
            public static final int FGTAX = 6000;
            public static final int TOTALTAX = 7000;
            public static final int MERCHANDISETOTAL = 8000;
            public static final int TOTAL = 9000;
            public static final int TAXABLE = 10000;
        }
    }

}
