/*-
 * #%L
 * BroadleafCommerce Framework
 * %%
 * Copyright (C) 2009 - 2024 Broadleaf Commerce
 * %%
 * Licensed under the Broadleaf Fair Use License Agreement, Version 1.0
 * (the "Fair Use License" located  at http://license.broadleafcommerce.org/fair_use_license-1.0.txt)
 * unless the restrictions on use therein are violated and require payment to Broadleaf in which case
 * the Broadleaf End User License Agreement (EULA), Version 1.1
 * (the "Commercial License" located at http://license.broadleafcommerce.org/commercial_license-1.1.txt)
 * shall apply.
 *
 * Alternatively, the Commercial License may be replaced with a mutually agreed upon license (the "Custom License")
 * between you and Broadleaf Commerce. You may not use this file except in compliance with the applicable license.
 * #L%
 */
package org.broadleafcommerce.core.catalog.domain;

import org.broadleafcommerce.common.copy.CreateResponse;
import org.broadleafcommerce.common.copy.MultiTenantCopyContext;
import org.broadleafcommerce.common.extensibility.jpa.copy.DirectCopyTransform;
import org.broadleafcommerce.common.extensibility.jpa.copy.DirectCopyTransformMember;
import org.broadleafcommerce.common.extensibility.jpa.copy.DirectCopyTransformTypes;
import org.broadleafcommerce.common.persistence.IdOverrideTableGenerator;
import org.broadleafcommerce.common.presentation.AdminPresentation;
import org.broadleafcommerce.common.presentation.AdminPresentationClass;
import org.broadleafcommerce.common.presentation.client.VisibilityEnum;
import org.hibernate.annotations.Cache;
import org.hibernate.annotations.CacheConcurrencyStrategy;
import org.hibernate.annotations.GenericGenerator;
import org.hibernate.annotations.Parameter;

import java.io.Serial;
import java.math.BigDecimal;

import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.GeneratedValue;
import jakarta.persistence.Id;
import jakarta.persistence.Inheritance;
import jakarta.persistence.InheritanceType;
import jakarta.persistence.JoinColumn;
import jakarta.persistence.ManyToOne;
import jakarta.persistence.Table;

import static org.broadleafcommerce.common.copy.MultiTenantCopyContext.MANUAL_DUPLICATION;

/**
 * The Class CategoryProductXrefImpl is the default implmentation of {@link Category}. This entity
 * is only used for executing a named query.
 * <p>
 * If you want to add fields specific to your implementation of BroadLeafCommerce you should extend
 * this class and add your fields.  If you need to make significant changes to the class then you
 * should implement your own version of {@link Category}.
 * <br>
 * <br>
 * This implementation uses a Hibernate implementation of JPA configured through annotations. The
 * Entity references the following tables: BLC_CATEGORY_PRODUCT_XREF,
 *
 * @author btaylor
 * @see {@link Category}, {@link ProductImpl}
 */
@Entity
@Inheritance(strategy = InheritanceType.JOINED)
@Table(name = "BLC_CATEGORY_PRODUCT_XREF")
@AdminPresentationClass(excludeFromPolymorphism = false)
@Cache(usage = CacheConcurrencyStrategy.READ_WRITE, region = "blCategoryProduct")
@DirectCopyTransform({
        @DirectCopyTransformMember(templateTokens = DirectCopyTransformTypes.SANDBOX, skipOverlaps = true),
        @DirectCopyTransformMember(templateTokens = DirectCopyTransformTypes.MULTITENANT_CATALOG)
})
public class CategoryProductXrefImpl implements CategoryProductXref {

    /**
     * The Constant serialVersionUID.
     */
    @Serial
    private static final long serialVersionUID = 1L;

    @Id
    @GeneratedValue(generator = "CategoryProductId")
    @GenericGenerator(
            name = "CategoryProductId",
            type = IdOverrideTableGenerator.class,
            parameters = {
                    @Parameter(name = "segment_value", value = "CategoryProductXrefImpl"),
                    @Parameter(name = "entity_name",
                            value = "org.broadleafcommerce.core.catalog.domain.CategoryProductXrefImpl")
            }
    )
    @Column(name = "CATEGORY_PRODUCT_ID")
    protected Long id;

    //todo: 6.3 cascade refresh was deleted due to potential hiberante issue, when after a clone process
    //we refresh original & clone records, and after refresh it will be flushed at some point
    //and during flush there will be exception about shared collection in sku.skuPriceData
    @ManyToOne(targetEntity = CategoryImpl.class, optional = false)
    @JoinColumn(name = "CATEGORY_ID")
    protected Category category = new CategoryImpl();

    //todo: 6.3 cascade refresh was deleted due to potential hiberante issue, when after a clone process
    //we refresh original & clone records, and after refresh it will be flushed at some point
    //and during flush there will be exception about shared collection in sku.skuPriceData
    /**
     * The product.
     */
    @ManyToOne(targetEntity = ProductImpl.class, optional = false)
    @JoinColumn(name = "PRODUCT_ID")
    protected Product product = new ProductImpl();

    /**
     * The display order.
     */
    @Column(name = "DISPLAY_ORDER", precision = 10, scale = 6)
    @AdminPresentation(friendlyName = "CategoryProductXrefImpl_displayOrder",
            visibility = VisibilityEnum.HIDDEN_ALL)
    protected BigDecimal displayOrder;

    @Column(name = "DEFAULT_REFERENCE")
    @AdminPresentation(visibility = VisibilityEnum.HIDDEN_ALL)
    protected Boolean defaultReference;

    @Override
    public BigDecimal getDisplayOrder() {
        return displayOrder;
    }

    @Override
    public void setDisplayOrder(BigDecimal displayOrder) {
        this.displayOrder = displayOrder;
    }

    @Override
    public Category getCategory() {
        return category;
    }

    @Override
    public void setCategory(Category category) {
        this.category = category;
    }

    @Override
    public Product getProduct() {
        return product;
    }

    @Override
    public void setProduct(Product product) {
        this.product = product;
    }

    @Override
    public Long getId() {
        return id;
    }

    @Override
    public void setId(Long id) {
        this.id = id;
    }

    @Override
    public Boolean getDefaultReference() {
        return defaultReference;
    }

    @Override
    public void setDefaultReference(Boolean defaultReference) {
        this.defaultReference = defaultReference;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null)
            return false;
        if (!getClass().isAssignableFrom(o.getClass()))
            return false;

        CategoryProductXrefImpl that = (CategoryProductXrefImpl) o;

        if (category != null ? !category.equals(that.category) : that.category != null)
            return false;
        if (product != null ? !product.equals(that.product) : that.product != null)
            return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = category != null ? category.hashCode() : 0;
        result = 31 * result + (product != null ? product.hashCode() : 0);
        return result;
    }

    @Override
    public <G extends CategoryProductXref> CreateResponse<G> createOrRetrieveCopyInstance(
            MultiTenantCopyContext context) throws CloneNotSupportedException {
        CreateResponse<G> createResponse = context.createOrRetrieveCopyInstance(this);
        if (createResponse.isAlreadyPopulated()) {
            return createResponse;
        }
        CategoryProductXref cloned = createResponse.getClone();
        cloned.setDisplayOrder(displayOrder);
        cloned.setDefaultReference(defaultReference);
        if (!context.getCopyHints().containsKey(MANUAL_DUPLICATION)) {
            if (product != null) {
                cloned.setProduct(product.createOrRetrieveCopyInstance(context).getClone());
            }
            if (category != null) {
                cloned.setCategory(category.createOrRetrieveCopyInstance(context).getClone());
            }
        } else {
            cloned.setProduct(product);
            cloned.setCategory(category);
        }
        return createResponse;
    }

}
