/*-
 * #%L
 * BroadleafCommerce Framework
 * %%
 * Copyright (C) 2009 - 2024 Broadleaf Commerce
 * %%
 * Licensed under the Broadleaf Fair Use License Agreement, Version 1.0
 * (the "Fair Use License" located  at http://license.broadleafcommerce.org/fair_use_license-1.0.txt)
 * unless the restrictions on use therein are violated and require payment to Broadleaf in which case
 * the Broadleaf End User License Agreement (EULA), Version 1.1
 * (the "Commercial License" located at http://license.broadleafcommerce.org/commercial_license-1.1.txt)
 * shall apply.
 *
 * Alternatively, the Commercial License may be replaced with a mutually agreed upon license (the "Custom License")
 * between you and Broadleaf Commerce. You may not use this file except in compliance with the applicable license.
 * #L%
 */
package org.broadleafcommerce.core.catalog.dao;

import org.broadleafcommerce.core.catalog.domain.Category;
import org.broadleafcommerce.core.catalog.domain.CategoryProductXref;
import org.broadleafcommerce.core.catalog.domain.Product;

import java.util.List;

import jakarta.annotation.Nonnull;

/**
 * {@code CategoryDao} provides persistence access to {@code Category} instances.
 *
 * @author Jeff Fischer
 * @see Category
 * @see Product
 */
public interface CategoryDao {

    /**
     * Retrieve a {@code Category} instance by its primary key
     *
     * @param categoryId the primary key of the {@code Category}
     * @return the {@code Category}  at the specified primary key
     */
    @Nonnull
    Category readCategoryById(@Nonnull Long categoryId);

    /**
     * Retrieves a List of Category IDs
     *
     * @param categoryIds
     * @return
     */
    List<Category> readCategoriesByIds(List<Long> categoryIds);

    /**
     * Retrieve a {@link Category} instance by the external id
     *
     * @param externalId
     * @return
     */
    Category readCategoryByExternalId(@Nonnull String externalId);

    /**
     * Retrieve a {@code Category} instance by its name.
     * <p>
     * Broadleaf allows more than one category to have the same name. Calling
     * this method could produce an exception in such situations. Use
     * {@link #readCategoriesByName(String)} instead.
     *
     * @param categoryName the name of the category
     * @return the Category having the specified name
     */
    @Nonnull
    @Deprecated(forRemoval = true)
    Category readCategoryByName(@Nonnull String categoryName);

    /**
     * Retrieve a list of {@code Category} instances by name.
     *
     * @param categoryName the name to search by
     * @return the Category instances having the specified name
     */
    @Nonnull
    List<Category> readCategoriesByName(@Nonnull String categoryName);

    @Nonnull
    List<Category> readCategoriesByName(@Nonnull String categoryName, int limit, int offset);

    @Nonnull
    List<Category> readCategoriesByNames(List<String> names);

    /**
     * Persist a {@code Category} instance to the datastore
     *
     * @param category the {@code Category} instance
     * @return the updated state of the passed in {@code Category} after being persisted
     */
    @Nonnull
    Category save(@Nonnull Category category);

    /**
     * Retrieve all categories in the datastore
     *
     * @return a list of all the {@code Category} instances in the datastore
     */
    @Nonnull
    List<Category> readAllCategories();

    /**
     * Retrieve a subset of all categories
     *
     * @param limit  the maximum number of results, defaults to 20
     * @param offset the starting point in the record set, defaults to 0
     * @return
     */
    @Nonnull
    List<Category> readAllCategories(@Nonnull int limit, @Nonnull int offset);

    Long readTotalCategoryCount();

    /**
     * Retrieve all products in the datastore
     *
     * @return a list of all {@code Category} instances in the datastore, regardless of their category association
     */
    @Nonnull
    List<Product> readAllProducts();

    @Nonnull
    List<Product> readAllProducts(@Nonnull int limit, @Nonnull int offset);

    /**
     * Retrieve a list of all child categories of the passed in {@code Category} instance
     *
     * @param category the parent category
     * @return a list of all child categories
     */
    @Nonnull
    List<Category> readAllSubCategories(@Nonnull Category category);

    /**
     * Retrieve a list of all child categories of the passed in {@code Category} instance
     *
     * @param id the parent category ID
     * @return a list of all child categories
     */
    @Nonnull
    List<Category> readAllSubCategories(@Nonnull Long id);

    /**
     * Retrieve a list of all child categories of the passed in {@code Category} instance
     *
     * @param category the parent category
     * @param limit    the maximum number of results to return
     * @param offset   the starting point in the record set
     * @return a list of all child categories
     */
    @Nonnull
    List<Category> readAllSubCategories(@Nonnull Category category, @Nonnull int limit, @Nonnull int offset);

    /**
     * Removed the passed in {@code Category} instance from the datastore
     *
     * @param category the {@code Category} instance to remove
     */
    void delete(@Nonnull Category category);

    /**
     * Create a new {@code Category} instance. The system will use the configuration in
     * {@code /BroadleafCommerce/core/BroadleafCommerceFramework/src/main/resources/bl-framework-applicationContext-entity.xml}
     * to determine which polymorphic version of {@code Category} to instantiate. To make Broadleaf instantiate your
     * extension of {@code Category} by default, include an entity configuration bean in your application context xml similar to:
     * <p>
     * <pre>
     * {@code
     *     <bean id="blEntityConfiguration" class="org.broadleafcommerce.common.persistence.EntityConfiguration">
     *          <property name="entityContexts">
     *              <list>
     *                  <value>classpath:myCompany-applicationContext-entity.xml</value>
     *              </list>
     *          </property>
     *      </bean>
     * }
     * </pre>
     * </p>
     * Declare the same key for your desired entity in your entity xml that is used in the Broadleaf entity xml, but change the value to the fully
     * qualified classname of your entity extension.
     *
     * @return a {@code Category} instance based on the Broadleaf entity configuration.
     */
    @Nonnull
    Category create();

    /**
     * Retrieve a list of all active child categories of the passed in {@code Category} instance.
     * This method bases its search on a current time value. To make the retrieval of values more
     * efficient, the current time is cached for a configurable amount of time. See
     * {@link #getCurrentDateResolution()}
     *
     * @param category the parent category
     * @return a list of all active child categories
     */
    @Nonnull
    List<Category> readActiveSubCategoriesByCategory(Category category);

    /**
     * Retrieve a list of all active child categories of the passed in {@code Category} instance.
     * This method bases its search on a current time value. To make the retrieval of values more
     * efficient, the current time is cached for a configurable amount of time. See
     * {@link #getCurrentDateResolution()}
     *
     * @param category the parent category
     * @param limit    the maximum number of results to return
     * @param offset   the starting point in the record set
     * @return a list of all active child categories
     */
    @Nonnull
    List<Category> readActiveSubCategoriesByCategory(@Nonnull Category category, @Nonnull int limit, @Nonnull int offset);

    Category findCategoryByURI(String uri);

    /**
     * Returns the number of milliseconds that the current date/time will be cached for queries before refreshing.
     * This aids in query caching, otherwise every query that utilized current date would be different and caching
     * would be ineffective.
     *
     * @return the milliseconds to cache the current date/time
     */
    Long getCurrentDateResolution();

    /**
     * Sets the number of milliseconds that the current date/time will be cached for queries before refreshing.
     * This aids in query caching, otherwise every query that utilized current date would be different and caching
     * would be ineffective.
     *
     * @param currentDateResolution the milliseconds to cache the current date/time
     */
    void setCurrentDateResolution(Long currentDateResolution);

    Long readCountAllActiveProductsByCategory(Category category);

    /**
     * returns a list of CategoryProductXref where provided category is default reference(xref.defaultReference=true)
     *
     * @param categoryId long, id of the category
     * @return returns a list of CategoryProductXref where provided category is default reference(xref.defaultReference=true)
     */
    List<CategoryProductXref> findXrefByCategoryWithDefaultReference(Long categoryId);

}
