/*-
 * #%L
 * BroadleafCommerce Common Libraries
 * %%
 * Copyright (C) 2009 - 2024 Broadleaf Commerce
 * %%
 * Licensed under the Broadleaf Fair Use License Agreement, Version 1.0
 * (the "Fair Use License" located  at http://license.broadleafcommerce.org/fair_use_license-1.0.txt)
 * unless the restrictions on use therein are violated and require payment to Broadleaf in which case
 * the Broadleaf End User License Agreement (EULA), Version 1.1
 * (the "Commercial License" located at http://license.broadleafcommerce.org/commercial_license-1.1.txt)
 * shall apply.
 *
 * Alternatively, the Commercial License may be replaced with a mutually agreed upon license (the "Custom License")
 * between you and Broadleaf Commerce. You may not use this file except in compliance with the applicable license.
 * #L%
 */
package org.broadleafcommerce.common.web.resource.resolver;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.broadleafcommerce.common.site.domain.Site;
import org.broadleafcommerce.common.web.BroadleafRequestContext;
import org.springframework.core.Ordered;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.resource.AbstractResourceResolver;
import org.springframework.web.servlet.resource.ResourceResolver;
import org.springframework.web.servlet.resource.ResourceResolverChain;

import java.util.List;

import jakarta.servlet.http.HttpServletRequest;

/**
 * A {@link ResourceResolver} that replaces the //BLC-SERVLET-CONTEXT and //BLC-SITE-BASEURL"
 * tokens before serving the BLC.js file.
 * <p>
 * This component modifies the path and works in conjunction with the {@link BLCJSResourceResolver}
 * which loads the modified file.
 * <p>
 * The processes were split to allow for caching of the resource but not the URL path.
 *
 * @author Reggie Cole
 * @author Brian Polster
 * @since 4.0
 * @since Broadleaf 4.0
 */
@Component("blBLCJSUrlPathResolver")
public class BLCJSUrlPathResolver extends AbstractResourceResolver implements Ordered {

    protected static final Log LOG = LogFactory.getLog(BLCJSUrlPathResolver.class);

    private static final String BLC_JS_NAME = "BLC.js";

    private int order = BroadleafResourceResolverOrder.BLC_JS_PATH_RESOLVER;

    @Override
    protected String resolveUrlPathInternal(
            String resourceUrlPath,
            List<? extends Resource> locations,
            ResourceResolverChain chain
    ) {
        if (resourceUrlPath.contains(BLC_JS_NAME)) {
            Site site = BroadleafRequestContext.getBroadleafRequestContext().getNonPersistentSite();
            if (site != null && site.getId() != null) {
                return addVersion(resourceUrlPath, "-" + site.getId());
            } else {
                return resourceUrlPath;
            }
        }
        return chain.resolveUrlPath(resourceUrlPath, locations);
    }

    @Override
    protected Resource resolveResourceInternal(
            HttpServletRequest request,
            String requestPath,
            List<? extends Resource> locations,
            ResourceResolverChain chain
    ) {
        return chain.resolveResource(request, requestPath, locations);
    }

    protected String addVersion(String requestPath, String version) {
        String baseFilename = StringUtils.stripFilenameExtension(requestPath);
        String extension = StringUtils.getFilenameExtension(requestPath);
        return baseFilename + version + "." + extension;
    }

    @Override
    public int getOrder() {
        return order;
    }

    public void setOrder(int order) {
        this.order = order;
    }

}
