/*-
 * #%L
 * BroadleafCommerce Common Libraries
 * %%
 * Copyright (C) 2009 - 2024 Broadleaf Commerce
 * %%
 * Licensed under the Broadleaf Fair Use License Agreement, Version 1.0
 * (the "Fair Use License" located  at http://license.broadleafcommerce.org/fair_use_license-1.0.txt)
 * unless the restrictions on use therein are violated and require payment to Broadleaf in which case
 * the Broadleaf End User License Agreement (EULA), Version 1.1
 * (the "Commercial License" located at http://license.broadleafcommerce.org/commercial_license-1.1.txt)
 * shall apply.
 *
 * Alternatively, the Commercial License may be replaced with a mutually agreed upon license (the "Custom License")
 * between you and Broadleaf Commerce. You may not use this file except in compliance with the applicable license.
 * #L%
 */
package org.broadleafcommerce.common.security.service;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.text.StringEscapeUtils;
import org.broadleafcommerce.common.exception.SecurityServiceException;
import org.broadleafcommerce.common.exception.ServiceException;
import org.broadleafcommerce.common.security.RandomGenerator;
import org.broadleafcommerce.common.util.BLCRequestUtils;
import org.broadleafcommerce.common.web.BroadleafRequestContext;
import org.owasp.validator.html.CleanResults;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.context.request.ServletWebRequest;

import java.security.NoSuchAlgorithmException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpSession;

/**
 * @author jfischer
 */
@Service("blExploitProtectionService")
public class ExploitProtectionServiceImpl implements ExploitProtectionService {

    private static final String CSRF_TOKEN = "csrfToken";
    private static final Log LOG = LogFactory.getLog(ExploitProtectionServiceImpl.class);
    private static final String JAVASCRIPT = "javascript";
    private static final String HTML_PATTERN = "<(\"[^\"]*\"|'[^']*'|[^'\">])*>";
    private final Pattern pattern = Pattern.compile(HTML_PATTERN);
    @Value("${exploitProtection.xsrfEnabled:true}")
    protected boolean xsrfProtectionEnabled;
    @Value("${exploitProtection.xssEnabled:true}")
    protected boolean xssProtectionEnabled;

    @Override
    public String cleanString(String string) throws ServiceException {
        if (!xssProtectionEnabled || StringUtils.isEmpty(string)) {
            return string;
        }
        try {
            AntisamyService instance = AntisamyServiceImpl.getInstance();
            CleanResults results = instance.getAntiSamy().scan(string, instance.getAntiSamyPolicy());
            return results.getCleanHTML();
        } catch (Exception e) {
            LOG.error("Unable to clean the passed in entity values", e);
            throw new ServiceException("Unable to clean the passed in entity values", e);
        }
    }

    @Override
    public String cleanStringWithResults(String string) throws ServiceException {
        if (!xssProtectionEnabled || StringUtils.isEmpty(string)) {
            return string;
        }
        AntisamyService instance = AntisamyServiceImpl.getInstance();
        try {
            CleanResults results = instance.getAntiSamy().scan(string, instance.getAntiSamyPolicy());
            if (results.getNumberOfErrors() > 0) {
                throw new CleanStringException(results);
            } else {
                //ok no errors, but what if it has regular string with javascript:xxx and it possibly can be used to construct anchor link
                if (string.toLowerCase().startsWith(JAVASCRIPT)) {
                    throw new CleanStringException(results);
                }
            }
            return results.getCleanHTML();
        } catch (CleanStringException e) {
            throw e;
        } catch (Exception e) {
            StringBuilder sb = new StringBuilder();
            sb.append("Unable to clean the passed in entity values");
            sb.append("\nNote - ");
            sb.append(instance.getAntiSamyPolicyFileLocation());
            sb.append(" policy in effect. Set a new policy file to modify validation behavior/strictness.");
            LOG.error(sb.toString(), e);
            throw new ServiceException(sb.toString(), e);
        }
    }

    @Override
    public void compareToken(String passedToken) throws ServiceException {
        if (xsrfProtectionEnabled) {
            if (!getCSRFToken().equals(passedToken)) {
                throw new SecurityServiceException("XSRF token mismatch (" + passedToken + "). Session may be expired.");
            } else {
                LOG.debug("Validated CSRF token");
            }
        }
    }

    @Override
    public String getCSRFToken() throws ServiceException {
        //If using any request wrapper (e.g. Spring Session) on site, we'll need to return the decorated instance.
        HttpServletRequest request = BroadleafRequestContext.getBroadleafRequestContext().getRequest();
        if (request == null) {
            //in the case of the admin, the brc is not used, so fallback to Spring's request context holder
            request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
        }

        if (BLCRequestUtils.isOKtoUseSession(new ServletWebRequest(request))) {
            HttpSession session = request.getSession();
            String token = (String) session.getAttribute(CSRF_TOKEN);
            if (StringUtils.isEmpty(token)) {
                try {
                    token = RandomGenerator.generateRandomId("SHA1PRNG", 32);
                } catch (NoSuchAlgorithmException e) {
                    LOG.error("Unable to generate random number", e);
                    throw new ServiceException("Unable to generate random number", e);
                }
                session.setAttribute(CSRF_TOKEN, token);
            }
            return token;
        }
        return null;
    }

    @Override
    public String getCsrfTokenParameter() {
        return CSRF_TOKEN;
    }

    @Override
    public String htmlDecode(String value) {
        if (xssProtectionEnabled) {
            return StringEscapeUtils.unescapeHtml4(value);
        } else {
            return value;
        }
    }

    public void setXssProtectionEnabled(boolean xssProtectionEnabled) {
        this.xssProtectionEnabled = xssProtectionEnabled;
    }

}
