/*-
 * #%L
 * BroadleafCommerce Common Libraries
 * %%
 * Copyright (C) 2009 - 2024 Broadleaf Commerce
 * %%
 * Licensed under the Broadleaf Fair Use License Agreement, Version 1.0
 * (the "Fair Use License" located  at http://license.broadleafcommerce.org/fair_use_license-1.0.txt)
 * unless the restrictions on use therein are violated and require payment to Broadleaf in which case
 * the Broadleaf End User License Agreement (EULA), Version 1.1
 * (the "Commercial License" located at http://license.broadleafcommerce.org/commercial_license-1.1.txt)
 * shall apply.
 *
 * Alternatively, the Commercial License may be replaced with a mutually agreed upon license (the "Custom License")
 * between you and Broadleaf Commerce. You may not use this file except in compliance with the applicable license.
 * #L%
 */
package org.broadleafcommerce.common.extension;

import org.broadleafcommerce.common.i18n.service.SparseTranslationOverrideStrategy;

import java.util.List;

import jakarta.persistence.TypedQuery;
import jakarta.persistence.criteria.CriteriaBuilder;
import jakarta.persistence.criteria.CriteriaQuery;
import jakarta.persistence.criteria.Order;
import jakarta.persistence.criteria.Predicate;
import jakarta.persistence.criteria.Root;

/**
 * Extension handler (generally for DAO usage) that allows contribution to a query (presumably from another module). The intent
 * of this handler is to manipulate a query to not include standard site catalogs (template only). This is useful in some
 * caching situations where it is advantageous to only look at template catalog values. {@link SparseTranslationOverrideStrategy}
 * is an example use case. This is generally used in multitenant scenarios.
 *
 * @author Jeff Fischer
 */
public interface TemplateOnlyQueryExtensionHandler extends ExtensionHandler {

    /**
     * Finish the query - possibly setting parameters
     *
     * @param type       the class type for the query
     * @param testObject supporting implementations may use this object to test for possible catalog query optimizations. This value can be null, in which case it is ignored.
     * @param query      the final Query instance to embellish
     * @return
     */
    ExtensionResultStatusType refineQuery(Class<?> type, Object testObject, TypedQuery query);

    /**
     * Add additional restrictions to the fetch query. Use in conjunction with {@link #refineQuery(Class, Object, TypedQuery)} to set
     * actual parameter values before retrieving results.
     *
     * @param type         the class type for the query
     * @param testObject   supporting implementations may use this object to test for possible catalog query optimizations. This value can be null, in which case it is ignored.
     * @param builder
     * @param criteria
     * @param root
     * @param restrictions any additional JPA criteria restrictions should be added here
     * @return the status of the extension operation
     */
    ExtensionResultStatusType refineParameterRetrieve(Class<?> type, Object testObject, CriteriaBuilder builder,
                                                      CriteriaQuery criteria, Root root, List<Predicate> restrictions);

    /**
     * Perform any setup operations. This is usually done before executing the query and can serve to prepare the BroadleafRequestContext (if applicable).
     *
     * @param type the class type for the query
     * @return the status of the extension operation
     */
    ExtensionResultStatusType setup(Class<?> type);

    /**
     * Perform any breakdown operations. This is usually done after executing the query and can serve to reset the BroadleafRequestContext (if applicable)
     *
     * @param type the class type for the query
     * @return the status of the extension operation
     */
    ExtensionResultStatusType breakdown(Class<?> type);

    /**
     * Add sorting to the fetch query
     *
     * @param type     the class type for the query
     * @param builder
     * @param criteria
     * @param root
     * @param sorts    any additional JPA order expressions should be added here
     * @return the status of the extension operation
     */
    ExtensionResultStatusType refineOrder(Class<?> type, CriteriaBuilder builder, CriteriaQuery criteria, Root root, List<Order> sorts);

    /**
     * Determine if the current thread is in a valid state for sparse cache handling
     *
     * @param response
     * @return
     */
    ExtensionResultStatusType isValidState(ExtensionResultHolder<Boolean> response);

    ExtensionResultStatusType buildStatus(Object entity, ExtensionResultHolder<ItemStatus> response);

    /**
     * Validate and filter the results. This can be interesting when you have restricted the query with a test object and
     * need to confirm the validity of the results. A nuanced example of this would be translations associated with a profile
     * entity (e.g. StructuredContent). If you filter by the StructuredContent, you will be filtering translations by owning
     * site. However, the resulting translations are not profile entities and will not available necessarily to the requesting
     * site. As a result, we filter here and check the translations based on catalog visibility (Translations are dual
     * discriminated). This is primarily to handle edge cases and will generally have no impact on the results.
     *
     * @param type
     * @param testObject
     * @param results
     * @return
     * @deprecated It's difficult to determine this efficiently at runtime. The current implementation will no longer check the use case described above.
     */
    @Deprecated
    ExtensionResultStatusType filterResults(Class<?> type, Object testObject, List results);

}
