/*-
 * #%L
 * BroadleafCommerce Common Libraries
 * %%
 * Copyright (C) 2009 - 2024 Broadleaf Commerce
 * %%
 * Licensed under the Broadleaf Fair Use License Agreement, Version 1.0
 * (the "Fair Use License" located  at http://license.broadleafcommerce.org/fair_use_license-1.0.txt)
 * unless the restrictions on use therein are violated and require payment to Broadleaf in which case
 * the Broadleaf End User License Agreement (EULA), Version 1.1
 * (the "Commercial License" located at http://license.broadleafcommerce.org/commercial_license-1.1.txt)
 * shall apply.
 *
 * Alternatively, the Commercial License may be replaced with a mutually agreed upon license (the "Custom License")
 * between you and Broadleaf Commerce. You may not use this file except in compliance with the applicable license.
 * #L%
 */
package org.broadleafcommerce.common.dao;

import org.broadleafcommerce.common.persistence.EntityConfiguration;
import org.broadleafcommerce.common.service.PersistenceService;
import org.broadleafcommerce.common.util.HibernateUtils;
import org.broadleafcommerce.common.util.StreamCapableTransactionalOperationAdapter;
import org.broadleafcommerce.common.util.StreamingTransactionCapableUtil;
import org.broadleafcommerce.common.util.TransactionUtils;
import org.broadleafcommerce.common.util.dao.DynamicDaoHelperImpl;
import org.broadleafcommerce.common.util.dao.TypedQueryBuilder;
import org.hibernate.FlushMode;
import org.hibernate.Session;
import org.hibernate.type.Type;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Repository;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.List;
import java.util.Map;

import jakarta.annotation.Resource;
import jakarta.persistence.EntityManager;
import jakarta.persistence.PersistenceContext;
import jakarta.persistence.TypedQuery;
import jakarta.persistence.criteria.CriteriaBuilder;
import jakarta.persistence.criteria.CriteriaQuery;
import jakarta.persistence.criteria.Root;

@Repository("blGenericEntityDao")
public class GenericEntityDaoImpl implements GenericEntityDao, ApplicationContextAware {

    private static ApplicationContext applicationContext;
    private static GenericEntityDaoImpl dao;
    @PersistenceContext(unitName = "blPU")
    protected EntityManager em;
    @Resource(name = "blPersistenceService")
    protected PersistenceService persistenceService;
    @Resource(name = "blEntityConfiguration")
    protected EntityConfiguration entityConfiguration;
    @Resource(name = "blStreamingTransactionCapableUtil")
    protected StreamingTransactionCapableUtil transactionUtil;
    protected DynamicDaoHelperImpl daoHelper = new DynamicDaoHelperImpl();

    public static GenericEntityDaoImpl getGenericEntityDao() {
        if (applicationContext == null) {
            return null;
        }
        if (dao == null) {
            dao = (GenericEntityDaoImpl) applicationContext.getBean("blGenericEntityDao");
        }
        return dao;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public <T> T readGenericEntity(Class<T> clazz, Object id) {
        clazz = (Class<T>) DynamicDaoHelperImpl.getNonProxyImplementationClassIfNecessary(clazz);
        //We need to get PU dynamically to handle cases when class is not in blPU, e.g ScheduledJobImpl
        EntityManager emForClass = getEntityManager(clazz);
        Map<String, Object> md = daoHelper.getIdMetadata(clazz, emForClass);
        Type type = (Type) md.get("type");
        Class<?> returnedClass = type.getReturnedClass();
        if (returnedClass.isAssignableFrom(Long.class)) {
            id = Long.parseLong(String.valueOf(id));
        } else if (returnedClass.isAssignableFrom(Integer.class)) {
            id = Integer.parseInt(String.valueOf(id));
        }

        return HibernateUtils.deproxy(emForClass.find(clazz, id));
    }

    @Override
    public <T> Long readCountGenericEntity(Class<T> clazz) {
        clazz = (Class<T>) DynamicDaoHelperImpl.getNonProxyImplementationClassIfNecessary(clazz);
        TypedQuery<Long> q = new TypedQueryBuilder<T>(clazz, "root").toCountQuery(em);
        return q.getSingleResult();
    }

    @Override
    public <T> List<T> readAllGenericEntity(Class<T> clazz, int limit, int offset) {
        clazz = (Class<T>) DynamicDaoHelperImpl.getNonProxyImplementationClassIfNecessary(clazz);
        TypedQuery<T> q = new TypedQueryBuilder<T>(clazz, "root").toQuery(em);
        q.setMaxResults(limit);
        q.setFirstResult(offset);
        return q.getResultList();
    }

    @Override
    public <T> List<T> readAllGenericEntity(Class<T> clazz) {
        clazz = (Class<T>) DynamicDaoHelperImpl.getNonProxyImplementationClassIfNecessary(clazz);
        TypedQuery<T> q = new TypedQueryBuilder<T>(clazz, "root").toQuery(em);
        return q.getResultList();
    }

    @Override
    public List<Long> readAllGenericEntityId(Class<?> clazz) {
        clazz = DynamicDaoHelperImpl.getNonProxyImplementationClassIfNecessary(clazz);
        CriteriaBuilder builder = em.getCriteriaBuilder();
        CriteriaQuery<Long> criteria = builder.createQuery(Long.class);
        Root root = criteria.from(clazz);
        criteria.select(root.get(getIdField(clazz).getName()).as(Long.class));
        criteria.orderBy(builder.asc(root.get(getIdField(clazz).getName())));

        return em.createQuery(criteria).getResultList();
    }

    @Override
    public Class<?> getImplClass(String className) {
        Class<?> clazz = null;
        try {
            clazz = entityConfiguration.lookupEntityClass(className);
        } catch (NoSuchBeanDefinitionException e) {
            //do nothing
        }
        if (clazz == null) {
            clazz = getCeilingImplClass(className);
        }
        return clazz;
    }

    @Override
    public Class<?> getCeilingImplClass(final String className) {
        final Class<?>[] clazz = new Class<?>[1];
        try {
            clazz[0] = Class.forName(className);
        } catch (ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
        //em.unwrap requires a transactional entity manager. We'll only take the hit to start a transaction here if one has not already been started.
        transactionUtil.runOptionalTransactionalOperation(new StreamCapableTransactionalOperationAdapter() {
            @Override
            public void execute() throws Throwable {
                Class<?>[] entitiesFromCeiling = daoHelper.getAllPolymorphicEntitiesFromCeiling(
                        clazz[0], true, true
                );
                if (entitiesFromCeiling == null || entitiesFromCeiling.length < 1) {
                    clazz[0] = DynamicDaoHelperImpl.getNonProxyImplementationClassIfNecessary(clazz[0]);
                    entitiesFromCeiling = daoHelper.getAllPolymorphicEntitiesFromCeiling(
                            clazz[0], true, true
                    );
                }
                if (entitiesFromCeiling == null || entitiesFromCeiling.length < 1) {
                    throw new IllegalArgumentException(
                            String.format("Unable to find ceiling implementation for the requested class name (%s)",
                                    className)
                    );
                }
                clazz[0] = entitiesFromCeiling[entitiesFromCeiling.length - 1];
            }
        }, RuntimeException.class, !TransactionUtils.isTransactionalEntityManager(em));
        return clazz[0];
    }

    @Override
    public Serializable getIdentifier(Object entity) {
        return daoHelper.getIdentifier(entity);
    }

    protected Field getIdField(Class<?> clazz) {
        return daoHelper.getIdField(clazz);
    }

    @Override
    public <T> T save(T object) {
        return em.merge(object);
    }

    @Override
    public void persist(Object object) {
        em.persist(object);
    }

    @Override
    public void remove(Object object) {
        em.remove(object);
    }

    @Override
    public void flush() {
        em.flush();
    }

    @Override
    public void clearAutoFlushMode() {
        em.unwrap(Session.class).setHibernateFlushMode(FlushMode.MANUAL);
    }

    @Override
    public void enableAutoFlushMode() {
        em.unwrap(Session.class).setHibernateFlushMode(FlushMode.AUTO);
    }

    @Override
    public void clear() {
        em.clear();
    }

    @Override
    public boolean sessionContains(Object object) {
        return em.contains(object);
    }

    @Override
    public boolean idAssigned(Object object) {
        return getIdentifier(object) != null;
    }

    @Override
    public EntityManager getEntityManager() {
        return em;
    }

    protected EntityManager getEntityManager(Class clazz) {
        return persistenceService.identifyEntityManager(clazz);
    }

}
