/*-
 * #%L
 * BroadleafCommerce Open Admin Platform
 * %%
 * Copyright (C) 2009 - 2024 Broadleaf Commerce
 * %%
 * Licensed under the Broadleaf Fair Use License Agreement, Version 1.0
 * (the "Fair Use License" located  at http://license.broadleafcommerce.org/fair_use_license-1.0.txt)
 * unless the restrictions on use therein are violated and require payment to Broadleaf in which case
 * the Broadleaf End User License Agreement (EULA), Version 1.1
 * (the "Commercial License" located at http://license.broadleafcommerce.org/commercial_license-1.1.txt)
 * shall apply.
 *
 * Alternatively, the Commercial License may be replaced with a mutually agreed upon license (the "Custom License")
 * between you and Broadleaf Commerce. You may not use this file except in compliance with the applicable license.
 * #L%
 */
package org.broadleafcommerce.openadmin.processor;

import org.broadleafcommerce.common.web.BroadleafRequestContext;
import org.broadleafcommerce.openadmin.server.security.domain.AdminSection;
import org.broadleafcommerce.presentation.condition.ConditionalOnTemplating;
import org.broadleafcommerce.presentation.dialect.AbstractBroadleafAttributeModifierProcessor;
import org.broadleafcommerce.presentation.dialect.BroadleafDialectPrefix;
import org.broadleafcommerce.presentation.model.BroadleafAttributeModifier;
import org.broadleafcommerce.presentation.model.BroadleafTemplateContext;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;

import jakarta.servlet.http.HttpServletRequest;

/**
 * A Thymeleaf processor that will generate the HREF of a given Admin Section.
 * This is useful in constructing the left navigation menu for the admin console.
 *
 * @author elbertbautista
 */
@Component("blAdminSectionHrefProcessor")
@ConditionalOnTemplating
public class AdminSectionHrefProcessor extends AbstractBroadleafAttributeModifierProcessor {

    protected static final String FOLDER_SUBSECTION = "/folder";

    @Override
    public String getName() {
        return "admin_section_href";
    }

    @Override
    public String getPrefix() {
        return BroadleafDialectPrefix.BLC_ADMIN;
    }

    @Override
    public int getPrecedence() {
        return 10002;
    }

    @Override
    public BroadleafAttributeModifier getModifiedAttributes(
            String tagName,
            Map<String, String> tagAttributes,
            String attributeName,
            String attributeValue,
            BroadleafTemplateContext context
    ) {
        String href = "#";

        AdminSection section = context.parseExpression(attributeValue);
        if (section != null) {
            HttpServletRequest request = BroadleafRequestContext.getBroadleafRequestContext().getRequest();

            final boolean noFolder = Boolean.parseBoolean(tagAttributes.get("data-no-folder"));
            final boolean requireFolder = Boolean.parseBoolean(tagAttributes.get("data-require-folder"));
            final boolean allowFolder = section.isFolderable();
            final boolean folderByDefault = section.isFolderedByDefault();

            final boolean shouldFolder;

            if (!allowFolder) {
                shouldFolder = false;
            } else {
                if (noFolder) {
                    shouldFolder = false;
                } else if (requireFolder) {
                    shouldFolder = true;
                } else {
                    // nothing specified, do default
                    shouldFolder = folderByDefault;
                }
            }

            final String folderPart = shouldFolder ? FOLDER_SUBSECTION : "";

            href = request.getContextPath() + folderPart + section.getUrl();
        }

        Map<String, String> attrs = new HashMap<>();
        attrs.put("href", href);
        return new BroadleafAttributeModifier(attrs);
    }

}
