/*-
 * #%L
 * BroadleafCommerce Open Admin Platform
 * %%
 * Copyright (C) 2009 - 2024 Broadleaf Commerce
 * %%
 * Licensed under the Broadleaf Fair Use License Agreement, Version 1.0
 * (the "Fair Use License" located  at http://license.broadleafcommerce.org/fair_use_license-1.0.txt)
 * unless the restrictions on use therein are violated and require payment to Broadleaf in which case
 * the Broadleaf End User License Agreement (EULA), Version 1.1
 * (the "Commercial License" located at http://license.broadleafcommerce.org/commercial_license-1.1.txt)
 * shall apply.
 *
 * Alternatively, the Commercial License may be replaced with a mutually agreed upon license (the "Custom License")
 * between you and Broadleaf Commerce. You may not use this file except in compliance with the applicable license.
 * #L%
 */
package org.broadleafcommerce.openadmin.processor;

import org.apache.commons.lang3.StringUtils;
import org.broadleafcommerce.openadmin.web.form.component.ListGrid;
import org.broadleafcommerce.openadmin.web.form.entity.Field;
import org.broadleafcommerce.presentation.condition.ConditionalOnTemplating;
import org.broadleafcommerce.presentation.dialect.AbstractBroadleafAttributeModifierProcessor;
import org.broadleafcommerce.presentation.dialect.BroadleafDialectPrefix;
import org.broadleafcommerce.presentation.model.BroadleafAttributeModifier;
import org.broadleafcommerce.presentation.model.BroadleafTemplateContext;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;

/**
 * A Thymeleaf processor that will generate the appropriate ID for a given admin component.
 *
 * @author Andre Azzolini (apazzolini)
 */
@Component("blAdminComponentIdProcessor")
@ConditionalOnTemplating
public class AdminComponentIdProcessor extends AbstractBroadleafAttributeModifierProcessor {

    @Override
    public String getName() {
        return "component_id";
    }

    @Override
    public String getPrefix() {
        return BroadleafDialectPrefix.BLC_ADMIN;
    }

    @Override
    public int getPrecedence() {
        return 10002;
    }

    @Override
    public BroadleafAttributeModifier getModifiedAttributes(
            String tagName,
            Map<String, String> tagAttributes,
            String attributeName,
            String attributeValue,
            BroadleafTemplateContext context
    ) {
        Object component = context.parseExpression(attributeValue);

        String fieldName = "";
        String id = "";

        if (component instanceof ListGrid) {
            ListGrid lg = (ListGrid) component;

            fieldName = "listGrid-" + lg.getListGridType();
            if (StringUtils.isNotBlank(lg.getSubCollectionFieldName())) {
                fieldName += "-" + lg.getSubCollectionFieldName();
            }
        } else if (component instanceof Field) {
            Field field = (Field) component;
            fieldName = "field-" + field.getName();
        }

        if (StringUtils.isNotBlank(fieldName)) {
            id = cleanCssIdString(fieldName);
        }

        Map<String, String> attrs = new HashMap<>();
        attrs.put("id", id);
        return new BroadleafAttributeModifier(attrs);
    }

    protected String cleanCssIdString(String in) {
        in = in.replaceAll("[^a-zA-Z0-9-]", "-");
        return in;
    }

}
