/*-
 * #%L
 * BroadleafCommerce CMS Module
 * %%
 * Copyright (C) 2009 - 2024 Broadleaf Commerce
 * %%
 * Licensed under the Broadleaf Fair Use License Agreement, Version 1.0
 * (the "Fair Use License" located  at http://license.broadleafcommerce.org/fair_use_license-1.0.txt)
 * unless the restrictions on use therein are violated and require payment to Broadleaf in which case
 * the Broadleaf End User License Agreement (EULA), Version 1.1
 * (the "Commercial License" located at http://license.broadleafcommerce.org/commercial_license-1.1.txt)
 * shall apply.
 *
 * Alternatively, the Commercial License may be replaced with a mutually agreed upon license (the "Custom License")
 * between you and Broadleaf Commerce. You may not use this file except in compliance with the applicable license.
 * #L%
 */
package org.broadleafcommerce.cms.page.service.type;

import org.broadleafcommerce.common.BroadleafEnumerationType;

import java.io.Serial;
import java.io.Serializable;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * An extendible enumeration of offer rule types.<BR>
 * REQUEST - indicates a rule based on the incoming http request.<BR>
 * TIME - indicates a rule based on {@link org.broadleafcommerce.common.TimeDTO time}<br>
 * PRODUCT - indicates a rule based on {@link org.broadleafcommerce.core.catalog.domain.Product product}
 * CUSTOMER - indicates a rule based on {@link org.broadleafcommerce.profile.core.domain}
 */
public class PageRuleType implements Serializable, BroadleafEnumerationType {

    @Serial
    private static final long serialVersionUID = 1L;
    private static final Map<String, PageRuleType> TYPES = new LinkedHashMap<>();

    public static final PageRuleType REQUEST = new PageRuleType("REQUEST", "Request");
    public static final PageRuleType TIME = new PageRuleType("TIME", "Time");
    public static final PageRuleType PRODUCT = new PageRuleType("PRODUCT", "Product");
    public static final PageRuleType CUSTOMER = new PageRuleType("CUSTOMER", "Customer");

    private String type;
    private String friendlyType;

    public PageRuleType() {
        //do nothing
    }

    /**
     * Initialize the type and friendlyType
     *
     * @param <code>type</code>
     * @param <code>friendlyType</code>
     */
    public PageRuleType(final String type, final String friendlyType) {
        this.friendlyType = friendlyType;
        setType(type);
    }

    /**
     * Allows translation from the passed in String to a <code>PageRuleType</code>
     *
     * @param type
     * @return The matching rule type
     */
    public static PageRuleType getInstance(final String type) {
        return TYPES.get(type);
    }

    /**
     * Gets the type
     *
     * @return
     */
    @Override
    public String getType() {
        return type;
    }

    /**
     * Sets the type
     *
     * @param type
     */
    public void setType(final String type) {
        this.type = type;
        if (!TYPES.containsKey(type)) {
            TYPES.put(type, this);
        }
    }

    /**
     * Gets the name of the type
     *
     * @return
     */
    @Override
    public String getFriendlyType() {
        return friendlyType;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((type == null) ? 0 : type.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!getClass().isAssignableFrom(obj.getClass())) {
            return false;
        }
        PageRuleType other = (PageRuleType) obj;
        if (type == null) {
            return other.type == null;
        } else
            return type.equals(other.type);
    }

}
