/*-
 * #%L
 * BroadleafCommerce Admin Module
 * %%
 * Copyright (C) 2009 - 2024 Broadleaf Commerce
 * %%
 * Licensed under the Broadleaf Fair Use License Agreement, Version 1.0
 * (the "Fair Use License" located  at http://license.broadleafcommerce.org/fair_use_license-1.0.txt)
 * unless the restrictions on use therein are violated and require payment to Broadleaf in which case
 * the Broadleaf End User License Agreement (EULA), Version 1.1
 * (the "Commercial License" located at http://license.broadleafcommerce.org/commercial_license-1.1.txt)
 * shall apply.
 *
 * Alternatively, the Commercial License may be replaced with a mutually agreed upon license (the "Custom License")
 * between you and Broadleaf Commerce. You may not use this file except in compliance with the applicable license.
 * #L%
 */
package org.broadleafcommerce.admin.server.service.handler;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.broadleafcommerce.common.exception.ServiceException;
import org.broadleafcommerce.common.presentation.client.PersistencePerspectiveItemType;
import org.broadleafcommerce.core.catalog.domain.ProductOptionValue;
import org.broadleafcommerce.core.catalog.domain.Sku;
import org.broadleafcommerce.core.catalog.domain.SkuBundleItem;
import org.broadleafcommerce.core.catalog.domain.SkuBundleItemImpl;
import org.broadleafcommerce.core.catalog.domain.SkuProductOptionValueXref;
import org.broadleafcommerce.openadmin.dto.ClassMetadata;
import org.broadleafcommerce.openadmin.dto.CriteriaTransferObject;
import org.broadleafcommerce.openadmin.dto.DynamicResultSet;
import org.broadleafcommerce.openadmin.dto.Entity;
import org.broadleafcommerce.openadmin.dto.FieldMetadata;
import org.broadleafcommerce.openadmin.dto.FilterAndSortCriteria;
import org.broadleafcommerce.openadmin.dto.ForeignKey;
import org.broadleafcommerce.openadmin.dto.MergedPropertyType;
import org.broadleafcommerce.openadmin.dto.PersistencePackage;
import org.broadleafcommerce.openadmin.dto.PersistencePerspective;
import org.broadleafcommerce.openadmin.dto.Property;
import org.broadleafcommerce.openadmin.server.dao.DynamicEntityDao;
import org.broadleafcommerce.openadmin.server.service.handler.CustomPersistenceHandlerAdapter;
import org.broadleafcommerce.openadmin.server.service.persistence.module.InspectHelper;
import org.broadleafcommerce.openadmin.server.service.persistence.module.RecordHelper;
import org.broadleafcommerce.openadmin.server.service.persistence.module.criteria.FilterMapping;
import org.springframework.stereotype.Component;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jakarta.annotation.Resource;

/**
 * Overridden to provide the option values field on the SkuBundleItem list
 *
 * @author Phillip Verheyden (phillipuniverse)
 */
@Component("blSkuBundleItemCustomPersistenceHandler")
public class SkuBundleItemCustomPersistenceHandler extends CustomPersistenceHandlerAdapter {

    private static final Log LOG = LogFactory.getLog(SkuBundleItemCustomPersistenceHandler.class);

    @Resource(name = "blSkuCustomPersistenceHandler")
    protected SkuCustomPersistenceHandler skuPersistenceHandler;

    @Override
    public Boolean canHandleInspect(PersistencePackage persistencePackage) {
        return canHandle(persistencePackage);
    }

    @Override
    public Boolean canHandleFetch(PersistencePackage persistencePackage) {
        return canHandle(persistencePackage);
    }

    protected Boolean canHandle(PersistencePackage persistencePackage) {
        String className = persistencePackage.getCeilingEntityFullyQualifiedClassname();
        try {
            return SkuBundleItem.class.isAssignableFrom(Class.forName(className));
        } catch (ClassNotFoundException e) {
            LOG.warn("Could not find the class " + className + ", skipping the inventory custom persistence handler");
            return false;
        }
    }

    @Override
    public DynamicResultSet inspect(
            PersistencePackage persistencePackage,
            DynamicEntityDao dynamicEntityDao,
            InspectHelper helper
    ) throws ServiceException {
        try {
            PersistencePerspective persistencePerspective = persistencePackage.getPersistencePerspective();
            Map<MergedPropertyType, Map<String, FieldMetadata>> allMergedProperties = new HashMap<>();

            //retrieve the default properties for Inventory
            Map<String, FieldMetadata> properties = helper.getSimpleMergedProperties(
                    SkuBundleItem.class.getName(), persistencePerspective
            );

            //add in the consolidated product options field
            FieldMetadata options = skuPersistenceHandler.createConsolidatedOptionField(SkuBundleItemImpl.class);
            options.setOrder(3);
            properties.put(SkuCustomPersistenceHandler.CONSOLIDATED_PRODUCT_OPTIONS_FIELD_NAME, options);

            allMergedProperties.put(MergedPropertyType.PRIMARY, properties);
            Class<?>[] entityClasses = dynamicEntityDao.getAllPolymorphicEntitiesFromCeiling(SkuBundleItem.class);
            ClassMetadata mergedMetadata = helper.buildClassMetadata(
                    entityClasses, persistencePackage, allMergedProperties
            );

            return new DynamicResultSet(mergedMetadata, null, null);

        } catch (Exception e) {
            String className = persistencePackage.getCeilingEntityFullyQualifiedClassname();
            ServiceException ex = new ServiceException("Unable to retrieve inspection results for " + className, e);
            LOG.error("Unable to retrieve inspection results for " + className, ex);
            throw ex;
        }
    }

    @Override
    public DynamicResultSet fetch(
            PersistencePackage persistencePackage,
            CriteriaTransferObject cto,
            DynamicEntityDao dynamicEntityDao,
            RecordHelper helper
    ) throws ServiceException {
        String ceilingEntityFullyQualifiedClassname = persistencePackage.getCeilingEntityFullyQualifiedClassname();
        try {
            PersistencePerspective persistencePerspective = persistencePackage.getPersistencePerspective();
            ForeignKey foreignKey = (ForeignKey) persistencePerspective.getPersistencePerspectiveItems()
                    .get(PersistencePerspectiveItemType.FOREIGNKEY);

            // sort it
            if (foreignKey != null && foreignKey.getSortField() != null) {
                FilterAndSortCriteria sortCriteria = cto.get(foreignKey.getSortField());
                sortCriteria.setSortAscending(foreignKey.getSortAscending());

            }
            //get the default properties from Inventory and its subclasses
            Map<String, FieldMetadata> originalProps = helper.getSimpleMergedProperties(
                    SkuBundleItem.class.getName(), persistencePerspective
            );

            //Pull back the Inventory based on the criteria from the client
            List<FilterMapping> filterMappings = helper.getFilterMappings(
                    persistencePerspective, cto, ceilingEntityFullyQualifiedClassname, originalProps
            );

            //attach the product option criteria
            skuPersistenceHandler.applyProductOptionValueCriteria(
                    filterMappings, cto, persistencePackage, "sku"
            );

            List<Serializable> records = helper.getPersistentRecords(
                    persistencePackage.getCeilingEntityFullyQualifiedClassname(),
                    filterMappings,
                    cto.getFirstResult(),
                    cto.getMaxResults()
            );
            //Convert Skus into the client-side Entity representation
            Entity[] payload = helper.getRecords(originalProps, records);

            int totalRecords = helper.getTotalRecords(
                    persistencePackage.getCeilingEntityFullyQualifiedClassname(), filterMappings
            );

            for (int i = 0; i < payload.length; i++) {
                Entity entity = payload[i];
                SkuBundleItem bundleItem = (SkuBundleItem) records.get(i);
                Sku bundleSku = bundleItem.getSku();
                entity.findProperty("sku").setDisplayValue(bundleSku.getName());
                List<ProductOptionValue> productOptionValues = new ArrayList<>();
                for (SkuProductOptionValueXref skuProductOptionValueXref : bundleSku.getProductOptionValueXrefs()) {
                    productOptionValues.add(skuProductOptionValueXref.getProductOptionValue());
                }
                Property optionsProperty = skuPersistenceHandler.getConsolidatedOptionProperty(productOptionValues);
                entity.addProperty(optionsProperty);
            }

            return new DynamicResultSet(null, payload, totalRecords);
        } catch (Exception e) {
            throw new ServiceException("There was a problem fetching inventory. See server logs for more details", e);
        }
    }

}
