#%%

import pandas as pd
import matplotlib.pyplot as plt
import tikzplotlib
import numpy as np

####################### Static ####################################

# Load the Excel file
df = pd.read_excel('waveGauge_1_wvG5_wv04.xlsx')

# Assign values to rho and g
rho = 1000  # density 
g = 9.8   # gravitatinoal acceleration
w = 0.4 # width of building

# rho * g * h * h * w 
stat_max = (df['Elevation_6 [m]']-0.75) * rho * g * (df['Elevation_6 [m]']-0.75) * w

# Multiply values in "Elevation_6 [m] or wg 6" column by rho and g
# Equivalent point load =  1/2 * rho * g * h * h * w
stat_distr = 0.5 * (df['Elevation_6 [m]']-0.75) * rho * g * ((df['Elevation_6 [m]']-0.75) * w)

# Convert the result to a Python array
F_stat_max = stat_max.to_numpy()
F_stat_distr = stat_distr.to_numpy()

# print(F_stat)

#%%

# # Get the number of elements in the array
# num_elements = len(elevation_6_array)

# print("Number of elements:", num_elements)


########################## Dynamics #######################################

# Load the CSV file using numpy
data = np.genfromtxt('wv0.4_PointsVelocity_Vel.m.csv', delimiter=';')

# Assign values to rho, Cd, and A
rho = 1000  # Example value, replace with your desired value
Cd = 1.25   # Example value, replace with your desired value
A = 0.20    # Example value, replace with your desired value

# Extract the relevant data from the array
heights = data[2, 2:].astype(float)  # Extract the height values from the second row
# print(heights)
vel_data = data[4:, 2:].astype(float)  # Select rows starting from the third row and columns starting from the third column

# Calculate the height range
height_range = heights[-1] - heights[0]
# print(height_range)

# Calculate the effective velocity
effective_velocity = np.trapz(vel_data, x=heights, axis=1) / height_range

# Calculate the squared average multiplied by the given factors
F_dyn = 0.5 * rho * Cd * A * (effective_velocity ** 2)


# print(F_Dyn)

# # Get the number of elements in the array
# num_elements = len(F_dyn)

# print("Number of elements:", num_elements)

################################### Total ###########################################

F_total_max = F_stat_max + F_dyn
F_total_distr = F_stat_distr + F_dyn
# print(F_total)

################################## Time #########################################

time_values = df['Time [s]']

time_values = time_values.to_numpy()

#%%
################################## SPH ########################################


# Read the csv file into a pandas dataframe
dfsph = pd.read_csv('forces_sph.csv', delimiter=';')

# Extract the columns you need
time = dfsph["Time [s]"].values
force_x = dfsph["ForceFluid.x [N]"].values


################################## Experimental ########################################

# Read the CSV file
data_exp = np.genfromtxt('press_exp.csv', delimiter=',')

# Extract the columns into separate numpy arrays
time_exp = data_exp[:, 0]
time_shift = time_exp - 53.9  # approximated to match sph
press_kpa = data_exp[:, 1]

force_exp = (press_kpa * 1000) * A

# Sort the time and force arrays based on the time values
sorted_indices = np.argsort(time_shift)
sorted_time_shift = time_shift[sorted_indices]

sorted_force_exp = force_exp[sorted_indices]



################################## ASCE ########################################

gamma_w = 9800 # unit weight of water N/m3
Cp = 1.6    # dynamic pressure coefficient
ds = df['Elevation_6 [m]'] - 0.75 # stillwater depth

Pmax = Cp*gamma_w*ds + 1.2*gamma_w*ds
F_asce = Pmax * A

# Assuming time and F_asce are pandas Series
time_array = time 
F_asce_array = F_asce.to_numpy()  # Convert F_asce to a NumPy array

# print(F_asce.shape)
# print(time.shape)
#%%


################################## Maximum Force Points ########################################

# Find the time point where the maximum force occurs for each case
max_force_time_exp = sorted_time_shift[np.argmax(sorted_force_exp)]
max_force_time_sph = time[np.argmax(force_x)]
max_force_time_asce = time_array[np.argmax(F_asce_array)]
max_force_time_analytical = time[np.argmax(F_total_distr)]

# Find the maximum force for each case
max_force_exp = np.max(sorted_force_exp)
max_force_sph = np.max(force_x)
max_force_asce = np.max(F_asce_array)
max_force_analytical = np.max(F_total_distr)


################################## PLot ########################################


# Create the plot

# plt.plot(time, F_total_max, label='Analytical - maximum static loading, mean velocity dynamic loading')

# plt.plot(sorted_time_shift, sorted_force_exp, label='Experimental' )
# color_exp = plt.gca().lines[0].get_color()
# plt.axhline(max_force_exp, color=color_exp, linestyle='--', label=f'Max Force Exp: {max_force_exp:.2f} N', xmin=0, xmax=max_force_time_exp / max(time))

# plt.plot(time, force_x, label='SPH')
# color_sph = plt.gca().lines[2].get_color()
# plt.axhline(max_force_sph, color=color_sph, linestyle='--', label=f'Max Force SPH: {max_force_sph:.2f} N', xmin=0, xmax=max_force_time_sph / max(time))

# plt.plot(time_array, F_asce_array, label='ASCE')
# color_asce = plt.gca().lines[4].get_color()
# plt.axhline(max_force_asce, color=color_asce, linestyle='--', label=f'Max Force ASCE: {max_force_asce:.2f} N', xmin=0, xmax=max_force_time_asce / max(time))

# # plt.plot(time, F_stat_distr, label='Static force (analytical)')
# # plt.plot(time, F_dyn, label='Dynamic force (analytical)')

# plt.plot(time, F_total_distr, label='Analytical')
# color_analytical = plt.gca().lines[6].get_color()
# plt.axhline(max_force_analytical, color=color_analytical, linestyle='--', label=f'Max Force Analytical: {max_force_analytical:.2f} N', xmin=0, xmax=max_force_time_analytical / max(time))



####
plt.plot(sorted_time_shift, sorted_force_exp, label='Experimental' )
plt.plot(time, force_x, label='SPH')
plt.plot(time_array, F_asce_array, label='ASCE')
plt.plot(time, F_total_distr, label='Analytical')


# Extract the colors of the lines for each case
color_exp = plt.gca().lines[0].get_color()
color_sph = plt.gca().lines[1].get_color()
color_asce = plt.gca().lines[2].get_color()
color_analytical = plt.gca().lines[3].get_color()

# Add dashed horizontal lines at the maximum force values for each case
plt.axhline(max_force_exp, color=color_exp, linestyle='--', xmin=0, xmax=max_force_time_exp / 18)
plt.axhline(max_force_sph, color=color_sph, linestyle='--', xmin=0, xmax=max_force_time_sph / 18)
plt.axhline(max_force_asce, color=color_asce, linestyle='--', xmin=0, xmax=max_force_time_asce / 18)
plt.axhline(max_force_analytical, color=color_analytical, linestyle='--', xmin=0, xmax=max_force_time_analytical / 18)




# Set the plot title and labels
# plt.title('Plot of Force vs. Time')
plt.xlabel('Time [s]')
plt.ylabel('Force [N]')

# Add a legend
plt.legend(loc='upper right')

# Set the x-axis limits from 5 to 12 seconds
plt.xlim(5, 12)

# Save the plot as a TikZ file
tikzplotlib.save('forces_compare.tex')

# Display the plot
plt.show()
# %%


# Calculate the errors
error_exp = 0  # The error of the experimental method will be zero since it's compared to itself
error_sph = ((max_force_exp - max_force_sph)/max_force_exp)*100
error_asce = ((max_force_exp - max_force_asce)/max_force_exp)*100
error_analytical = ((max_force_exp - max_force_analytical)/max_force_exp)*100

# Create a bar chart for the errors
methods = ['SPH', 'ASCE', 'Analytical']
errors = [error_sph, error_asce, error_analytical]

plt.bar(methods, errors)

# Set the plot title and labels
# plt.title('Comparison of Maximum Force Errors')
# plt.xlabel('Methods')
plt.ylabel('Error (%)')

tikzplotlib.save('forces_compare_error.tex')

# Display the plot
plt.show()





from scipy.interpolate import interp1d

# Create an interpolation function for the experimental data
interp_function = interp1d(sorted_time_shift, sorted_force_exp, kind='linear')

# Generate new time points for interpolation (e.g., 601 points between the minimum and maximum time)
new_time_points = np.linspace(sorted_time_shift.min(), sorted_time_shift.max(), 601)

# Interpolate the experimental data at the new time points
interpolated_force_exp = interp_function(new_time_points)

# Calculate the errors at each time step for all methods
error_sph_values = ((force_x - interpolated_force_exp) / interpolated_force_exp) * 100
error_asce_values = ((F_asce_array - interpolated_force_exp) / interpolated_force_exp) * 100
error_analytical_values = ((F_total_distr - interpolated_force_exp) / interpolated_force_exp) * 100

# Create a new figure for the error plot
plt.figure()

# Plot the error values over time for each method
plt.plot(new_time_points, error_sph_values, label='SPH')
plt.plot(time_array, error_asce_values, label='ASCE')
plt.plot(time, error_analytical_values, label='Analytical')

# Set the plot title and labels
plt.title('Percentage Error vs. Time')
plt.xlabel('Time [s]')
plt.ylabel('Percentage Error')

# Add a horizontal line at y=0 for reference (0% error line)
plt.axhline(0, color='gray', linestyle='--')

# Add a legend
plt.legend(loc='upper right')

# Save the error plot as a TikZ file
tikzplotlib.save('error_plot.tex')

# Display the error plot
plt.show()

