/*
 * Copyright ConsenSys Software Inc., 2022
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */

package tech.pegasys.teku.beaconrestapi.handlers.v1.validator;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.when;
import static tech.pegasys.teku.infrastructure.http.HttpStatusCodes.SC_BAD_REQUEST;
import static tech.pegasys.teku.infrastructure.http.HttpStatusCodes.SC_INTERNAL_SERVER_ERROR;
import static tech.pegasys.teku.infrastructure.http.HttpStatusCodes.SC_OK;
import static tech.pegasys.teku.infrastructure.restapi.MetadataTestUtil.getRequestBodyFromMetadata;
import static tech.pegasys.teku.infrastructure.restapi.MetadataTestUtil.verifyMetadataEmptyResponse;
import static tech.pegasys.teku.infrastructure.restapi.MetadataTestUtil.verifyMetadataErrorResponse;

import com.fasterxml.jackson.core.JsonProcessingException;
import java.io.IOException;
import java.util.List;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import tech.pegasys.teku.beaconrestapi.AbstractMigratedBeaconHandlerTest;
import tech.pegasys.teku.infrastructure.async.SafeFuture;
import tech.pegasys.teku.infrastructure.unsigned.UInt64;

class PostSubscribeToBeaconCommitteeSubnetTest extends AbstractMigratedBeaconHandlerTest {

  @BeforeEach
  void setup() {
    when(validatorDataProvider.subscribeToBeaconCommittee(any())).thenReturn(SafeFuture.COMPLETE);
    setHandler(new PostSubscribeToBeaconCommitteeSubnet(validatorDataProvider));
  }

  @Test
  public void shouldReturnSuccessWhenSubscriptionToBeaconCommitteeIsSuccessful()
      throws JsonProcessingException {
    final PostSubscribeToBeaconCommitteeSubnet.CommitteeSubscriptionData data =
        new PostSubscribeToBeaconCommitteeSubnet.CommitteeSubscriptionData(
            1, 1, UInt64.ONE, UInt64.ONE, false);

    request.setRequestBody(List.of(data));

    handler.handleRequest(request);

    assertThat(request.getResponseCode()).isEqualTo(SC_OK);
    assertThat(request.getResponseBody()).isNull();
  }

  @Test
  void shouldReadRequestBody() throws IOException {
    final String data =
        "[{\"validator_index\":\"1\",\"committee_index\":\"1\",\"committees_at_slot\":\"1\","
            + "\"slot\":\"1\",\"is_aggregator\":true}]";
    assertThat(getRequestBodyFromMetadata(handler, data))
        .isEqualTo(
            List.of(
                new PostSubscribeToBeaconCommitteeSubnet.CommitteeSubscriptionData(
                    1, 1, UInt64.ONE, UInt64.ONE, true)));
  }

  @Test
  void metadata_shouldHandle400() throws JsonProcessingException {
    verifyMetadataErrorResponse(handler, SC_BAD_REQUEST);
  }

  @Test
  void metadata_shouldHandle500() throws JsonProcessingException {
    verifyMetadataErrorResponse(handler, SC_INTERNAL_SERVER_ERROR);
  }

  @Test
  void metadata_shouldHandle200() {
    verifyMetadataEmptyResponse(handler, SC_OK);
  }
}
