#include "bucket_open_list.h"

#include "open_list.h"

#include "../option_parser.h"
#include "../plugin.h"

#include "../utils/memory.h"

#include <cassert>
#include <deque>
#include <limits>
#include <memory>
#include <vector>

using namespace std;


class ScalarEvaluator;


template<class Entry>
class BucketOpenList : public OpenList<Entry> {
    typedef deque<Entry> Bucket;
    vector<Bucket> buckets;
    mutable int lowest_bucket;
    int size;

    ScalarEvaluator *evaluator;

protected:
    virtual void do_insertion(EvaluationContext &eval_context,
                              const Entry &entry) override;

public:
    explicit BucketOpenList(const Options &opts);
    virtual ~BucketOpenList() override = default;

    virtual Entry remove_min(vector<ap_float> *key = nullptr) override;
    virtual bool empty() const override;
    virtual void clear() override;
    virtual void get_involved_heuristics(set<Heuristic *> &hset) override;
    virtual bool is_dead_end(
        EvaluationContext &eval_context) const override;
    virtual bool is_reliable_dead_end(
        EvaluationContext &eval_context) const override;
};


template<class Entry>
BucketOpenList<Entry>::BucketOpenList(const Options &opts)
    : OpenList<Entry>(opts.get<bool>("pref_only")),
      lowest_bucket(numeric_limits<ap_float>::max()),
      size(0),
      evaluator(opts.get<ScalarEvaluator *>("eval")) {
}

template<class Entry>
void BucketOpenList<Entry>::do_insertion(
    EvaluationContext &eval_context, const Entry &entry) {
    ap_float key = eval_context.get_heuristic_value(evaluator);
    assert(key >= 0);
    int num_buckets = buckets.size();
    if (key >= num_buckets)
        buckets.resize(key + 1);
    if (key < lowest_bucket)
        lowest_bucket = key;
    buckets[key].push_back(entry);
    ++size;
}

template<class Entry>
Entry BucketOpenList<Entry>::remove_min(vector<ap_float> *key) {
    assert(size > 0);
    while (buckets[lowest_bucket].empty())
        ++lowest_bucket;
    --size;
    if (key) {
        assert(key->empty());
        key->push_back(lowest_bucket);
    }
    Entry result = buckets[lowest_bucket].front();
    buckets[lowest_bucket].pop_front();
    return result;
}

template<class Entry>
bool BucketOpenList<Entry>::empty() const {
    return size == 0;
}

template<class Entry>
void BucketOpenList<Entry>::clear() {
    buckets.clear();
    lowest_bucket = numeric_limits<ap_float>::max();
    size = 0;
}

template<class Entry>
void BucketOpenList<Entry>::get_involved_heuristics(
    set<Heuristic *> &hset) {
    evaluator->get_involved_heuristics(hset);
}

template<class Entry>
bool BucketOpenList<Entry>::is_dead_end(
    EvaluationContext &eval_context) const {
    return eval_context.is_heuristic_infinite(evaluator);
}

template<class Entry>
bool BucketOpenList<Entry>::is_reliable_dead_end(
    EvaluationContext &eval_context) const {
    return is_dead_end(eval_context) && evaluator->dead_ends_are_reliable();
}

BucketOpenListFactory::BucketOpenListFactory(
    const Options &options)
    : options(options) {
}

unique_ptr<StateOpenList>
BucketOpenListFactory::create_state_open_list() {
    return utils::make_unique_ptr<BucketOpenList<StateOpenListEntry>>(options);
}

unique_ptr<EdgeOpenList>
BucketOpenListFactory::create_edge_open_list() {
    return utils::make_unique_ptr<BucketOpenList<EdgeOpenListEntry>>(options);
}

static shared_ptr<OpenListFactory> _parse(OptionParser &parser) {
    parser.document_synopsis(
        "Bucket-based open list",
        "Bucket-based open list implementation that uses a single evaluator. "
        "Ties are broken in FIFO order.");
    parser.add_option<ScalarEvaluator *>("eval", "scalar evaluator");
    parser.add_option<bool>(
        "pref_only",
        "insert only nodes generated by preferred operators",
        "false");

    Options opts = parser.parse();
    if (parser.dry_run())
        return nullptr;
    else
        return make_shared<BucketOpenListFactory>(opts);
}

static PluginShared<OpenListFactory> _plugin("single_buckets", _parse);
