'''Create example NetCDF file from an input ASCII file
in spherical coordinate system.
The columns in the ASCII file are formatted
based on the convention for spherical coordinates in ASPECT,
i.e., 'r', 'phi', 'theta' in 3D in the first three columns,
followed by the data.

see https://unidata.github.io/netcdf4-python/ for more information

These are also used in the ASPECT unit tests located in
./unit_tests/netcdf.cc to check that we can load files correctly.

'''

import netCDF4 as nc
import numpy as np

def ascii_to_netcdf(filename):
    
    # input file with ASCII data such that the first three
    # columns represent r, phi, and theta, respectively.
    ascii_input = np.loadtxt(filename)
    print (ascii_input.shape)
    
    ds = nc.Dataset('test-3d-spherical.nc','w', format='NETCDF4') 
    
    # Write metadata attributes
    ds.title = 'Spherical 3d example file'
    ds.source = ('generated by an example python script as part of github.com/geodynamics/aspect')
    
    print(f'\n\nInfo:\n{ds}')
    
    n_r     = len(np.unique(ascii_input[:, 0]))
    n_phi   = len(np.unique(ascii_input[:, 1]))
    n_theta = len(np.unique(ascii_input[:, 2]))
    
    print (n_r, n_phi, n_theta)

    # Create dimensions.
    dims = [ds.createDimension('radius', n_r),
            ds.createDimension('longitude', n_phi),
            ds.createDimension('latitude',  n_theta)]

    # Create variables for each of the data columns.
    radius           = ds.createVariable('radius', np.float64, ('radius'), fill_value=np.nan)
    radius.units     = 'meters'
    longitudes       = ds.createVariable('longitude', np.float64, ('longitude'), fill_value=np.nan)
    longitudes.units = 'degrees_east'
    latitudes        = ds.createVariable('latitude', np.float64, ('latitude'), fill_value=np.nan)
    latitudes.units  = 'degrees_north'
    
    # Create the data field variable. 
    # Note that the ordering looks wrong, but it is consistent with how the data needs to be
    # reshaped for reading into Paraview
    vel         = ds.createVariable('vs_anomaly', np.float64,('latitude', 'longitude', 'radius'), fill_value=np.nan)
    vel.units   = 'percent'
    
    # Fill the coordinate and field data. We convert phi and theta from spherical coordinate
    # system to geographical
    radius[:]     = np.unique(ascii_input[:, 0])
    longitudes[:] = np.unique(ascii_input[:, 1])
    latitudes[:]  = np.unique(ascii_input[:, 2])
    
    vel[:]        = np.reshape(ascii_input[:, 3], (n_theta, n_phi, n_r))
     
    # Finish and close file
    ds.close()
   
   
ascii_to_netcdf("netcdf_test.txt")
