/*
  Copyright (C) 2023 - 2024 by the authors of the ASPECT code.

  This file is part of ASPECT.

  ASPECT is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  ASPECT is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with ASPECT; see the file doc/COPYING.  If not see
  <http://www.gnu.org/licenses/>.
*/


#ifndef _aspect_cookbooks_tomography_based_plate_motions_h
#define _aspect_cookbooks_tomography_based_plate_motions_h

#include <aspect/material_model/interface.h>
#include <aspect/lateral_averaging.h>
#include <aspect/material_model/grain_size.h>
#include <aspect/simulator_access.h>
#include <aspect/material_model/rheology/ascii_depth_profile.h>
#include <array>

namespace aspect
{
  namespace MaterialModel
  {
    using namespace dealii;


    /**
     * A material model to set up mantle flow models based on
     * an input tomography model and a temperature model.
     * The viscosity is computed based on dislocation and diffusion
     * creep with lateral averages scaled to a reference viscosity
     * profile.
     * This material model only works if a compositional field
     * named 'grain_size' is present. In the diffusion
     * creep regime, the viscosity depends on this grain size.
     * The model allows the user to prescribe different viscosity at
     * fault boundaries and in continental regions at locations defined
     * by the the compositional field 'faults' or 'cratons', respectively.
     * The material model also allows the option to compute and use
     * equilibrium grain size following paleowattmeter approximation
     * by Austin and Evans, (2007), Paleowattmeters: "A scaling relation for
     * dynamically recrystallized grain size", Geology, 35, 343.
     * Other material parameters are either based on (1) an input tomography
     * model and scaling profiles between temperature/density relative to
     * the seismic velocity anomalies or a temperature model and constant
     * thermal expansivities and compressibilities relative to a reference
     * temperature, as done in Osei Tutu et al., (2018): "Evaluating the influence
     * of plate boundary friction and mantle viscosity on plate velocities",
     * Geochem. Geophys. Geosyst., 19 (3), 642–666, or (2) prescribed similar to
     * the 'simple' material model, or (3) read from data files
     * that were generated by the Perplex or Hefesto software. The material model
     * is described in more detail in Dannberg, J., Z. Eilon, U. Faul,
     * R. Gassmöller, P. Moulik, and R. Myhill (2017): "The importance of
     * grain size to mantle dynamics and seismological observations",
     * Geochem. Geophys. Geosyst., 18, 3034–3061, doi:10.1002/2017GC006944.,
     * and in Saxena, A., Dannberg J., Gassmöller, Fraters, M., Heister, T., & Styron, (2023):
     * "High-resolution mantle flow models reveal importance of plate boundary geometry and slab pull
     * forces on generating tectonic plate motions", J. Geophys. Research.:Solid Earth,
     * 128, e2022JB025877.
     *
     * @ingroup MaterialModels
     */
    template <int dim>
    class TomographyBasedPlateMotions : public MaterialModel::Interface<dim>, public ::aspect::SimulatorAccess<dim>
    {
      public:
        /**
         * Initialization function. Loads the material data and sets up
         * pointers. Gets the reference viscosity profile from a file.
         */
        void
        initialize () override;

        /**
         * Compute the laterally averaged viscosity
         */
        void
        update() override;

        /**
         * Compute the reference viscosity against which the viscosities in the model
         * are scaled. The reference viscosity is from a depth-dependent ascii profile and
         * the function returns as the second parameter in the pair the support point in
         * that profile that has the next smaller depth value than the input parameter
         * @p depth.
         */
        std::pair<double, unsigned int>
        get_reference_viscosity (const double depth) const;

        /**
         * Return the depth of the base of the uppermost mantle. Below that depth,
         * material properties are based on seismic tomography. Above that depth, material
         * properties are computed based on the model of Tutu et al., 2018.
         */
        double
        get_depth_of_base_of_uppermost_mantle () const;

        /**
         * Compute the scaling factors for each depth layer such that the laterally
         * averaged viscosiy in that layer is the same as the reference vicosity.
         */
        double
        compute_viscosity_scaling (const double depth) const;

        /**
         * Return whether the model is compressible or not.
         */
        bool
        is_compressible () const override;

        void
        evaluate(const typename Interface<dim>::MaterialModelInputs &in,
                 typename Interface<dim>::MaterialModelOutputs &out) const override;

        /**
         * @name Functions used in dealing with run-time parameters
         * @{
         */
        /**
         * Declare the parameters this class takes through input files.
         */
        static
        void
        declare_parameters (ParameterHandler &prm);

        /**
         * Read the parameters this class declares from the parameter file.
         */
        void
        parse_parameters (ParameterHandler &prm) override;

        void
        create_additional_named_outputs (MaterialModel::MaterialModelOutputs<dim> &out) const override;
        /**
         * @}
         */

      protected:
        double reference_rho;
        double thermal_alpha;
        double reference_specific_heat;

        /**
         * The constant compressibility.
         */
        double reference_compressibility;

        /**
         * The thermal conductivity.
         */
        double k_value;

        /**
         * Parameters controlling the grain size evolution.
         */
        std::vector<double> grain_growth_activation_energy;
        std::vector<double> grain_growth_activation_volume;
        std::vector<double> grain_growth_rate_constant;
        std::vector<double> grain_growth_exponent;
        double              minimum_grain_size;
        std::vector<double> reciprocal_required_strain;
        std::vector<double> recrystallized_grain_size;
        bool                equilibrate_grain_size;

        /**
         * Parameters controlling the dynamic grain recrystallization.
         * (following paleowattmeter as described in Austin, N. J., & Evans, B.
         * (2007). Paleowattmeters: A scaling relation for dynamically
         * recrystallized grain size. Geology, 35(4), 343-346.).
         */
        std::vector<double> grain_boundary_energy;
        std::vector<double> boundary_area_change_work_fraction;
        std::vector<double> geometric_constant;

        /**
         * Parameters controlling the viscosity.
         */
        double dislocation_viscosity_iteration_threshold;
        unsigned int dislocation_viscosity_iteration_number;
        std::vector<double> dislocation_creep_exponent;
        std::vector<double> dislocation_activation_energy;
        std::vector<double> dislocation_activation_volume;
        std::vector<double> dislocation_creep_prefactor;
        std::vector<double> diffusion_creep_exponent;
        std::vector<double> diffusion_activation_energy;
        std::vector<double> diffusion_activation_volume;
        std::vector<double> diffusion_creep_prefactor;
        std::vector<double> diffusion_creep_grain_size_exponent;

        /**
         * Reference viscosity profile coordinates, and the corresponding viscosity.
         */
        std::vector<double> reference_viscosity_coordinates;
        std::unique_ptr<Rheology::AsciiDepthProfile<dim>> reference_viscosity_profile;

        /**
         * A reference profile for density scaling.
         */
        Utilities::AsciiDataProfile<dim> rho_vs_depth_profile;

        /**
         * The column indices of the density scaling column in the ascii profile file.
         */
        unsigned int density_scaling_index;


        /**
         * A reference profile for the thermal expansivity.
         */
        Utilities::AsciiDataProfile<dim> thermal_expansivity_profile;
        unsigned int thermal_expansivity_column_index;

        /**
         * A reference profile for temperatura scaling. Values are from Steinberger and
         * Calderwood, 2006.
         */
        Utilities::AsciiDataProfile<dim> dT_vs_depth_profile;

        /**
         * The column indices of the density scaling column in the ascii profile file.
         */
        unsigned int temperature_scaling_index;

        /**
         * An object of ascii data boundary to input crustal depths.
         */
        Utilities::AsciiDataBoundary<dim> crustal_boundary_depth;

        /**
         * A depth-profile of the laterally averaged viscosity in each layer
         * in the current model. Can be used to compare (and potentially scale)
         * the computed viscosity to the reference profile.
         */
        std::vector<double> average_viscosity_profile;

        /**
         * Variable returned  to determine if the evaluate () function is called and
         * viscosities are computed. Initially, it is set to false and then updated
         * to true in the update () function.
         */
        bool initialized;

        /**
         * Because of the nonlinear nature of this material model many
         * parameters need to be kept within bounds to ensure stability of the
         * solution. These bounds can be adjusted as input parameters.
         */
        double max_temperature_dependence_of_eta;
        double min_eta;
        double max_eta;
        double min_specific_heat;
        double max_specific_heat;
        double min_thermal_expansivity;
        double max_thermal_expansivity;
        unsigned int max_latent_heat_substeps;
        double min_grain_size;

        double diffusion_viscosity (const double      temperature,
                                    const double      pressure,
                                    const std::vector<double>    &compositional_fields,
                                    const SymmetricTensor<2,dim> &strain_rate,
                                    const Point<dim> &position) const;

        /**
         * Compute the equilibrium grain size for a given temperature and
         * pressure.
         * This computation is based on the rate of grain size growth
         * (Ostwald ripening) or reduction (due to dynamic recrystallization
         * and phase transformations) in dependence on temperature, pressure,
         * strain rate, mineral phase and creep regime.
         * We use the grain size growth laws as for example described
         * in Behn, M. D., Hirth, G., & Elsenbeck, J. R. (2009). Implications
         * of grain size evolution on the seismic structure of the oceanic
         * upper mantle. Earth and Planetary Science Letters, 282(1), 178-189.
         *
         * For the rate of grain size reduction due to dynamic crystallization,
         * we use paleowattmeter approximation (Austins and
         * Evans, 2007).
         */
        void compute_equilibrium_grain_size (const typename Interface<dim>::MaterialModelInputs &in,
                                             typename Interface<dim>::MaterialModelOutputs      &out) const;

        /**
         * This function calculates the dislocation viscosity. For this purpose
         * we need the dislocation component of the strain rate, which we can
         * only compute by knowing the dislocation viscosity. Therefore, we
         * iteratively solve for the dislocation viscosity and update the
         * dislocation strain rate in each iteration using the new value
         * obtained for the dislocation viscosity. The iteration is started
         * with a dislocation viscosity calculated for the whole strain rate
         * unless a guess for the viscosity is provided, which can reduce the
         * number of iterations significantly.
         */
        double dislocation_viscosity (const double                  temperature,
                                      const double                  pressure,
                                      const std::vector<double>    &compositional_fields,
                                      const SymmetricTensor<2,dim> &strain_rate,
                                      const Point<dim>             &position,
                                      const double                  viscosity_guess = 0) const;

        /**
         * This function calculates the dislocation viscosity for a given
         * dislocation strain rate.
         */
        double dislocation_viscosity_fixed_strain_rate (const double      temperature,
                                                        const double      pressure,
                                                        const double      second_strain_rate_invariant,
                                                        const Point<dim> &position) const;

        double density (const double temperature,
                        const double pressure,
                        const std::vector<double> &compositional_fields,
                        const Point<dim> &position) const;

        /**
         * The column indices of the density column in the reference profile file.
         */
        unsigned int density_index;

        double compressibility (const double temperature,
                                const double pressure,
                                const std::vector<double> &compositional_fields,
                                const Point<dim> &position) const;

        double thermal_expansivity (const double temperature,
                                    const double pressure,
                                    const std::vector<double> &compositional_fields,
                                    const Point<dim> &position) const;

        /**
         * Returns the p-wave velocity as calculated by HeFESTo.
         */
        double seismic_Vp (const double      temperature,
                           const double      pressure,
                           const std::vector<double> &compositional_fields,
                           const Point<dim> &position) const;

        /**
         * Returns the s-wave velocity as calculated by HeFESTo.
         */
        double seismic_Vs (const double      temperature,
                           const double      pressure,
                           const std::vector<double> &compositional_fields,
                           const Point<dim> &position) const;

        /**
         * Function that defines the phase transition interface
         * (0 above, 1 below the phase transition). This is done
         * individually for each transition and summed up in the end.
         */
        double
        phase_function (const Point<dim> &position,
                        const double temperature,
                        const double pressure,
                        const unsigned int phase) const;

        /**
         * Function that returns the phase for a given
         * position, temperature, pressure and compositional
         * field index.
         */
        unsigned int
        get_phase_index (const Point<dim> &position,
                         const double temperature,
                         const double pressure) const;


        /**
         * Lists of depth, width, and Clapeyron slopes for the different phase
         * transitions and in which phase they occur.
         */
        std::vector<double> transition_depths;
        std::vector<double> transition_temperatures;
        std::vector<double> transition_slopes;
        std::vector<double> transition_widths;


        /**
         * The following variables are properties of the material files
         * we read in.
         */
        std::string data_directory;
        std::vector<std::string> material_file_names;
        std::vector<std::string> derivatives_file_names;
        unsigned int n_material_data;
        bool use_table_properties;

        /**
         * Parameter value that determines whether to read the viscosity with depth
         * from an ascii data file.
         */
        bool use_depth_dependent_viscosity;

        /**
         * Parameter value that determines whether to read the density scaling with depth
         * from an ascii data file.
         */
        bool use_depth_dependent_rho_vs;

        /**
         * Parameter value that determines whether to read the thermal expansivity
         * from an ascii data file.
         */
        bool use_depth_dependent_thermal_expansivity;

        /**
         * Parameter value that determines whether to read the temperature scaling with depth
         * from an ascii data file.
         */
        bool use_depth_dependent_dT_vs;

        /**
         * Parameter that determines if faults or plate boundaries are used as another
         * compositional field.
         */
        bool use_faults;

        /**
         * Parameter that determines the viscosity of faults or plate boundaries.
         */
        double fault_viscosity;

        /**
         * Parameter that determines if ridges or trenches have different viscosities
         */
        bool use_varying_fault_viscosity;

        /**
         * Parameter that determines if lithospheric depths vary or not
         */
        bool use_constant_lithosphere_thickness;

        /**
         * Parameter that determines the viscosity of faults at ridges.
         */
        double ridge_viscosity;

        /**
         * Parameter that determines the viscosity of faults at trenches.
         */
        double trench_viscosity;

        /**
         * Parameter that determines the asthenosphere viscosity. By default, use
         * the value from the Steinberger & Calderwood reference viscosity profile.
         */
        double asthenosphere_viscosity;

        /**
         * Parameter that determines if we want to use viscous and neutrally buoyant cratons as
         * another compositional field.
         */
        bool use_cratons;

        /**
         * Parameter that determines the viscosity for cratons.
         */
        double craton_viscosity;

        /**
         * Approximate lithosphere thickness used to separate the regions of
         * temperature derived from seismic tomography and linear temperature
         * gradient.
         */
        double lithosphere_thickness;

        /**
         * Parameter used to describe the uppermost mantle based on Tutu (2018).
         */
        double depth_to_base_of_uppermost_mantle;

        /**
         * The format of the provided material files. Currently we support
         * the PERPLEX and HeFESTo data formats.
         */
        enum formats
        {
          perplex,
          hefesto
        } material_file_format;

        /**
         * List of pointers to objects that read and process data we get from
         * material data files. There is one pointer/object per compositional
         * field provided.
         */
        std::vector<std::shared_ptr<MaterialModel::MaterialUtilities::Lookup::MaterialLookup>> material_lookup;

        /**
         * A shared pointer to the initial temperature object
         * that ensures that the current object can continue
         * to access the initial composition object beyond the
         * first time step.
         */
        std::shared_ptr<const aspect::InitialTemperature::Manager<dim>> initial_temperature_manager;
    };

  }
}

#endif
