#!/usr/bin/python3

# This Python script reads in a parameters.json file as generated by
# ASPECT that contains information about all possible parameters and
# converts them to Markdown files (stored under ./doc/sphinx/parameters/)
#
# Note: You do not need to use this script directly. Call
# "./update_parameters.sh" in the ./doc/ directory instead.
#
# We require the .json file to be generated by deal.II 9.4 or
# newer. Somewhere between 9.3 and 9.4 the content in the json file
# changed (and names are no longer mangled()).

import json
import re
from sys import argv, exit, stderr


def mangle(text):
    """
    Mangle the special characters in the given string as _XY where XY
    is the ASCII code encoded in base64.

    Reimplementation of mangle() in deal.II's parameter_handler.cc

    """
    output = ""
    lookup = ['0','1','2','3','4','5','6','7','8','9','a','b','c','d','e','f']
    for t in text:
        if (t >= 'a' and t <= 'z') or (t>= 'A' and t <= 'Z') or (t>= '0' and t <= '9'):
            output += t
        else:
            output += '_'
            c = ord(t)
            assert(c<256)
            output += lookup[c // 16]
            output += lookup[c % 16]
    return output

def demangle(text):
    """
    The reverse operation of mangle().

    Reimplementation of demangle() in deal.II's parameter_handler.cc

    """
    output = ""
    lookup = ['0','1','2','3','4','5','6','7','8','9','a','b','c','d','e','f']
    step = 0
    c = 0
    for t in text:
        if step == 0:
            if t=='_':
                step = 1
            else:
                output += t
        elif step == 1:
            c = lookup.index(t)
            step = 2
        elif step == 2:
            c = c*16 + lookup.index(t)
            output += chr(c)
            step = 0        
    return output

def test_mangling():
    assert(mangle("aBD403") == "aBD403")
    assert(mangle("dashdash--space end") == "dashdash_2d_2dspace_20end")
    assert(mangle("underscore_end") == "underscore_5fend")
    strings = ["underscore_end", "sadflhsdgkj", "- _#$%", "_20__20_"]
    for s in strings:
        assert(demangle(mangle(s)) == s)
    
    
# Our doc strings are written for latex, but here we want to output them
# in markdown. Escape some common latex-isms.
def escape_doc_string(text) :
    # First some simple text replacements:
    tmp = (text
           .replace("``", "&ldquo;")
           .replace("''", "&rdquo;")
           .replace("\\`", "[[[backtick]]]") # temporarily replace escaped backticks by a placeholder
           .replace("`", "&lsquo;")
           .replace("'", "&rsquo;")
           .replace("[[[backtick]]]", "`") # and now change them to actual backticks
           .replace("\\aspect{}", "ASPECT")
           .replace("\\dealii{}", "deal.II")
    )

    # Then also deal with italics and boldface and typewriter:
    tmp = re.sub(r'\\textit\{(.*?)\}',
                 r'*\1*',
                 tmp)
    tmp = re.sub(r'\\textbf\{(.*?)\}',
                 r'**\1**',
                 tmp)
    tmp = re.sub(r'\\texttt\{(.*?)\}',
                 r'`\1`',
                 tmp)
    tmp = re.sub(r'\\cite\{(.*?)\}',
                 r'{cite}`\1`',
                 tmp)
    tmp = re.sub(r'Section~\\ref\{(.*?)\}',
                 r'{ref}`\1`',
                 tmp)

    # Finally escape some characters that have special meaning in markdown:
    tmp = re.sub(r'\[(.*)\]\(',
                 r'\[\1\](',
                 tmp)

    return tmp;




def print_if_parameter_or_alias(entry, path_str, true_name, cur_path, output_file) :
    if "value" in entry:
        # this is a parameter
        print("(parameters:" + path_str + ")=", file=output_file)
        print("### __Parameter name:__ " + true_name, file=output_file)
        print("**Default value:** " +  entry["default_value"] + " \n", file=output_file)
        print("**Pattern:** " +  entry["pattern_description"] + " \n", file=output_file)
        print("**Documentation:** " + escape_doc_string(entry["documentation"]) + " \n", file=output_file)
    elif "alias" in entry:
        # This is an alias for a parameter
        print("(parameters:" + path_str + ")=", file=output_file)
        aliased_name = entry["alias"]
        alias_path_str = "/".join(cur_path) + "/" + mangle(aliased_name)
        print("### __Parameter name__: " +  true_name, file=output_file)
        print("**Alias:** [" + aliased_name + "](parameters:" + alias_path_str + ")\n", file=output_file)
        print("**Deprecation Status:** " + entry["deprecation_status"] + "\n", file=output_file)


def handle_subsection(data, cur_path, output_file):
    keys = list(data.keys())
    keys.sort()
    # Since there is no end to a markdown heading besides another markdown
    # heading of equal or greater size, we need to first do parameters and
    # aliases, and then do subsections otherwise parameters get nested
    # incorrectly
    for key in keys:
        path_str = "/".join(cur_path) + "/"  + mangle(key)
        true_name = demangle(key)

        print_if_parameter_or_alias(data[key], path_str, true_name, cur_path, output_file)

    for key in keys:
        path_str = "/".join(cur_path) + "/"  + mangle(key)

        entry = data[key]

        if (not "value" in entry) and (not "alias" in entry):
            # This is a subsection
            print("(parameters:" + path_str + ")=", file=output_file)
            new_path = cur_path + [mangle(key)]
            section_name = demangle(path_str)
            section_name = section_name.replace("/"," / ")
            print("## **Subsection:** " + section_name, file=output_file)
            handle_subsection(entry, new_path, output_file)

def handle_parameters(data):
    global_output_file = open("doc/sphinx/parameters/index.md", "w")
    global_output_file.write("(parameters)=\n"
                             "# Parameter Documentation\n"
                             "\n"
                             ":::{toctree}\n"
                             "---\n"
                             "maxdepth: 2\n"
                             "---\n"
                             "global.md\n")
    global_parameters = open("doc/sphinx/parameters/global.md", "w")
    print("(parameters:global)=", file=global_parameters)
    print("# Global parameters\n\n", file=global_parameters)
    print("## **Subsection:** No subsection\n\n", file=global_parameters)

    cur_path = []
    keys = list(data.keys())
    keys.sort()
    # Since there is no end to a markdown heading besides another markdown
    # heading of equal or greater size, we need to first do parameters and
    # aliases, and then do subsections otherwise parameters get nested
    # incorrectly
    for key in keys:
        path_str = mangle(key)
        true_name = key

        print_if_parameter_or_alias(data[key], path_str, true_name, cur_path, global_parameters)

    for key in keys:
        path_str = mangle(key)
        entry = data[key]
        if (not "value" in entry) and (not "alias" in entry):
            # This is a subsection
            # Add this to the toctree
            print(path_str + ".md", file=global_output_file)
            # Create the new file and write to it now
            subfile = open("doc/sphinx/parameters/" + path_str + ".md", "w")
            print("(parameters:" + path_str + ")=", file=subfile)
            new_path = cur_path + [path_str]
            section_name = demangle(path_str).replace(":", "/")
            print("# " + section_name + "\n\n", file=subfile)
            print("## **Subsection:** " + section_name + "\n\n", file=subfile)
            handle_subsection(entry, new_path, subfile)
            subfile.close()
    global_output_file.write(":::\n")
    global_output_file.close()

if __name__  == "__main__":
    test_mangling()
    
    if (len(argv) !=2):
        exit("ERROR: Please specify the parameters.json file as an input.")

    input_file = open(argv[1], "r")
    data = json.load(input_file)
    handle_parameters(data)
    exit(0)
