import numpy as np
import sys 

###################################################################################################
# This python script reads in an ascii file generated by the ASPECT gravity postprocessor and
# generates a vtu file of the measuring points with the gravity vector (gx,gy,gz) and potential U.
# 
# How to use it:
#
# > python3 gravity_ascii_to_vtu_map.py file_name nlon nlat
#
# If no file_name is provided this scripts uses gravity-00000 by default.
# The generated vtu file is named: file_name_map.vtu
#
# For questions/remarks: c.thieulot@uu.nl
###################################################################################################

if int(len(sys.argv) == 4): 
   filename = sys.argv[1]
   nlon = int(sys.argv[2])
   nlat = int(sys.argv[3])
else:
   print("*****Error*****")
   print("please provide file_name followed by nlon and nlat as follows:")
   print("> python3 gravity_ascii_to_vtu_map.py file_name nlon nlat")
   exit()
   
print('processing file:',filename)

###################################################################################################
# generating connectivity array for vtu file
###################################################################################################

nnp = nlon * nlat
nel = (nlon - 1) * (nlat - 1)

icon = np.empty((4, nel), dtype=np.int32)
counter = 0
for ely in range(0, nlat-1):
   for elx in range(0, nlon-1):
      icon[0, counter] = ely + elx * nlat	
      icon[1, counter] = ely + elx * nlat + 1
      icon[2, counter] = ely + (elx + 1) * nlat + 1
      icon[3, counter] = ely + (elx + 1) * nlat
      counter += 1

###################################################################################################
# Creating list for all values
###################################################################################################

lons = []
lats = []
sat_x = []
sat_y = []
sat_z = []
grav_x = []
grav_y = []
grav_z = []
grav_norm = []
grav_norm2 = []
grav_pot = []
grav_pot2 = []
grav_pot_theory = []
grav_theory = []
grav_anom_x = []
grav_anom_y = []
grav_anom_z = []
grav_anom_norm = []
grav_grad_xx = []
grav_grad_yy = []
grav_grad_zz = []
grav_grad_xy = []
grav_grad_xz = []
grav_grad_yz = []
grav_grad_theory_xx = []
grav_grad_theory_yy = []
grav_grad_theory_zz = []
grav_grad_theory_xy = []
grav_grad_theory_xz = []
grav_grad_theory_yz = []

###################################################################################################
# Reading ascii file
###################################################################################################

counter = 0
with open(filename) as fi:
   for line in fi:
      if counter > 29:
         fields = line.split()
         lons.append(float(fields[1]))
         lats.append(float(fields[2]))
         sat_x.append(float(fields[3]))
         sat_y.append(float(fields[4]))
         sat_z.append(float(fields[5]))
         grav_x.append(float(fields[6]))
         grav_y.append(float(fields[7]))
         grav_z.append(float(fields[8]))
         grav_norm.append(float(fields[9]))
         grav_theory.append(float(fields[10]))
         grav_pot.append(float(fields[11]))
         grav_pot_theory.append(float(fields[12]))
         grav_anom_x.append(float(fields[13]))
         grav_anom_y.append(float(fields[14]))
         grav_anom_z.append(float(fields[15]))
         grav_anom_norm.append(float(fields[16]))
         grav_grad_xx.append(float(fields[17]))
         grav_grad_yy.append(float(fields[18]))
         grav_grad_zz.append(float(fields[19]))
         grav_grad_xy.append(float(fields[20]))
         grav_grad_xz.append(float(fields[21]))
         grav_grad_yz.append(float(fields[22]))
         grav_grad_theory_xx.append(float(fields[23]))
         grav_grad_theory_yy.append(float(fields[24]))
         grav_grad_theory_zz.append(float(fields[25]))
         grav_grad_theory_xy.append(float(fields[26]))
         grav_grad_theory_xz.append(float(fields[27]))
         grav_grad_theory_yz.append(float(fields[28]))
      counter += 1

###################################################################################################
# remove average of fields
###################################################################################################

grav_norm2[:]=grav_norm[:]-np.sum(grav_norm)/nnp

grav_pot2[:]=grav_pot[:]-np.sum(grav_pot)/nnp

###################################################################################################
# Creating vtu file
###################################################################################################

vtufile=open(filename+'_map.vtu',"w")
vtufile.write("<VTKFile type='UnstructuredGrid' version='0.1' byte_order='BigEndian'> \n")
vtufile.write("<UnstructuredGrid> \n")
vtufile.write("<Piece NumberOfPoints=' %5d ' NumberOfCells=' %5d '> \n" %(nnp,nel))
####
vtufile.write("<Points> \n")
vtufile.write("<DataArray type='Float32' NumberOfComponents='3' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f %10f %10f \n" %(lons[i],90-lats[i],0.))
vtufile.write("</DataArray>\n")
vtufile.write("</Points> \n")
####
vtufile.write("<PointData Scalars='scalars'>\n")
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_x' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_x[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_y' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_y[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_z' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_z[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_norm' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_norm[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_norm (zero avrg)' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_norm2[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_theory' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_theory[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_potential' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_pot[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_potential (zero avrg)' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_pot2[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_potential_theory' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_pot_theory[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_anomaly_x' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_anom_x[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_anomaly_y' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_anom_y[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_anomaly_z' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_anom_z[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_anomaly_norm' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_anom_norm[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_gradient_xx' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_grad_xx[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_gradient_yy' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_grad_yy[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_gradient_zz' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_grad_zz[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_gradient_xy' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_grad_xy[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_gradient_xz' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_grad_xz[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_gradient_yz' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_grad_yz[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_gradient_theory_xx' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_grad_theory_xx[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_gradient_theory_yy' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_grad_theory_yy[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_gradient_theory_zz' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_grad_theory_zz[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_gradient_theory_xy' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_grad_theory_xy[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_gradient_theory_xz' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_grad_theory_xz[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Float32' NumberOfComponents='1' Name='gravity_gradient_theory_yz' Format='ascii'> \n")
for i in range(0,nnp):
   vtufile.write("%10f \n" %grav_grad_theory_yz[i])
vtufile.write("</DataArray>\n")
#--
vtufile.write("</PointData>\n")
####
vtufile.write("<Cells>\n")
#--
vtufile.write("<DataArray type='Int32' Name='connectivity' Format='ascii'> \n")
for iel in range (0,nel):
    vtufile.write("%d %d %d %d\n" %(icon[0,iel],icon[1,iel],icon[2,iel],icon[3,iel]))
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Int32' Name='offsets' Format='ascii'> \n")
for iel in range (0,nel):
   vtufile.write("%d \n" %((iel+1)*4))
vtufile.write("</DataArray>\n")
#--
vtufile.write("<DataArray type='Int32' Name='types' Format='ascii'>\n")
for iel in range (0,nel):
   vtufile.write("%d \n" %9)
vtufile.write("</DataArray>\n")
#--
vtufile.write("</Cells>\n")
####
vtufile.write("</Piece>\n")
vtufile.write("</UnstructuredGrid>\n")
vtufile.write("</VTKFile>\n")
vtufile.close()

print("generated", filename+'_map.vtu')

###################################################################################################
