import copy
import gc
from datetime import datetime
from logging import Logger
from pathlib import Path
from typing import Dict, Iterable, List, Self, Any

import numpy as np
import pandas as pd
import xarray as xr
import yaml
import numba

from scipy.interpolate import CubicSpline

from src.config import Config, get_config, get_logger

config: Config = get_config()
logger: Logger = get_logger(config, __name__)

IntToIntMap = Dict[int, int]
IntToIntArrayMap = Dict[int, List[int]]

g_d_cp = 9.81 / 1005.0
cp_d_rd = 1005.0 / 287.0
rd_d_rv = 287.0 / 461.51
rd_d_cp = 287.0 / 1005.0
a = 611.0
b = 17.67
c = 273.15
d = 29.65


class Namelist(dict):
    """_summary_"""

    def __init__(self, *args, **kwargs) -> None:
        super(Namelist, self).__init__(*args, **kwargs)
        self._output = []

    def copy(self) -> Self:
        return copy.deepcopy(self)

    def to_f90_namelist(self, path: Path) -> None:
        self._output = []

        self._output.append(
            f"! Generated by {__class__.__module__}.{__class__.__name__} "
            f"at {datetime.now()} \n\n"
        )

        for key, item in self.items():
            self._output.append(f"&{key}\n")  # begin parameter section

            self._max_key_len = max(len(key_str) for key_str in item.keys())

            for key, value in item.items():
                self._write_parameter(key, value)
            self._output.append("/\n\n")  # section separator

        with open(path, "w") as nml:
            nml.writelines(self._output)
        self._output = []

    def _write_parameter(self, key, item):
        if item is None:
            logger.warning(f"omitting key {key} with null value from Namelist")
            return
        out = f"    {key:<{self._max_key_len}} ="
        if isinstance(item, str):
            out += f" '{item}',\n"
        elif isinstance(item, bool):
            out += f" {'.T.' if item else '.F.'},\n"
        elif isinstance(item, int):
            out += f" {item},\n"
        elif isinstance(item, float):
            out += f" {item:#g},\n"
        elif isinstance(item, Iterable):
            indent = " " * len(out)
            for index, list_item in enumerate(item):
                if not index == 0:
                    out += indent
                if isinstance(list_item, str):
                    out += f" '{list_item}',"
                elif isinstance(list_item, bool):
                    out += f" {'.T.' if list_item else '.F.'},"
                elif isinstance(list_item, int):
                    out += f" {list_item},"
                elif isinstance(list_item, float):
                    out += f" {list_item:#g},"
                else:
                    e = f"invalid list item type for key {key} in Namelist: {type(list_item)}"
                    logger.error(e)
                    raise TypeError(e)
                out += "\n"
        else:
            e = f"invalid value type for key {key} in Namelist: {type(item)}"
            logger.error(e)
            raise TypeError(e)
        self._output.append(out)


def read_namelist(path: Path) -> Namelist:
    with open(path, "r") as nml:
        namelist = Namelist(yaml.safe_load(nml))
    return namelist


class Driver:
    """_summary_"""

    def __init__(self) -> None:
        self.ds: xr.Dataset = xr.Dataset()
        self.p3d: Namelist | None = None

    def copy(self) -> Self:
        return copy.deepcopy(self)

    def set_grid(self, grids: str = "uvws", vertical: bool = True) -> None:
        assert isinstance(self.p3d, Namelist)

        nx = self.p3d["initialization_parameters"]["nx"] + 1
        ny = self.p3d["initialization_parameters"]["ny"] + 1
        dx = self.p3d["initialization_parameters"]["dx"]
        dy = self.p3d["initialization_parameters"]["dy"]

        # Horizontal grid
        create_x = False
        create_y = False
        if "u" in grids:
            coord = np.arange(dx, nx * dx, dx, dtype=np.float32)
            if "xu" not in self.ds.coords:
                self.ds = self.ds.assign_coords(xu=coord)
                self.ds["xu"].attrs["standard_name"] = "x coordinate of cell faces"
                self.ds["xu"].attrs["units"] = "m"
            else:
                self.ds["xu"] = coord
            create_y = True
        if "v" in grids:
            coord = np.arange(dy, ny * dy, dy, dtype=np.float32)
            if "yv" not in self.ds.coords:
                self.ds = self.ds.assign_coords(yv=coord)
                self.ds["yv"].attrs["standard_name"] = "y coordinate of cell faces"
                self.ds["yv"].attrs["units"] = "m"
            else:
                self.ds["yv"] = coord
            create_x = True
        if "s" in grids:
            create_x = True
            create_y = True

        if create_x:
            coord = np.arange(dx / 2, nx * dx, dx, dtype=np.float32)
            if "x" not in self.ds.coords:
                self.ds = self.ds.assign_coords(x=coord)
                self.ds["x"].attrs["standard_name"] = "x coordinate of cell centers"
                self.ds["x"].attrs["units"] = "m"
            else:
                self.ds["x"] = coord

        if create_y:
            coord = np.arange(dy / 2, ny * dy, dy, dtype=np.float32)
            if "y" not in self.ds.coords:
                self.ds = self.ds.assign_coords(y=coord)
                self.ds["y"].attrs["standard_name"] = "y coordinate of cell centers"
                self.ds["y"].attrs["units"] = "m"
            else:
                self.ds["y"] = coord

        if vertical:
            # @todo add possibility for grid streching
            nz = self.p3d["initialization_parameters"]["nz"]
            dz = self.p3d["initialization_parameters"]["dz"]
            coord = np.arange(dz / 2, nz * dz, dz, dtype=np.float32)
            if "z" not in self.ds.coords:
                self.ds = self.ds.assign_coords(z=coord)
                self.ds["z"].attrs["standard_name"] = "z coordinate of cell centers"
                self.ds["z"].attrs["units"] = "m"
            else:
                self.ds["z"] = coord

            if "w" in grids:
                coord = np.arange(dz, nz * dz, dz, dtype=np.float32)
                if "zw" not in self.ds.coords:
                    self.ds = self.ds.assign_coords(zw=coord)
                    self.ds["zw"].attrs["standard_name"] = "z coordinate of cell faces"
                    self.ds["zw"].attrs["units"] = "m"
                else:
                    self.ds["zw"] = coord

    def set_zsoil(self) -> None:
        assert isinstance(self.p3d, Namelist)

        dz_soil = self.p3d["land_surface_parameters"].get(
            "dz_soil", [0.01, 0.02, 0.04, 0.06, 0.14, 0.26, 0.54, 1.86]
        )

        # Soil depth at layer centers
        z_soil = np.cumsum(dz_soil) - np.array(dz_soil) * 0.5
        if "zsoil" not in self.ds.coords:
            self.ds = self.ds.assign_coords(zsoil=z_soil)
            self.ds["zsoil"].attrs["standard_name"] = "depth_below_land"
            self.ds["zsoil"].attrs["units"] = "m"
        else:
            self.ds["zsoil"] = z_soil

    def set_time(
        self, freq: str | pd.tseries.frequencies.DateOffset, coord: str = "time"
    ) -> None:
        assert isinstance(self.p3d, Namelist)

        if isinstance(freq, str):
            freq_temp = pd.tseries.frequencies.to_offset(freq)
            if not isinstance(freq_temp, pd.tseries.frequencies.DateOffset):
                e = f"invalid frequency {freq} for time dimension"
                logger.error(e)
                raise ValueError(e)
            else:
                freq = freq_temp
        freq_seconds = freq.nanos / 1e9

        end_time = self.p3d["runtime_parameters"]["end_time"]
        num_steps = int(end_time / freq_seconds) + 1
        time = np.linspace(0.0, end_time, num_steps)

        if coord not in self.ds.coords:
            self.ds = self.ds.assign_coords({coord: time})
            self.ds[coord].attrs["standard_name"] = "time"
            self.ds[coord].attrs["units"] = (
                "seconds since "
                + self.p3d["initialization_parameters"]["origin_date_time"]
            )
        else:
            self.ds[coord] = time

    def set_attrs(self, path: Path):
        self.ds.attrs["creation_time"] = datetime.utcnow().strftime(
            "%Y-%m-%d %H:%M:%S +00"
        )
        with open(path, "r") as attrf:
            attrs = dict(yaml.safe_load(attrf))
        self.ds = self.ds.assign_attrs(attrs)

    def __repr__(self) -> str:
        return self.ds.__repr__()

    def _repr_html_(self) -> str:
        return self.ds._repr_html_()

    def write(self, path: Path):
        # First, set correct dtypes
        for coord in self.ds.coords:
            if np.issubdtype(self.ds[coord].dtype, np.floating):
                self.ds[coord] = self.ds[coord].astype(np.float32)
                self.ds[coord].attrs["_FillValue"] = -9999.0
            elif np.issubdtype(self.ds[coord].dtype, np.integer):
                self.ds[coord] = self.ds[coord].astype(np.int8)
                self.ds[coord].attrs["_FillValue"] = -127

        for var in self.ds.variables:
            if "force_dtype" in self.ds[var].attrs.keys():
                del self.ds[var].attrs["force_dtype"]
                continue
            if np.issubdtype(self.ds[var].dtype, np.floating):
                self.ds[var] = self.ds[var].astype(np.float32)
                self.ds[var] = self.ds[var].fillna(-9999.0)
                self.ds[var].attrs["_FillValue"] = -9999.0
            elif np.issubdtype(self.ds[var].dtype, np.integer):
                self.ds[var] = self.ds[var].fillna(-127)
                self.ds[var] = self.ds[var].astype(np.int8)
                self.ds[var].attrs["_FillValue"] = -127

        self.ds.attrs["origin_time"] = self.p3d["initialization_parameters"][
            "origin_date_time"
        ]

        self.ds.to_netcdf(path)


class Job:
    """A job configuration for a PALM simulation."""

    def __init__(self, name: str) -> None:
        self._name = name
        self.drivers: Dict[str, xr.Dataset] = {}
        self.p3d: Namelist | None = None
        self.p3dr: Namelist | None = None
        self.jobdir: Path = Path(config.path.data.jobs) / name
        self.children: Dict[int, Job] = {}

    @property
    def name(self) -> str:
        return self._name

    @name.setter
    def name(self, name: str) -> None:
        self._name = name
        self.jobdir = Path(config.path.data.jobs) / name

    def copy(self) -> Self:
        return copy.deepcopy(self)

    def write_dynamic(self):
        if "dynamic" in self.drivers.keys():
            self.drivers["dynamic"].write(
                self.jobdir / "INPUT" / (self.name + "_dynamic")
            )
        else:
            logger.warning(f"no dynamic driver specified for job {self.name}")

    def write(self, nest_id: int | None = None):
        logger.info(f"writing input files for job {self.name} in {self.jobdir} ...")
        self.jobdir.mkdir(parents=True, exist_ok=True)
        (self.jobdir / "INPUT").mkdir(parents=True, exist_ok=True)
        (self.jobdir / "USER_CODE").mkdir(parents=True, exist_ok=True)
        (self.jobdir / "MONITORING").mkdir(parents=True, exist_ok=True)
        (self.jobdir / "OUTPUT").mkdir(parents=True, exist_ok=True)

        if nest_id is not None:
            nest_char = f"_N{nest_id:02}"
        else:
            nest_char = ""

        if self.p3d is not None:
            pass
            # self.p3d.to_f90_namelist(self.jobdir / "INPUT" / (self.name + "_p3d" + nest_char))
        else:
            logger.warning(f"job {self.name} is missing p3d namelist")

        if self.p3dr is not None:
            pass
            # self.p3dr.to_f90_namelist(self.jobdir / "INPUT" / (self.name + "_p3dr" + nest_char))
        else:
            logger.info(
                f"no p3dr namelist specified for job {self.name} - restart runs won't be possible"
            )

        if "static" in self.drivers.keys():
            # self.drivers["static"].write(
            #    self.jobdir / "INPUT" / (self.name + "_static" + nest_char)
            # )
            pass
        else:
            logger.warning(f"job {self.name} is missing static driver")

        if "dynamic" in self.drivers.keys():
            self.drivers["dynamic"].write(
                self.jobdir / "INPUT" / (self.name + "_dynamic" + nest_char)
            )
        else:
            logger.warning(f"no dynamic driver specified for job {self.name}")

        if "slurb" in self.drivers.keys():
            pass
            # self.drivers["slurb"].write(self.jobdir / "INPUT" / (self.name + "_slurb" + nest_char))

        # Write all child domain
        if len(self.children.keys()) > 0:
            for child in self.children.values():
                child.write()

    def register_driver(self, name: str, driver: Driver):
        self.drivers[name] = driver
        self.drivers[name].p3d = self.p3d

    def driver_from_config(self, path: Path, driver_name: str) -> None:
        with open(path, "r") as cfile:
            config: Dict = yaml.safe_load(cfile)
        ds = self.drivers[driver_name].ds
        for var_name, var_def in config.items():
            dims = var_def["dims"]
            target_shape = tuple(len(ds[dim]) for dim in dims)
            if "value" in var_def.keys():
                var = np.array(var_def["value"])
                var = np.broadcast_to(var.T, target_shape[::-1]).T
                ds[var_name] = (dims, var)
            elif "external" in var_def.keys():
                pass

    def set_turbulent_inflow(
        self,
        inflow_source_file: Path,
        init_source_file: Path,
        dynamic_source_file: Path,
    ):
        if "dynamic" not in self.drivers.keys():
            self.register_driver("dynamic", Driver())
            self.drivers["dynamic"].set_grid("uvws", vertical=True)

        inflow_source = xr.open_dataset(
            inflow_source_file, decode_times=False, chunks={"time": "auto"}
        )
        end_index = None
        for ti, time in enumerate(inflow_source.time):
            if time > 1e6:
                end_index = ti
                break
        inflow_source = inflow_source.isel(time=slice(0, end_index))
        inflow_source = inflow_source.squeeze()
        inflow_source = inflow_source.isel(zu=slice(1, -1))
        inflow_source = inflow_source.isel(zw=slice(1, -2))
        inflow_source = inflow_source.dropna(dim="time")
        # offset the inflow time dimension to zero origin
        time_inflow = inflow_source["time"] - inflow_source["time"][0]
        if time_inflow[-1] < self.p3d["runtime_parameters"]["end_time"]:
            logger.error(
                f"source file {inflow_source_file} for turbulent inflow of job"
                f" {self.name} does not cover the job duration: excpected"
                f">={self.p3d["runtime_parameters"]["end_time"]} s, got"
                f" {time_inflow[-1].data:.1f} s"
            )

        self.drivers["dynamic"].ds = self.drivers["dynamic"].ds.assign_coords(
            {"time_inflow": time_inflow.data}
        )
        self.drivers["dynamic"].ds.chunk({"time_inflow": "auto"})

        self.drivers["dynamic"].ds["inflow_plane_u"] = (
            ("time_inflow", "z", "y"),
            inflow_source["u_yz"].data,
        )
        # Ensure non-negative inflow
        self.drivers["dynamic"].ds["inflow_plane_u"] = (
            self.drivers["dynamic"]
            .ds["inflow_plane_u"]
            .where(self.drivers["dynamic"].ds["inflow_plane_u"] >= 0.0, other=0.0)
        )

        self.drivers["dynamic"].ds["inflow_plane_u"].attrs["units"] = "m/s"

        self.drivers["dynamic"].ds["inflow_plane_v"] = (
            ("time_inflow", "z", "y"),
            inflow_source["v_yz"].data,
        )
        self.drivers["dynamic"].ds["inflow_plane_v"].attrs["units"] = "m/s"

        self.drivers["dynamic"].ds["inflow_plane_w"] = (
            ("time_inflow", "zw", "y"),
            inflow_source["w_yz"].data,
        )
        self.drivers["dynamic"].ds["inflow_plane_w"].attrs["units"] = "m/s"

        self.drivers["dynamic"].ds["inflow_plane_e"] = (
            ("time_inflow", "z", "y"),
            inflow_source["e_yz"].data,
        )
        self.drivers["dynamic"].ds["inflow_plane_e"].attrs["units"] = "m2/s2"

        self.drivers["dynamic"].ds["inflow_plane_pt"] = (
            ("time_inflow", "z", "y"),
            inflow_source["theta_yz"].data,
        )
        self.drivers["dynamic"].ds["inflow_plane_pt"].attrs["units"] = "K"

        self.drivers["dynamic"].ds["inflow_plane_q"] = (
            ("time_inflow", "z", "y"),
            inflow_source["q_yz"].data,
        )
        self.drivers["dynamic"].ds["inflow_plane_q"].attrs["units"] = "kg/kg"

        inflow_source.close()

        self.drivers["dynamic"].ds["inflow_plane_q"] = xr.apply_ufunc(
            limit_q_to_saturation,
            self.drivers["dynamic"].ds["inflow_plane_q"],
            self.drivers["dynamic"].ds["inflow_plane_pt"],
            self.drivers["dynamic"].ds["z"],
            dask="parallelized",
            output_dtypes=[float],
        )

        # set initial volume data
        init_source = (
            xr.open_dataset(
                init_source_file,
                decode_times=False,
                chunks={"z": "auto", "zw": "auto"},
            )
            .isel(time=0)
            .squeeze()
        )
        init_source = init_source.isel(zu_3d=slice(1, -1))
        init_source = init_source.isel(zw_3d=slice(1, -2))
        init_source = init_source.mean(dim=("x", "xu", "y", "yv"))
        self.drivers["dynamic"].ds["init_atmosphere_u"] = (
            ("z",),
            init_source["u"].data,
        )
        self.drivers["dynamic"].ds["init_atmosphere_u"] = (
            self.drivers["dynamic"]
            .ds["init_atmosphere_u"]
            .where(self.drivers["dynamic"].ds["init_atmosphere_u"] >= 0.0, other=0.0)
        )
        self.drivers["dynamic"].ds["init_atmosphere_u"].attrs["units"] = "m/s"
        self.drivers["dynamic"].ds["init_atmosphere_u"].attrs["lod"] = 1

        self.drivers["dynamic"].ds["init_atmosphere_v"] = (
            ("z",),
            init_source["v"].data,
        )
        self.drivers["dynamic"].ds["init_atmosphere_v"].attrs["units"] = "m/s"
        self.drivers["dynamic"].ds["init_atmosphere_v"].attrs["lod"] = 1

        self.drivers["dynamic"].ds["init_atmosphere_w"] = (
            ("zw",),
            init_source["w"].data,
        )
        self.drivers["dynamic"].ds["init_atmosphere_w"].attrs["units"] = "m/s"
        self.drivers["dynamic"].ds["init_atmosphere_w"].attrs["lod"] = 1

        self.drivers["dynamic"].ds["init_atmosphere_pt"] = (
            ("z",),
            init_source["theta"].data,
        )
        self.drivers["dynamic"].ds["init_atmosphere_pt"].attrs["units"] = "K"
        self.drivers["dynamic"].ds["init_atmosphere_pt"].attrs["lod"] = 1

        self.drivers["dynamic"].ds["init_atmosphere_qv"] = (
            ("z",),
            init_source["q"].data,
        )
        self.drivers["dynamic"].ds["init_atmosphere_qv"].attrs["units"] = "kg/kg"
        self.drivers["dynamic"].ds["init_atmosphere_qv"].attrs["lod"] = 1

        # set geostrophic forcing
        dynamic_source = xr.open_dataset(dynamic_source_file)
        self.drivers["dynamic"].ds["ls_forcing_ug"] = dynamic_source["ls_forcing_ug"]
        self.drivers["dynamic"].ds["ls_forcing_vg"] = dynamic_source["ls_forcing_vg"]


class ExperimentCatalog:
    baseline: Job | None = None
    experiments: List[str] = []
    config: Dict = {}
    baseline_turbulent_inflow: Path | None = None

    def copy(self) -> Self:
        return copy.deepcopy(self)

    def read_config(self, path: Path) -> None:
        with open(path, "r") as cfile:
            self.config = yaml.safe_load(cfile)

    def generate_experiments(self) -> None:
        self.experiments = []

        self._process_inflow_experiments(self.config)
        # self._process_radiation_experiments(self.config)
        # self._process_namelist_experiments(self.config)
        # self._process_slurb_driver_experiments(self.config)

    def job_from_baseline(self, name: str) -> None:
        job = Job(name)
        job.p3d = self.baseline.p3d.copy()
        job.p3dr = self.baseline.p3dr.copy()
        job.register_driver("slurb", self.baseline.drivers["slurb"].copy())
        job.register_driver("static", self.baseline.drivers["static"].copy())
        return job

    def _process_inflow_experiments(self, exp_config: Dict) -> None:
        for experiment_name, experiment in exp_config.items():
            if experiment["category"] != "inflow":
                continue

            if experiment["type"] == "inflow_driver":
                for val in experiment["values"]:
                    name = val.split("_")[-1]
                    job = self.job_from_baseline(f"slurb_s_{name}")
                    job.register_driver("dynamic", Driver())
                    job.drivers["dynamic"].ds = xr.open_dataset(
                        self.baseline.jobdir
                        / "INPUT"
                        / (self.baseline.name + "_dynamic"),
                        chunks={"time": "auto", "time_rad": "auto"},
                    )
                    job.drivers["dynamic"].ds = job.drivers["dynamic"].ds.drop_dims(
                        "time_inflow"
                    )
                    job.set_turbulent_inflow(
                        inflow_source_file=Path(config.path.data.jobs)
                        / val
                        / "OUTPUT"
                        / f"{val}_yz.001.nc",
                        init_source_file=Path(config.path.data.jobs)
                        / val
                        / "OUTPUT"
                        / f"{val}_3d.001.nc",
                        dynamic_source_file=Path(config.path.data.jobs)
                        / val
                        / "INPUT"
                        / f"{val}_dynamic",
                    )
                    self.write_job(job)

    def _process_radiation_experiments(self, config: Dict) -> None:
        for experiment_name, experiment in config.items():
            if experiment["category"] != "radiation":
                continue

            if experiment["type"] == "difference":
                for parameter_key, parameter in experiment["parameter"].items():
                    for val in parameter["values"]:
                        job = self.job_from_baseline(
                            f"slurb_s_{experiment_name}+{val:.0f}"
                        )
                        job.register_driver("dynamic", Driver())
                        job.drivers["dynamic"].ds = xr.open_dataset(
                            self.baseline.jobdir
                            / "INPUT"
                            / (self.baseline.name + "_dynamic"),
                            chunks={"time": "auto", "time_rad": "auto"},
                        )
                        job.drivers["dynamic"].ds[parameter_key] = (
                            job.drivers["dynamic"].ds[parameter_key] + val
                        )
                        job.drivers["dynamic"].ds[parameter_key].attrs["lod"] = 1
                        self.experiments.append(job.name)
                        self.write_job(job)

                        job = self.job_from_baseline(
                            f"slurb_s_{experiment_name}-{val:.0f}"
                        )
                        job.register_driver("dynamic", Driver())
                        job.drivers["dynamic"].ds = xr.open_dataset(
                            self.baseline.jobdir
                            / "INPUT"
                            / (self.baseline.name + "_dynamic"),
                            chunks={"time": "auto", "time_rad": "auto"},
                        )
                        job.drivers["dynamic"].ds[parameter_key] = (
                            job.drivers["dynamic"].ds[parameter_key] - val
                        )
                        job.drivers["dynamic"].ds[parameter_key].attrs["lod"] = 1
                        self.experiments.append(job.name)
                        self.write_job(job)

            if experiment["type"] == "factor":
                for parameter_key, parameter in experiment["parameter"].items():
                    for val in parameter["values"]:
                        job = self.job_from_baseline(
                            f"slurb_s_{experiment_name}+x{val:.1f}"
                        )
                        job.register_driver("dynamic", Driver())
                        job.drivers["dynamic"].ds = xr.open_dataset(
                            self.baseline.jobdir
                            / "INPUT"
                            / (self.baseline.name + "_dynamic"),
                            chunks={"time": "auto", "time_rad": "auto"},
                        )
                        job.drivers["dynamic"].ds[parameter_key] = (
                            job.drivers["dynamic"].ds[parameter_key]
                            + job.drivers["dynamic"].ds[parameter_key] * val
                        )
                        job.drivers["dynamic"].ds[parameter_key].attrs["lod"] = 1
                        self.experiments.append(job.name)
                        self.write_job(job)

                        job = self.job_from_baseline(
                            f"slurb_s_{experiment_name}-x{val:.1f}"
                        )
                        job.register_driver("dynamic", Driver())
                        job.drivers["dynamic"].ds = xr.open_dataset(
                            self.baseline.jobdir
                            / "INPUT"
                            / (self.baseline.name + "_dynamic"),
                            chunks={"time": "auto", "time_rad": "auto"},
                        )
                        job.drivers["dynamic"].ds[parameter_key] = (
                            job.drivers["dynamic"].ds[parameter_key]
                            - job.drivers["dynamic"].ds[parameter_key] * val
                        )
                        job.drivers["dynamic"].ds[parameter_key].attrs["lod"] = 1
                        self.experiments.append(job.name)
                        self.write_job(job)

    def _process_namelist_experiments(self, config: Dict) -> None:
        for experiment_name, experiment in config.items():
            if experiment["category"] != "namelist":
                continue

            if experiment["type"] == "parameter":
                for parameter_key, parameter in experiment["parameter"].items():
                    for val in parameter["values"]:
                        job = self.job_from_baseline(
                            f"slurb_s_{experiment_name}_{str(val).lower()}"
                        )
                        job.p3d[parameter["section"]][parameter_key] = val
                        job.p3dr[parameter["section"]][parameter_key] = val
                        if parameter_key == "anisotropic_street_canyons":
                            job.drivers["slurb"].ds["street_canyon_orientation"] = (
                                ("y", "x"),
                                np.random.rand(
                                    job.drivers["slurb"].ds.y.size,
                                    job.drivers["slurb"].ds.x.size,
                                )
                                * 90.0,
                            )
                        self.experiments.append(job.name)
                        self.write_job(job)

            if experiment["type"] == "parameter_combination":
                job = self.job_from_baseline(f"slurb_s_{experiment_name}")
                for parameter_key, parameter in experiment["parameters"].items():
                    val = parameter["value"]
                    job.p3d[parameter["section"]][parameter_key] = val
                    job.p3dr[parameter["section"]][parameter_key] = val
                self.experiments.append(job.name)
                self.write_job(job)

    def _process_slurb_driver_experiments(self, config: Dict) -> None:
        for experiment_name, experiment in config.items():
            if experiment["category"] != "slurb_driver":
                continue

            if experiment["type"] == "factor":
                for parameter_key, parameter in experiment["parameter"].items():
                    for val in parameter["values"]:
                        job = self.job_from_baseline(
                            f"slurb_s_{experiment_name}+x{val:.1f}"
                        )
                        job.drivers["slurb"].ds[parameter_key] = (
                            job.drivers["slurb"]
                            .ds[parameter_key]
                            .where(
                                job.drivers["slurb"].ds[parameter_key]
                                == job.drivers["slurb"]
                                .ds[parameter_key]
                                .attrs.get("_FillValue", -9999.0),
                                other=(
                                    job.drivers["slurb"].ds[parameter_key]
                                    + job.drivers["slurb"].ds[parameter_key] * val
                                ),
                            )
                        )
                        self.experiments.append(job.name)
                        self.write_job(job)

                        job = self.job_from_baseline(
                            f"slurb_s_{experiment_name}-x{val:.1f}"
                        )
                        job.drivers["slurb"].ds[parameter_key] = (
                            job.drivers["slurb"]
                            .ds[parameter_key]
                            .where(
                                job.drivers["slurb"].ds[parameter_key]
                                == job.drivers["slurb"]
                                .ds[parameter_key]
                                .attrs.get("_FillValue", -9999.0),
                                other=(
                                    job.drivers["slurb"].ds[parameter_key]
                                    - job.drivers["slurb"].ds[parameter_key] * val
                                ),
                            )
                        )
                        self.experiments.append(job.name)
                        self.write_job(job)

            if experiment["type"] == "complement_factor":
                for parameter_key, parameter in experiment["parameter"].items():
                    for val in parameter["values"]:
                        job = self.job_from_baseline(
                            f"slurb_s_{experiment_name}+cx{val:.1f}"
                        )
                        job.drivers["slurb"].ds[parameter_key] = (
                            job.drivers["slurb"]
                            .ds[parameter_key]
                            .where(
                                job.drivers["slurb"].ds[parameter_key]
                                == job.drivers["slurb"]
                                .ds[parameter_key]
                                .attrs.get("_FillValue", 0.0),
                                other=(
                                    job.drivers["slurb"].ds[parameter_key]
                                    + (1.0 - job.drivers["slurb"].ds[parameter_key])
                                    * val
                                ),
                            )
                        )
                        self.experiments.append(job.name)
                        self.write_job(job)

                        job = self.job_from_baseline(
                            f"slurb_s_{experiment_name}-cx{val:.1f}"
                        )
                        job.drivers["slurb"].ds[parameter_key] = (
                            job.drivers["slurb"]
                            .ds[parameter_key]
                            .where(
                                job.drivers["slurb"].ds[parameter_key]
                                == job.drivers["slurb"]
                                .ds[parameter_key]
                                .attrs.get("_FillValue", 0.0),
                                other=(
                                    job.drivers["slurb"].ds[parameter_key]
                                    - (1.0 - job.drivers["slurb"].ds[parameter_key])
                                    * val
                                ),
                            )
                        )
                        self.experiments.append(job.name)
                        self.write_job(job)

            if experiment["type"] == "difference":
                for parameter_key, parameter in experiment["parameter"].items():
                    for val in parameter["values"]:
                        job = self.job_from_baseline(
                            f"slurb_s_{experiment_name}+{val:.1f}"
                        )
                        job.drivers["slurb"].ds[parameter_key] = job.drivers[
                            "slurb"
                        ].ds[parameter_key] = (
                            job.drivers["slurb"]
                            .ds[parameter_key]
                            .where(
                                job.drivers["slurb"].ds[parameter_key]
                                == job.drivers["slurb"]
                                .ds[parameter_key]
                                .attrs.get("_FillValue", -9999.0),
                                other=(job.drivers["slurb"].ds[parameter_key] + val),
                            )
                        )
                        self.experiments.append(job.name)
                        self.write_job(job)

                        job = self.job_from_baseline(
                            f"slurb_s_{experiment_name}-{val:.1f}"
                        )
                        job.drivers["slurb"].ds[parameter_key] = job.drivers[
                            "slurb"
                        ].ds[parameter_key] = (
                            job.drivers["slurb"]
                            .ds[parameter_key]
                            .where(
                                job.drivers["slurb"].ds[parameter_key]
                                == job.drivers["slurb"]
                                .ds[parameter_key]
                                .attrs.get("_FillValue", -9999.0),
                                other=(job.drivers["slurb"].ds[parameter_key] - val),
                            )
                        )
                        self.experiments.append(job.name)
                        self.write_job(job)

    def write_job(self, job: Job) -> None:
        job.write()
        # symlink the baseline dynamic driver containing the turbulent inflow
        if "dynamic" not in job.drivers.keys():
            logger.info(
                f"linking turbulent inflow for {job.name} from {self.baseline_turbulent_inflow}"
            )
            try:
                (job.jobdir / "INPUT" / (job.name + "_dynamic")).symlink_to(
                    self.baseline_turbulent_inflow
                )
            except FileExistsError:
                pass

        gc.collect()


class JobNest(Job):
    def __init__(
        self,
        *args,
        root: Job | None = None,
        nest_id: int | None = None,
        **kwargs,
    ):
        self.root = root
        self.nest_id = nest_id
        super().__init__(*args, name=self.root.name, **kwargs)
        self.jobdir = self.root.jobdir
        self.root.children[nest_id] = self

    def write(self):
        if "nesting_parameters" in self.p3d.keys():
            del self.p3d["nesting_parameters"]
        if "nesting_parameters" in self.p3dr.keys():
            del self.p3dr["nesting_parameters"]
        super().write(self.nest_id)


def dict_to_intarraymap(str_intarray_dict: IntToIntArrayMap) -> IntToIntMap:
    intarray_map: IntToIntMap = {}
    for key_str, values in str_intarray_dict.items():
        key_int = int(key_str)
        for val in values:
            intarray_map[val] = key_int
    return intarray_map


def update_dict_recursive(a: Dict[str, Any], b: Dict[str, Any]) -> None:
    for key, value in b.items():
        if isinstance(value, dict) and key in a and isinstance(a[key], dict):
            update_dict_recursive(a[key], value)
        else:
            a[key] = value


def limit_q_to_saturation(
    q: np.array,
    pt: np.array,
    z: np.array,
    p0: float = 1e5,
    t0: float = 300.0,
) -> np.array:
    # Compute hydrostatic pressure
    p = p0 * ((t0 - g_d_cp * z) / t0) ** cp_d_rd
    # Comptue Exner function
    exner = (p / p0) ** rd_d_cp
    # Compute saturation vapor pressure using Magnus formula
    e_s = a * np.exp(
        b * (pt * exner[np.newaxis, ...] - c) / (pt * exner[np.newaxis, ...] - d)
    )
    # Compute saturation mixing ratio
    q_s = rd_d_rv * e_s / (p[np.newaxis, ...] - e_s)
    # Return array where mixing ratio is limited to saturation
    q = np.where(q < q_s, q, q_s)
    return q


@numba.stencil(cval=0.0)
def grid_cell_wall_area(
    building_height: np.array = np.array([[]]), dx: float = 1.0, dy: float = 1.0
) -> numba.float64[:, :]:
    # Using negative difference helps with out-of-bounds (which are cval=0.0).
    dz_bottom = building_height[0, 0] - building_height[-1, 0]
    dz_top = building_height[0, 0] - building_height[1, 0]
    dz_left = building_height[0, 0] - building_height[0, -1]
    dz_right = building_height[0, 0] - building_height[0, 1]

    grid_cell_wall_area = 0.0

    # Checking for negative difference only ensures the walls aren't double-counted.
    if dz_top < 0.0:
        grid_cell_wall_area -= dz_top * dx  # Top wall
    if dz_bottom < 0.0:
        grid_cell_wall_area -= dz_bottom * dx  # Bottom wall
    if dz_left < 0.0:
        grid_cell_wall_area -= dz_left * dy  # Left wall
    if dz_right < 0.0:
        grid_cell_wall_area -= dz_right * dy  # Right wall

    return grid_cell_wall_area


@numba.njit(parallel=True, looplift=True)
def compute_wall_area(
    building_height=np.array([[]]),
    dx: float = 1.0,
    dy: float = 1.0,
) -> float:
    return np.sum(grid_cell_wall_area(building_height, dx, dy))


def get_urban_morphology_from_usm_driver(
    lcz_map: xr.DataArray,
    usm_driver: xr.Dataset,
) -> xr.Dataset:
    zones = np.unique(lcz_map)
    params = {}
    dx_usm = float(usm_driver.x[1] - usm_driver.x[0])
    dy_usm = float(usm_driver.y[1] - usm_driver.y[0])

    for zone in zones:
        params[zone] = {}

        lcz_mask = lcz_map == zone
        building_mask = lcz_mask & (usm_driver["building_type"] > 0)
        street_mask = lcz_mask & (usm_driver["street_type"] > 0)

        total_gridpoints = lcz_mask.sum()
        total_area = total_gridpoints * dx_usm * dy_usm

        # Mean building height
        params[zone]["building_height"] = float(
            usm_driver["buildings_2d"].where(building_mask).mean()
        )

        # Urban fraction
        params[zone]["urban_fraction"] = float(
            (building_mask | street_mask).sum() / total_gridpoints
        )

        # Plan area fraction
        params[zone]["building_plan_area_fraction"] = float(
            building_mask.sum() / total_gridpoints
        )

        # Compute total wall area
        building_height = usm_driver["buildings_2d"].where(building_mask).fillna(0.0)
        wall_area = compute_wall_area(
            building_height=building_height.data, dx=dx_usm, dy=dy_usm
        )

        wall_area_density = wall_area / total_area

        # Street canyon aspect ratio
        params[zone]["street_canyon_aspect_ratio"] = float(
            0.5
            * wall_area_density
            / (1.0 - params[zone]["building_plan_area_fraction"])
        )

        # Frontal area fraction
        params[zone]["building_frontal_area_fraction"] = float(wall_area_density / 4)

        # Roughness length following Kanda et al. (2017)
        # Building height standard deviation
        building_std = float(usm_driver["buildings_2d"].where(building_mask).std())
        print(building_std)
        # Parameter Y in Eq. (12b)

        param_a = 3.59  # almost-square array
        param_b = 1.0
        param_clb = 1.2
        k = 0.4
        # Displacement height
        d_dh = 1 + param_a ** (-params[zone]["building_plan_area_fraction"]) * (
            params[zone]["building_plan_area_fraction"] - 1
        )
        # Roughness length after Macdonald (1998)
        params[zone]["z0_mac"] = (
            params[zone]["building_height"]
            * (1.0 - d_dh)
            * np.exp(
                -(
                    (
                        0.5
                        * param_b
                        * param_clb
                        / k**2
                        * (1 - d_dh)
                        * params[zone]["building_frontal_area_fraction"]
                    )
                    ** -0.5
                )
            )
        )

        # Roughness length after Kanda et al. (2017)
        param_y = (
            params[zone]["building_plan_area_fraction"]
            * building_std
            / params[zone]["building_height"]
        )

        param_a1 = 0.71
        param_b1 = 20.21
        param_c1 = -0.77
        params[zone]["z0_kanda"] = params[zone]["z0_mac"] * (
            param_b1 * param_y**2 + param_c1 * param_y + param_a1
        )

    return params


def set_urban_morphology_to_slurb_driver(
    lcz_map: xr.DataArray,
    slurb_driver: xr.Dataset,
    urban_params: Dict,
    offset_x: float = 0.0,  # offset from domain bottom left corner [m]
    offset_y: float = 0.0,  # offset from domain bottom left corner [m]
    buffer: int = 0,  # buffer zone in grid points from domain edges free of urban area
):
    # Coarsen lcz map to slurb driver resolution
    dx_slurb = float(slurb_driver.x[1] - slurb_driver.x[0])
    dy_slurb = float(slurb_driver.y[1] - slurb_driver.y[0])
    dx_lcz = float(lcz_map.x[1] - lcz_map.x[0])
    dy_lcz = float(lcz_map.y[1] - lcz_map.y[0])

    coarsen_x = int(dx_slurb // dx_lcz)
    coarsen_y = int(dy_slurb // dy_lcz)

    lcz_map = lcz_map.isel(
        x=slice(None, None, coarsen_x), y=slice(None, None, coarsen_y)
    )
    lcz_map = lcz_map.astype(np.int8)
    # Set buffer points to NaN
    edge_mask = np.ones(lcz_map.shape, dtype=bool)
    edge_mask[:buffer, :] = False
    edge_mask[-buffer:, :] = False
    edge_mask[:, :buffer] = False
    edge_mask[:, -buffer:] = False
    lcz_map = lcz_map.where(edge_mask)

    # Compute the domain offset in grid points
    # Add half gridlength to offsets as they are in u/v grid and static input uses grid cell centers
    offset_i = np.argmin(np.abs(slurb_driver.x.data - (offset_x + dx_slurb / 2)))
    offset_j = np.argmin(np.abs(slurb_driver.y.data - (offset_y + dy_slurb / 2)))

    # Pad the LCZ map
    if any([off is not None for off in [offset_x, offset_y]]):
        lcz_map = lcz_map.pad(
            pad_width={"x": offset_i, "y": offset_j},
            mode="constant",
            constant_values=np.nan,
        )
        lcz_map = lcz_map.isel(
            x=slice(None, slurb_driver.x.size),
            y=slice(None, slurb_driver.y.size),
        )

    lcz_map["x"] = slurb_driver.x.data
    lcz_map["y"] = slurb_driver.y.data

    fill_value = np.nan

    slurb_driver["urban_fraction"] = (
        ("y", "x"),
        xr.apply_ufunc(
            lambda x: urban_params.get(x, {}).get("urban_fraction", 0.0),
            lcz_map,
            vectorize=True,
        ).data,
    )

    slurb_driver["building_frontal_area_fraction"] = (
        ("y", "x"),
        xr.apply_ufunc(
            lambda x: urban_params.get(x, {}).get(
                "building_frontal_area_fraction", fill_value
            ),
            lcz_map,
            vectorize=True,
        ).data,
    )

    slurb_driver["building_plan_area_fraction"] = (
        ("y", "x"),
        xr.apply_ufunc(
            lambda x: urban_params.get(x, {}).get(
                "building_plan_area_fraction", fill_value
            ),
            lcz_map,
            vectorize=True,
        ).data,
    )

    slurb_driver["building_height"] = (
        ("y", "x"),
        xr.apply_ufunc(
            lambda x: urban_params.get(x, {}).get("building_height", fill_value),
            lcz_map,
            vectorize=True,
        ).data,
    )

    slurb_driver["street_canyon_aspect_ratio"] = (
        ("y", "x"),
        xr.apply_ufunc(
            lambda x: urban_params.get(x, {}).get(
                "street_canyon_aspect_ratio", fill_value
            ),
            lcz_map,
            vectorize=True,
        ).data,
    )

    slurb_driver["street_canyon_aspect_ratio"] = (
        ("y", "x"),
        xr.apply_ufunc(
            lambda x: urban_params.get(x, {}).get(
                "street_canyon_aspect_ratio", fill_value
            ),
            lcz_map,
            vectorize=True,
        ).data,
    )

    slurb_driver["z0_urb"] = (
        ("y", "x"),
        xr.apply_ufunc(
            lambda x: urban_params.get(x, {}).get("z0_mac", fill_value),
            lcz_map,
            vectorize=True,
        ).data,
    )

    slurb_driver = slurb_driver.where(lcz_map > 0, other=fill_value)
    slurb_driver["urban_fraction"] = slurb_driver["urban_fraction"].where(
        lcz_map > 0, other=0.0
    )

    return slurb_driver


def set_radiation_to_dynamic(
    rad_ds: xr.Dataset, dynamic_driver: Driver, time_offset: float = 0.0
):
    """
    This function is explained in 02-precursor.ipynb.
    """
    rad_ds["secondsofday"] = rad_ds.time.dt.hour * 3600.0 - 1800.0
    rad_ds.coords["solar_time"] = (
        (np.pi * rad_ds.secondsofday + 43200.0 * np.deg2rad(rad_ds.longitude))
        / np.pi
        % (24 * 3600)
    )  # modulo to wrap the next day to the begin of current day
    hourly_bins = np.arange(0, 25, 1) * 3600.0
    bin_centers = hourly_bins[:-1] + 1800.0
    rad_ds = rad_ds.groupby_bins("solar_time", hourly_bins, labels=bin_centers).mean(
        dim=xr.ALL_DIMS
    )
    first_vals = rad_ds.isel(solar_time_bins=0)
    first_vals["solar_time_bins"] = 84600.0 + 3600.0
    rad_ds = xr.concat([rad_ds, first_vals], dim="solar_time_bins")
    dynamic_driver.set_time(freq="1min", coord="time_rad")
    # as our simulation (and the reference time) starts at 3 AM, shift the time backwards by 3 hours to align with reference time
    rad_ds["solar_time_bins"] = rad_ds["solar_time_bins"] - time_offset

    cs_rad_sw_in = CubicSpline(
        rad_ds.solar_time_bins,
        rad_ds.msdwswrf,
        bc_type="periodic",
    )
    dynamic_driver.ds["rad_sw_in"] = (
        ("time_rad",),
        cs_rad_sw_in(dynamic_driver.ds.time_rad),
    )

    cs_rad_lw_in = CubicSpline(
        rad_ds.solar_time_bins,
        rad_ds.msdwlwrf,
        bc_type="periodic",
    )
    dynamic_driver.ds["rad_lw_in"] = (
        ("time_rad",),
        cs_rad_lw_in(dynamic_driver.ds.time_rad),
    )
    dynamic_driver.ds["rad_sw_in"].attrs["lod"] = 1
    dynamic_driver.ds["rad_lw_in"].attrs["lod"] = 1

    return dynamic_driver


def set_initial_soil_conditions_from_precursor(
    pre_3d: xr.Dataset,
    pre_vegetation_type: xr.DataArray,
    dynamic_driver: Driver,
    vegetation_type: xr.DataArray,
):
    pre_soil = pre_3d[["t_soil", "m_soil"]].isel(time=0)

    mean_t = {}
    mean_t_all = pre_soil["t_soil"].mean(dim=("y", "x")).squeeze()
    mean_t_all[:] = mean_t_all.isel(zs_3d=-1).squeeze()
    mean_t_all = mean_t_all.data
    mean_m = {}
    mean_m_all = pre_soil["m_soil"].mean(dim=("y", "x")).squeeze().data
    for veg_type in np.unique(pre_vegetation_type):
        mean_t[veg_type] = (
            pre_soil["t_soil"]
            .where(pre_vegetation_type == veg_type)
            .mean(dim=("y", "x"))
            .squeeze()
            .data
        )
        mean_m[veg_type] = (
            pre_soil["m_soil"]
            .where(pre_vegetation_type == veg_type)
            .mean(dim=("y", "x"))
            .squeeze()
            .data
        )

    dynamic_driver.ds["init_soil_t"] = (
        ("zsoil", "y", "x"),
        np.moveaxis(
            np.vectorize(lambda x: mean_t.get(x, mean_t_all), signature="()->(k)")(
                vegetation_type
            ),
            -1,
            0,
        ),
    )

    dynamic_driver.ds["init_soil_m"] = (
        ("zsoil", "y", "x"),
        np.moveaxis(
            np.vectorize(lambda x: mean_m.get(x, mean_m_all), signature="()->(k)")(
                vegetation_type
            ),
            -1,
            0,
        ),
    )

    dynamic_driver.ds["init_soil_t"].attrs["lod"] = 2
    dynamic_driver.ds["init_soil_t"].attrs["units"] = "K"
    dynamic_driver.ds["init_soil_t"].attrs["long_name"] = "initial soil temperature"
    dynamic_driver.ds["init_soil_m"].attrs["lod"] = 2
    dynamic_driver.ds["init_soil_m"].attrs["units"] = "m**3 m**-3"
    dynamic_driver.ds["init_soil_m"].attrs["long_name"] = (
        "initial soil volumetric moisture"
    )

    return dynamic_driver


def set_slurb_deep_soil_temperature_from_dynamic(
    slurb_driver: Driver, dynamic_driver: Driver
):
    slurb_driver.ds["deep_soil_temperature"] = (
        dynamic_driver.ds["init_soil_t"]
        .isel(zsoil=-1)
        .squeeze()
        .where(slurb_driver.ds["urban_fraction"] > 0.0, np.nan)
    )

    return slurb_driver


def set_spinup_parameters_from_precursor(
    pre_pr: xr.Dataset, p3d: Namelist, p3dr: Namelist
):
    pre_pr = pre_pr["theta"].interp(ztheta=100.0)
    p3d["initialization_parameters"]["spinup_pt_mean"] = float(pre_pr.mean().values)
    p3d["initialization_parameters"]["spinup_pt_amplitude"] = float(
        pre_pr.quantile(0.997) - pre_pr.quantile(0.003)
    )
    p3dr["initialization_parameters"]["spinup_pt_mean"] = p3d[
        "initialization_parameters"
    ]["spinup_pt_mean"]
    p3dr["initialization_parameters"]["spinup_pt_amplitude"] = p3d[
        "initialization_parameters"
    ]["spinup_pt_amplitude"]
    return p3d, p3dr


def set_surface_pressure_to_dynamic(dynamic_driver: Driver, p0=1e5):
    if "time" not in dynamic_driver.ds.dims.keys():
        dynamic_driver.set_time(freq="1min", coord="time")
    dynamic_driver.ds["surface_forcing_surface_pressure"] = (
        ("time",),
        [
            p0,
        ]
        * dynamic_driver.ds.time.size,
    )
    dynamic_driver.ds["surface_forcing_surface_pressure"].attrs["lod"] = 1
    dynamic_driver.ds["surface_forcing_surface_pressure"].attrs["units"] = "Pa"
    dynamic_driver.ds["surface_forcing_surface_pressure"].attrs["long_name"] = (
        "surface_pressure"
    )

    return dynamic_driver


def get_nest_mask(
    template: xr.DataArray,
    offset_x: float,
    offset_y: float,
    gridpoints_x: int,
    gridpoints_y: int,
):
    dx_static = float(template.x[1] - template.x[0])
    dy_static = float(template.y[1] - template.y[0])

    offset_i = np.argmin(np.abs(template.x.data - (offset_x + dx_static / 2)))
    offset_j = np.argmin(np.abs(template.y.data - (offset_y + dy_static / 2)))

    mask = xr.full_like(template, False, dtype=bool)

    mask[offset_j : offset_j + gridpoints_y, offset_i : offset_i + gridpoints_x] = True
    mask.name = "mask"

    return mask


def apply_buffer_to_usm_driver(static_driver: Driver, buffer: int):
    edge_mask = xr.ones_like(static_driver["buildings_2d"], dtype=bool)
    edge_mask[:buffer, :] = False
    edge_mask[-buffer:, :] = False
    edge_mask[:, :buffer] = False
    edge_mask[:, -buffer:] = False
    for var_name, var_data in static_driver.data_vars.items():
        if np.issubdtype(var_data.dtype, np.floating):
            static_driver[var_name] = var_data.where(edge_mask, other=np.nan)
        elif np.issubdtype(var_data.dtype, np.integer):
            static_driver[var_name] = var_data.where(edge_mask, other=-127)
    vegetation_fraction = static_driver["surface_fraction"].isel(nsurface_fraction=0)
    static_driver["surface_fraction"][0, :, :] = vegetation_fraction.where(
        edge_mask, other=1
    )
    return static_driver, edge_mask
