(* ::Package:: *)

(* ::Title:: *)
(*Package to compute master integrals for the families RL1, RL2, PL1 and PT4 at 3 loops*)


(* ::Section:: *)
(*Initialisation and input*)


$dir = If[$FrontEnd=!=Null,NotebookDirectory[],DirectoryName[$InputFileName]];
SetDirectory[$dir];


(* Specify path to DiffExp.m *)
PathToDiffExp = "~/diffexp/DiffExp.m";

If[!FileExistsQ[PathToDiffExp],
  Print["DiffExp.m not found. Please edit PathToDiffExp in DiffExpRun.wl"]; 
  Exit[];
];

Get[PathToDiffExp];


(*Possible choices of invariants*)
sij = {s12,s23,m32,m42};
xyz = {x,y,z};


(* Default values *)
family = RL1;
vars = sij;
filein = "testpoint.m";
fileout = "out.m";

CmdOptions = {
  "-family",
  "-vars",
  "-filein",
  "-fileout"
};

Do[
  pos = Flatten[Position[$CommandLine,CmdOptions[[oo]]]];
  If[Length[pos]==1,

    Switch[oo,
      1,  
      family = ToExpression[$CommandLine[[pos[[1]]+1]]];,
      2,
      vars = ToExpression[$CommandLine[[pos[[1]]+1]]];,
      3,
      filein = $CommandLine[[pos[[1]]+1]];,
      4,
      fileout = $CommandLine[[pos[[1]]+1]];
    ];
  ];
,{oo,1,Length[CmdOptions]}];


If[FileExistsQ[filein],
  Print["Reading point from ", filein];
  point = Get[filein];,
  Print[filein, " not found"];
  Exit[]];


Print["Computing integrals for family ",family," at point ", point, " in the variables ", vars];


point = Association[point];


SetDirectory[ToString[family]];


(* ::Section:: *)
(*Alphabet and DEs*)


$deqs = "/MIs_DEs/deqsij/";
If[vars==xyz,$deqs="/MIs_DEs/deqxyz"];


If[!DirectoryQ[$dir<>ToString[family]<>$deqs],
  Print["Computing DEs from dlog..."];
  CreateDirectory[$deqs]; 
  
  If[vars==sij, 
	dAtilde = Import[FileNameJoin[{$dir,ToString[family],"MIs_DEs/dAtilde.m"}]]; {roots,alphabet} = Import[FileNameJoin[{$dir,"GlobalDefinitions/alphabet.m"}]],
	dAtilde = Import[FileNameJoin[{$dir,ToString[family],"MIs_DEs/dAtildexyz.m"}]]; alphabet = Import[FileNameJoin[{$dir,"GlobalDefinitions/alphabetxyz.m"}]]; roots={};
	];
	
	Do[ 
      Export[$dir<>ToString[family]<>$deqs<>"/d"<>ToString[vars[[v]]]<>"_1.m",
      dAtilde/.dlog[a_]:>D[Log[a/.alphabet/.roots],vars[[v]]]];
	,{v,Length[vars]}];
  
  Print["Computed DEs and stored them in " $deqs];  
  ,
  Print["Found DEs in " $deqs];
];


(* ::Section:: *)
(*Evaluation*)


boundarypoint={m32 -> 1, s12 -> 13/4, s23 -> -3/4, m42 -> 9/20};
If[vars==xyz,boundarypoint={x->3/2,y->1/5,z->1/2}];


If[vars==sij,
	boundaryInt = Import[FileNameJoin[{$dir,ToString[family],"MIs_DEs/boundaries.m"}]],
	boundaryInt = Import[FileNameJoin[{$dir,ToString[family],"MIs_DEs/boundariesxyz.m"}]]
];


DEfamily={
Variables->vars,
DeltaPrescriptions->{},
MatrixDirectory->$dir<>ToString[family]<>$deqs,
UseMobius->False,
UsePade->False,
EpsilonOrder->6,
AccuracyGoal->40,ExpansionOrder->50,ChopPrecision->200,
LineParameter->t
};

LoadConfiguration[DEfamily];


DiffExpBC=PrepareBoundaryConditions[boundaryInt,Association[boundarypoint]];


Results=TransportTo[DiffExpBC,point];


(* ::Section:: *)
(*Output*)


ToIntegrals[result_]:= Module[{point, epsorder, Integrals,NInt, Ints},
	point = result[[1]];
	Integrals = result[[2]];
	NInt = Length[Integrals[[All,1]]];
	epsorder = Length[Integrals[[2]]];
	Ints = Range[NInt];
	Do[Ints[[i]] = Sum[Integrals[[i,j]] \[Epsilon]^(j-1),{j,1,epsorder}],{i,1,NInt}];
	Ints	
];


Export[FileNameJoin[{$dir,fileout}],ToIntegrals[Results]];
