/*
 * Copyright (C) 2018-2024 University of Waterloo.
 *
 * This file is part of Perses.
 *
 * Perses is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3, or (at your option) any later version.
 *
 * Perses is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * Perses; see the file LICENSE.  If not see <http://www.gnu.org/licenses/>.
 */
package org.perses.reduction.reducer.vulcan

import com.google.common.collect.ImmutableList
import com.google.common.collect.ImmutableMultimap
import com.google.common.primitives.ImmutableIntArray
import org.perses.antlr.ast.PersesAlternativeBlockAst
import org.perses.antlr.ast.RuleNameRegistry.RuleNameHandle
import org.perses.reduction.AbstractTokenReducer
import org.perses.reduction.FixpointReductionState
import org.perses.reduction.ReducerAnnotation
import org.perses.reduction.ReducerContext
import org.perses.spartree.AbstractSparTreeNode
import org.perses.spartree.ChildHoistingActionSet
import org.perses.spartree.MinimalSparTreeGenerator
import org.perses.spartree.SparTree
import org.perses.spartree.TreeNodeFilterResult
import org.perses.util.toImmutableList

class SubTreeReplacementReducer(
  reducerContext: ReducerContext,
) : AbstractNonDeletionBasedReducer(META, reducerContext) {

  private val generator: MinimalSparTreeGenerator by lazy {
    // Make generator global to avoid instantiating it multiple times
    // It is costly to create a new generator.
    reducerContext.computePiggybackPayloadIfAbsent(
      PiggybackPayloadKey(
        owningClass = this::class.java,
        payloadName = "minimal_spar_tree_generator",
        extraData = ImmutableList.of(reducerContext.configuration.parserFacade),
      ),
    ) {
      MinimalSparTreeGenerator(
        reducerContext.configuration.parserFacade,
        sparTreeNodeFactory = reducerContext.sparTreeNodeFactory,
      )
    } as MinimalSparTreeGenerator
  }

  override fun internalReduce(fixpointReductionState: FixpointReductionState) {
    val tree = fixpointReductionState.sparTree.getTreeRegardlessOfParsability()
    mutationHistory.add(tree.programSnapshot)
    val nodesWithTheirAlternativeRules = computeCandidateNodesWithAlternativeRules(tree)
    for (entry in nodesWithTheirAlternativeRules.entries()) {
      val node = entry.key
      val rule = entry.value
      val originalLexerRuleNodeList = node.leafNodeSequence().toImmutableList()
      val indicesOfSmallerOrEqualAlternatives = ImmutableIntArray.builder().apply {
        addAll(
          generator
            .getIndicesOfAlternativesWithSmallerSize(rule, originalLexerRuleNodeList.size),
        )
        addAll(
          generator
            .getIndicesOfAlternativesWithSameSize(rule, originalLexerRuleNodeList.size),
        )
      }.build()
      for (index in indicesOfSmallerOrEqualAlternatives.asList()) {
        val replacement = generator.generateNodeFromDesignatedAlternative(
          originalLexerRuleNodeList,
          rule,
          index,
        ) ?: continue
        val edit = tree.createAnyNodeReplacementEdit(
          ChildHoistingActionSet.createByReplacingSingleNode(
            targetNode = node,
            replacingNode = replacement,
            actionsDescription = "replace a node with a node generated by " +
              "another alternative rule",
          ),
        )
        if (testAndCacheUnseenResultedProgram(edit) != null) {
          tree.applyEdit(edit)
          return
        }
      }
    }
  }

  companion object {
    const val NAME = "subtree_replacer"

    private fun computeCandidateNodesWithAlternativeRules(
      tree: SparTree,
    ): ImmutableMultimap<AbstractSparTreeNode, RuleNameHandle> {
      val builder = ImmutableMultimap.builder<AbstractSparTreeNode, RuleNameHandle>()

      tree.root.boundedBFSChildren { node, _ ->
        val rules = findAllAlternativeRulesInPayload(node)
        if (rules.isNotEmpty()) {
          builder.putAll(node, rules)
        }
        TreeNodeFilterResult.CONTINUE
      }
      return builder.build()
    }

    val META = object : ReducerAnnotation(
      shortName = NAME,
      description = "Randomly pick up a subtree, " +
        "and replace it with another subtree of the same type.",
      deterministic = false,
      reductionResultSizeTrend = ReductionResultSizeTrend.BEST_RESULT_SIZE_REMAIN,
    ) {
      override fun create(reducerContext: ReducerContext): ImmutableList<AbstractTokenReducer> {
        return ImmutableList.of(SubTreeReplacementReducer(reducerContext))
      }
    }

    private fun findAllAlternativeRulesInPayload(
      node: AbstractSparTreeNode,
    ): ImmutableList<RuleNameHandle> {
      val rules = node.payload?.asSinglePayloadList?.map {
        it.expectedAntlrRuleType
      } ?: return ImmutableList.of()
      return rules.filterNotNull()
        .filter {
          it.ruleDef.body is PersesAlternativeBlockAst
        }.map { it.ruleDef.ruleNameHandle }.toImmutableList()
    }
  }
}
