# Regress and plot growth rate against temperature
# Project "Saiwan_data_processing"
# Author: Niels J. de Winter
# Study: "Living on the edge: Response of rudist bivalves (Hippuritida) to hot and highly seasonal climate in the low-latitude Saiwan site, Oman"

require(tidyverse)
require(ggpubr)
require(ggforce)
require(ggridges)

# Import data
Saiwan_SI <- read.csv("8_Results_monthly_extremes/Saiwan_SI_dated2.csv") |>
    mutate(profile = specimen_name) |> # Add new specimen column to merge data from profiles in H579
    rename(specimen = specimen_name) # Rename the specimen column
# monthly_extremes2 <- read.csv("8_Results_monthly_extremes/Monthly_extremes.csv") # Monthly extremes
Monthly_data <- read.csv("8_Results_monthly_extremes/Monthly_SI_summary2.csv") |>
    mutate(profile = specimen) # Add new specimen column to merge data from profiles in H579 

# Merge profiles within specimen H579
Monthly_data$specimen[which(Monthly_data$specimen %in% c("H579A", "H579B", "H579C", "H579D", "H579E"))] <- "H579"
Saiwan_SI$specimen[which(Saiwan_SI$specimen %in% c("H579A", "H579B", "H579C", "H579D", "H579E"))] <- "H579"

# Plot growth rate against temperature
GR_T_plot_all <- ggplot(data = Monthly_data) +
# Add gradient of maximum tolerated temperatures from Roebuck Bay (Compton et al., 2007)
geom_rect(
    alpha = 0.01,
    xmin = 32.72,
    xmax = 41.76,
    ymin = -Inf,
    ymax = Inf,
    fill = "lightcoral",
    color = NA
) +
geom_ellipse(
    aes(
        x0 = Tmean,
        y0 = mean.GR,
        a = Tsd,
        b = sd.GR,
        fill = species,
        angle = 0
    ),
    alpha = 0.02,
    color = NA
) +
geom_point(
    aes(
        x = Tmean,
        y = mean.GR,
        color = species,
        shape = specimen
    )
) +
geom_smooth(
    aes(
        x = Tmean,
        y = mean.GR,
        fill = species
    ),
    color = "black",
    se = TRUE,
    method = "lm"
) +
# Add line of apparent growth limit
geom_abline(
    slope = (0 - 125) / (55 - 30),
    intercept = 275,
    linewidth = 1.5,
    linetype = "dashed"
) +
scale_x_continuous("Temperature (degrees C)",
    breaks = seq(20, 50, 5),
    limits = c(20, 55)
) +
scale_y_continuous("Growth rate (um/d)",
    breaks = seq(0, 175, 25)
) +
scale_shape_manual(values = 1:11) +
theme_bw() +
ggtitle("Relationship between growth rate and temperature\nin all three species")

# Plot growth rate against temperature for Torreites sanchezi specimens
GR_T_plot_TS <- ggplot(data = filter(Monthly_data, species == "Torreites sanchezi" & specimen != "R1")) +
# Add gradient of maximum tolerated temperatures from Roebuck Bay (Compton et al., 2007)
geom_rect(
    alpha = 0.01,
    xmin = 32.72,
    xmax = 41.76,
    ymin = -Inf,
    ymax = Inf,
    fill = "lightcoral",
    color = NA
) +
geom_ellipse(
    aes(
        x0 = Tmean,
        y0 = mean.GR,
        a = Tsd,
        b = sd.GR,
        fill = specimen,
        angle = 0
    ),
    alpha = 0.02,
    color = NA
) +
geom_point(
    aes(
        x = Tmean,
        y = mean.GR,
        color = specimen,
        shape = specimen
    )
) +
geom_smooth(
    aes(
        x = Tmean,
        y = mean.GR,
        fill = specimen
    ),
    color = "black",
    se = TRUE,
    method = "lm"
) +
# Add line of apparent growth limit
geom_abline(
    slope = (0 - 125) / (55 - 30),
    intercept = 275,
    linewidth = 1.5,
    linetype = "dashed"
) +
scale_x_continuous("Temperature (degrees C)",
    breaks = seq(20, 50, 5),
    limits = c(20, 55)
) +
scale_y_continuous("Growth rate (um/d)",
    breaks = seq(0, 175, 25)
) +
scale_shape_manual(values = 1:9) +
scale_fill_brewer(type = "div") +
scale_color_brewer(type = "div") +
theme_bw() +
ggtitle("Relationship between growth rate and temperature\nin Torreites sanchezi")

# Plot histogram of growth over temperature for all species
Growth_hist_all <- ggplot(Saiwan_SI) +
# Add gradient of maximum tolerated temperatures from Roebuck Bay (Compton et al., 2007)
geom_rect(
    alpha = 0.01,
    xmin = 32.72,
    xmax = 41.76,
    ymin = -Inf,
    ymax = Inf,
    fill = "lightcoral",
    color = NA
) +
geom_density_ridges(
    aes(
        x = Tmean,
        y = species,
        fill = species
    ),
    alpha = 0.5,
    scale = 0.95,
    rel_min_height = 0.01,
    quantile_lines = TRUE
) +
# geom_freqpoly(
#     aes(
#         x = Tmean,
#         color = species
#     ),
#     binwidth = 2,
#     position = "dodge",
#     linewidth = 2,
#     alpha = 0.5
# ) +
# geom_histogram(
#     aes(
#         x = Tmean,
#         fill = species
#     ),
#     binwidth = 2,
#     position = "dodge"
# ) +
scale_x_continuous("Temperature (degrees C)",
    breaks = seq(20, 50, 5),
    limits = c(20, 55)
) +
scale_y_discrete(expand = expand_scale(add = c(-0.2, 1))) +
theme_bw() +
ggtitle("Histogram of temperatures recorded\nper species")

# Plot histogram of growth over temperature for Torreites sanchezi
Growth_hist_TS <- ggplot(filter(Saiwan_SI, species == "Torreites sanchezi")) +
# Add gradient of maximum tolerated temperatures from Roebuck Bay (Compton et al., 2007)
geom_rect(
    alpha = 0.01,
    xmin = 32.72,
    xmax = 41.76,
    ymin = -Inf,
    ymax = Inf,
    fill = "lightcoral",
    color = NA
) +
geom_density_ridges(
    aes(
        x = Tmean,
        y = specimen,
        fill = specimen
    ),
    alpha = 0.5,
    scale = 0.95,
    rel_min_height = 0.01,
    quantile_lines = TRUE
) +
# geom_freqpoly(
#     aes(
#         x = Tmean,
#         color = specimen
#     ),
#     binwidth = 2,
#     position = "dodge",
#     linewidth = 2,
#     alpha = 0.5
# ) +
# geom_histogram(
#     aes(
#         x = Tmean,
#         fill = specimen
#     ),
#     binwidth = 2,
#     position = "dodge"
# ) +
scale_x_continuous("Temperature (degrees C)",
    breaks = seq(20, 50, 5),
    limits = c(20, 55)
) +
scale_y_discrete(expand = expand_scale(add = c(-0.2, 1))) +
scale_fill_brewer(type = "div") +
scale_color_brewer(type = "div") +
theme_bw() +
ggtitle("Histogram of temperatures recorded per\nTorreites sanchezi specimen")

# ------------------------------------------------------------------------------
# Figure 9
# ------------------------------------------------------------------------------

# Combine growth rate vs temperature plots
GR_T_plot_combined <- ggarrange(
    ggarrange(
        Growth_hist_all + theme(legend.position = "none"),
        GR_T_plot_all + coord_cartesian(ylim = c(0, 150)) + theme(legend.position = "none"),
        labels = c("A", "C"),
        ncol = 1
    ) + theme(legend.position = "none"),
    ggarrange(
        Growth_hist_TS + theme(legend.position = "none"),
        GR_T_plot_TS + coord_cartesian(ylim = c(0, 150)) + theme(legend.position = "none"),
        labels = c("B", "D"),
        ncol = 1
    ),
    nrow = 1
)