# Plot monthly and extreme seasonal data
# Project "Saiwan_data_processing"
# Author: Niels J. de Winter
# Study: "Living on the edge: Response of rudist bivalves (Hippuritida) to hot and highly seasonal climate in the low-latitude Saiwan site, Oman"

require(tidyverse)
require(ggpubr)

# Import data
monthly_extremes2 <- read.csv("8_Results_monthly_extremes/Monthly_extremes.csv") # Monthly extremes
Monthly_data <- read.csv("8_Results_monthly_extremes/Monthly_SI_summary2.csv") |>
    mutate(profile = specimen) # Add new specimen column to merge data from profiles in H579 
Saiwan_SI <- read.csv("8_Results_monthly_extremes/Saiwan_SI_dated2.csv") |>
    mutate(profile = specimen_name) |> # Add new specimen column to merge data from profiles in H579
    rename(specimen = specimen_name) # Rename the specimen column

# Merge profiles within specimen H579
Monthly_data$specimen[which(Monthly_data$specimen %in% c("H579A", "H579B", "H579C", "H579D", "H579E"))] <- "H579"
Saiwan_SI$specimen[which(Saiwan_SI$specimen %in% c("H579A", "H579B", "H579C", "H579D", "H579E"))] <- "H579"

# Calculate max and min values for T, d18Ow and GR within specimen H579
monthly_extremes_H579 <- filter(Monthly_data, specimen == "H579") |>
    group_by(profile) |>
    summarize(
        species = first(species),
        profile = first(profile),
        d18O_mean = mean(d18Omean),
        d18O_max = max(d18Omean),
        d18O_min = min(d18Omean),
        d13C_mean = mean(d13Cmean),
        d13C_max = max(d13Cmean),
        d13C_min = min(d13Cmean),
        T_mean = mean(Tmean),
        T_max = max(Tmean),
        T_min = min(Tmean),
        T_mean_1VSMOW = mean(Tmean_1VSMOW),
        T_max_1VSMOW = max(Tmean_1VSMOW),
        T_min_1VSMOW = min(Tmean_1VSMOW),
        T_mean_clusterVSMOW = mean(Tmean_clusterVSMOW),
        T_max_clusterVSMOW = max(Tmean_clusterVSMOW),
        T_min_clusterVSMOW = min(Tmean_clusterVSMOW),
        dw_mean = mean(d18Owmean),
        dw_max = max(d18Owmean),
        dw_min = min(d18Owmean),
        GR_mean = mean(mean.GR),
        GR_max = max(mean.GR),
        GR_min = min(mean.GR)
    )

# plot temperature, d18Ow and growth rate ranges per species
plotting_order <- c("B11", # Define order of specimens for easier plotting
    "B6",
    unique(
        Saiwan_SI$specimen[which(Saiwan_SI$species == "Torreites sanchezi")]
    )
)

# Plot ranges of temperatures per specimen
Trange_plot <- ggplot(Saiwan_SI) +
    # plot monthly extremes
    geom_rect(data = filter(monthly_extremes2, season == "warmest"),
        aes(xmin = -Inf,
            xmax = Inf,
            ymin = Tmin,
            ymax = Tmax
        ),
        fill = "red",
        alpha = 0.3) +
    geom_hline(data = filter(monthly_extremes2, season == "warmest"),
        aes(yintercept = Tmean),
        color = "red",
        linewidth = 2
    ) +
    geom_rect(data = filter(monthly_extremes2, season == "coldest"),
        aes(xmin = -Inf,
            xmax = Inf,
            ymin = Tmin,
            ymax = Tmax
        ),
        fill = "blue",
        alpha = 0.3) +
    geom_hline(data = filter(monthly_extremes2, season == "coldest"),
        aes(yintercept = Tmean),
        color = "blue",
        linewidth = 2
    ) +
    geom_violin(
        aes(
            x = factor(
                specimen,
                level = plotting_order
            ),
            y = Tmean,
            fill = species
        ),
        alpha = 0.5,
        trim = FALSE,
        color = NA,
        na.rm = TRUE,
        scale = "width"
    ) +
    geom_point(data = Monthly_data,
        aes(
            x = factor(
                specimen,
                level = plotting_order
            ),
            y = Tmean
        ),
        size = 2
    ) +
    scale_y_continuous("Temperature\n(degrees C)",
        breaks = seq(20, 50, 5)
    ) +
    scale_x_discrete("Specimen") +
    theme_bw() +
    coord_cartesian(ylim = c(15, 50)) +
    ggtitle("Monthly ranges per specimen")

# Plot ranges of temperatures within specimen H579
Trange_plot_H579 <- ggplot(filter(Saiwan_SI, specimen == "H579")) +
    # plot monthly extremes
    geom_rect(data = monthly_extremes_H579,
        xmin = -Inf,
        xmax = Inf,
        ymin = min(monthly_extremes_H579$T_max),
        ymax = max(monthly_extremes_H579$T_max),
        fill = "red",
        alpha = 0.1) +
    geom_hline(yintercept = mean(monthly_extremes_H579$T_max),
        color = "red",
        linewidth = 2
    ) +
    geom_rect(data = monthly_extremes_H579,
        xmin = -Inf,
            xmax = Inf,
            ymin = min(monthly_extremes_H579$T_min),
            ymax = min(monthly_extremes_H579$T_min),
        fill = "blue",
        alpha = 0.1) +
    geom_hline(yintercept = mean(monthly_extremes_H579$T_min),
        color = "blue",
        linewidth = 2
    ) +
    geom_violin(
        aes(x = profile,
            y = Tmean,
            fill = profile
        ),
        alpha = 0.5,
        trim = FALSE,
        color = "black"
    ) +
    geom_point(data = filter(Monthly_data, specimen == "H579"),
        aes(x = profile,
            y = Tmean,
            fill = profile
        ),
        size = 2
    ) +
    scale_y_continuous("Temperature\n(degrees C)",
        breaks = seq(20, 50, 5)
    ) +
    scale_x_discrete("Profile") +
    scale_fill_brewer(palette = "Greens") +
    theme_bw() +
    coord_cartesian(ylim = c(15, 50)) +
    ggtitle("Specimen H579 profiles")

# Plot ranges of d18O of calcification fluid per specimen
dwrange_plot <- ggplot(Saiwan_SI) +
    # plot monthly extremes
    geom_rect(data = filter(monthly_extremes2, season == "warmest"),
        aes(xmin = -Inf,
            xmax = Inf,
            ymin = dwmin,
            ymax = dwmax
        ),
        fill = "red",
        alpha = 0.3) +
    geom_hline(data = filter(monthly_extremes2, season == "warmest"),
        aes(yintercept = dwmean),
        color = "red",
        linewidth = 2
    ) +
    geom_rect(data = filter(monthly_extremes2, season == "coldest"),
        aes(xmin = -Inf,
            xmax = Inf,
            ymin = dwmin,
            ymax = dwmax
        ),
        fill = "blue",
        alpha = 0.3) +
    geom_hline(data = filter(monthly_extremes2, season == "coldest"),
        aes(yintercept = dwmean),
        color = "blue",
        linewidth = 2
    ) +
    geom_violin(
        aes(
            x = factor(
                specimen,
                level = plotting_order
            ),
            y = d18Owmean,
            fill = species
        ),
        alpha = 0.5,
        trim = TRUE,
        color = NA,
        na.rm = TRUE,
        scale = "width"
    ) +
    geom_point(data = Monthly_data,
        aes(
            x = factor(
                specimen,
                level = plotting_order
            ),
            y = d18Owmean
        ),
        size = 2
    ) +
    scale_y_continuous("Reconstructed d18O of\nseawater (VSMOW)",
        breaks = seq(-10, 10, 1)
    ) +
    scale_x_discrete("Specimen") +
    theme_bw() +
    coord_cartesian(ylim = c(-5, 2))

# Plot ranges of d18Ow within specimen H579
dwrange_plot_H579 <- ggplot(filter(Saiwan_SI, specimen == "H579")) +
    # plot monthly extremes
    geom_rect(data = monthly_extremes_H579,
        xmin = -Inf,
        xmax = Inf,
        ymin = min(monthly_extremes_H579$dw_max),
        ymax = max(monthly_extremes_H579$dw_max),
        fill = "red",
        alpha = 0.1) +
    geom_hline(yintercept = mean(monthly_extremes_H579$dw_max),
        color = "red",
        linewidth = 2
    ) +
    geom_rect(data = monthly_extremes_H579,
        xmin = -Inf,
            xmax = Inf,
            ymin = min(monthly_extremes_H579$dw_min),
            ymax = min(monthly_extremes_H579$dw_min),
        fill = "blue",
        alpha = 0.1) +
    geom_hline(yintercept = mean(monthly_extremes_H579$dw_min),
        color = "blue",
        linewidth = 2
    ) +
    geom_violin(
        aes(x = profile,
            y = d18Owmean,
            fill = profile
        ),
        alpha = 0.5,
        trim = FALSE,
        color = "black"
    ) +
    geom_point(data = filter(Monthly_data, specimen == "H579"),
        aes(x = profile,
            y = d18Owmean,
            fill = profile
        ),
        size = 2
    ) +
    scale_y_continuous("Reconstructed d18O of\nseawater (VSMOW)",
        breaks = seq(-10, 10, 1)
    ) +
    scale_x_discrete("Profile") +
    scale_fill_brewer(palette = "Greens") +
    theme_bw() +
    coord_cartesian(ylim = c(-5, 2))

# Plot range of growth rates per specimen
GRrange_plot <- ggplot(Saiwan_SI) +
    # plot monthly extremes
    geom_rect(data = filter(monthly_extremes2, season == "warmest"),
        aes(xmin = -Inf,
            xmax = Inf,
            ymin = GRmin,
            ymax = GRmax
        ),
        fill = "red",
        alpha = 0.3) +
    geom_hline(data = filter(monthly_extremes2, season == "warmest"),
        aes(yintercept = GRmean),
        color = "red",
        linewidth = 2
    ) +
    geom_rect(data = filter(monthly_extremes2, season == "coldest"),
        aes(xmin = -Inf,
            xmax = Inf,
            ymin = GRmin,
            ymax = GRmax
        ),
        fill = "blue",
        alpha = 0.3) +
    geom_hline(data = filter(monthly_extremes2, season == "coldest"),
        aes(yintercept = GRmean),
        color = "blue",
        linewidth = 2
    ) +
    geom_violin(
        aes(
            x = factor(
                specimen,
                level = plotting_order
            ),
            y = mean.GR,
            fill = species
        ),
        alpha = 0.5,
        trim = TRUE,
        color = NA,
        na.rm = TRUE,
        scale = "width"
    ) +
    geom_point(data = Monthly_data,
        aes(
            x = factor(
                specimen,
                level = plotting_order
            ),
            y = mean.GR
        ),
        size = 2
    ) +
    scale_y_continuous("Growth rate\n(um/d)",
        breaks = seq(0, 175, 25)
    ) +
    scale_x_discrete("Specimen") +
    theme_bw() +
    coord_cartesian(ylim = c(0, 175))

# Plot ranges of growth rate within specimen H579
GRrange_plot_H579 <- ggplot(filter(Saiwan_SI, specimen == "H579")) +
    # plot monthly extremes
    geom_rect(data = monthly_extremes_H579,
        xmin = -Inf,
        xmax = Inf,
        ymin = min(monthly_extremes_H579$GR_max),
        ymax = max(monthly_extremes_H579$GR_max),
        fill = "red",
        alpha = 0.1) +
    geom_hline(yintercept = mean(monthly_extremes_H579$GR_max),
        color = "red",
        linewidth = 2
    ) +
    geom_rect(data = monthly_extremes_H579,
        xmin = -Inf,
            xmax = Inf,
            ymin = min(monthly_extremes_H579$GR_min),
            ymax = min(monthly_extremes_H579$GR_min),
        fill = "blue",
        alpha = 0.1) +
    geom_hline(yintercept = mean(monthly_extremes_H579$GR_min),
        color = "blue",
        linewidth = 2
    ) +
    geom_violin(
        aes(x = profile,
            y = mean.GR,
            fill = profile
        ),
        alpha = 0.5,
        trim = FALSE,
        color = "black"
    ) +
    geom_point(data = filter(Monthly_data, specimen == "H579"),
        aes(x = profile,
            y = mean.GR,
            fill = profile
        ),
        size = 2
    ) +
    scale_y_continuous("Growth rate\n(um/d)",
        breaks = seq(0, 175, 25)
    ) +
    scale_x_discrete("Profile") +
    scale_fill_brewer(palette = "Greens") +
    theme_bw() +
    coord_cartesian(ylim = c(0, 175))

# Combine monthly ranges plots
Monthly_ranges_plots <- ggarrange(
    ggarrange(
        Trange_plot + theme(axis.title.x = element_blank(), axis.text.x = element_blank()),
        dwrange_plot + theme(axis.title.x = element_blank(), axis.text.x = element_blank()),
        GRrange_plot,
        common.legend = TRUE,
        ncol = 1,
        labels = c("A", "C", "E")
    ),
    ggarrange(
        Trange_plot_H579 + theme(axis.title.x = element_blank(), axis.text.x = element_blank()),
        dwrange_plot_H579 + theme(axis.title.x = element_blank(), axis.text.x = element_blank()),
        GRrange_plot_H579,
        common.legend = TRUE,
        ncol = 1,
        labels = c("B", "D", "F")
    ),
    ncol = 2,
    widths = c(1, 0.7),
    common.legend = TRUE
)

# Export extremes of H579 profiles
write.csv(monthly_extremes_H579, "8_Results_monthly_extremes/Monthly_extremes_H579_profiles.csv", row.names = FALSE)