# Combine isotope data with clumped data to find extreme temperatures
# Project "Saiwan_data_processing"
# Author: Niels J. de Winter
# Study: "Living on the edge: Response of rudist bivalves (Hippuritida) to hot and highly seasonal climate in the low-latitude Saiwan site, Oman"

require(tidyverse)
require(ggpubr) 
source("calculate_temp_and_d18Osw_NJW.R") # load d18Oc-d18Ow conversion functions

# Load stable stable and clumped isotope data
Saiwan_SI <- read.csv("3_Combine_monthly_SI_data/3_Saiwan_SI_dated.csv") # Add datapoints
Clumped_data <- read.csv("7_Results_clumped_clustering/Clumped_TS_intcorr_cluster_review.csv") # Clumped data
Clumped_clusters <- read.csv("7_Results_clumped_clustering/Clumped_TS_clusters_location.csv") # Import cluster results by location
Monthly_data <- read.csv("3_Combine_monthly_SI_data/3_Monthly_SI_summary.csv") # Monthly means
Monthly_data <- Monthly_data[complete.cases(Monthly_data), ] # remove NA lines

Saiwan_SI$specimen_name[which(Saiwan_SI$specimen_name == "HU_027")] <- "HU-027" # Rename specimen HU-027 for consistency with clumped data

# ------------------------------------------------------------------------------
# Regress temperature and d18Ow against d18Oc
# Using location-based clusters
loess_T47_d18Oc_loc <- loess(
    formula = T47 ~ d18O,
    data = Clumped_data,
    span = 1,
    control = loess.control(surface = "direct")
)

fitted_loess_T47_d18Oc_loc <- data.frame(
    d18Oc = unname(loess_T47_d18Oc_loc$x),
    T47 = loess_T47_d18Oc_loc$fitted
)

loess_d18Ow_d18Oc_loc <- loess(
    formula = d18Ow ~ d18O,
    data = Clumped_data,
    span = 1,
    control = loess.control(surface = "direct")
)
fitted_loess_d18Ow_d18Oc_loc <- data.frame(
    d18Oc = unname(loess_d18Ow_d18Oc_loc$x),
    d18Ow = loess_d18Ow_d18Oc_loc$fitted
)

# ------------------------------------------------------------------------------
# Apply regressions to predict temperature and d18Ow for all data
Monthly_data <- Monthly_data |>
    mutate(
        Tmean = predict(loess_T47_d18Oc_loc, newdata = data.frame(d18O = d18Omean)),
        Tsd = abs(predict(loess_T47_d18Oc_loc, newdata = data.frame(d18O = d18Omean + d18Osd)) - predict(loess_T47_d18Oc_loc, newdata = data.frame(d18O = d18Omean - d18Osd))) / 2,
        d18Owmean = predict(loess_d18Ow_d18Oc_loc, newdata = data.frame(d18O = d18Omean)),
        d18Owsd = abs(predict(loess_d18Ow_d18Oc_loc, newdata = data.frame(d18O = d18Omean + d18Osd)) + predict(loess_d18Ow_d18Oc_loc, newdata = data.frame(d18O = d18Omean - d18Osd))) / 2,
    ) |>
    merge( # Add species to monthly data
        unique(
            data.frame(
                specimen = Saiwan_SI$specimen_name,
                species = Saiwan_SI$species
            )
        ),
        by = "specimen"
    )

# Similar for individual measurements
Saiwan_SI <- Saiwan_SI |>
    mutate(
        Tmean = predict(loess_T47_d18Oc_loc, newdata = data.frame(d18O = d18O)),
        Tsd = abs(predict(loess_T47_d18Oc_loc, newdata = data.frame(d18O = d18O + d18O_SD)) - predict(loess_T47_d18Oc_loc, newdata = data.frame(d18O = d18O - d18O_SD))) / 2,
        d18Owmean = predict(loess_d18Ow_d18Oc_loc, newdata = data.frame(d18O = d18O)),
        d18Owsd = abs(predict(loess_d18Ow_d18Oc_loc, newdata = data.frame(d18O = d18O + d18O_SD)) - predict(loess_d18Ow_d18Oc_loc, newdata = data.frame(d18O = d18O - d18O_SD))) / 2
    )

# Apply constant d18Ow value for Oscillopha figari (B11) and Vaccinites vesiculosus (B6) specimens
Monthly_data$Tmean[which(Monthly_data$species %in% c("Oscillopha figari", "Vaccinites vesiculosus"))] <- T_from_d18O_d18Ow(
    d18Oc = Monthly_data$d18Omean[which(Monthly_data$species %in% c("Oscillopha figari", "Vaccinites vesiculosus"))],
    d18Ow = d18Ow_from_d18O_T(mean(Clumped_data$d18O), mean(Clumped_data$T47) + 273.15),
    mineral = "calcite"
) - 273.15
Monthly_data$d18Owmean[which(Monthly_data$species %in% c("Oscillopha figari", "Vaccinites vesiculosus"))] <- d18Ow_from_d18O_T(
    d18Oc = Monthly_data$d18Omean[which(Monthly_data$species %in% c("Oscillopha figari", "Vaccinites vesiculosus"))],
    T = Monthly_data$Tmean[which(Monthly_data$species %in% c("Oscillopha figari", "Vaccinites vesiculosus"))] + 273.15,
    mineral = "calcite"
)

# Apply constant d18Ow of -1 per mille to test temperature
Monthly_data$Tmean_1VSMOW <- T_from_d18O_d18Ow(
    d18Oc = Monthly_data$d18Omean,
    d18Ow = -1,
    mineral = "calcite"
) - 273.15

# Apply constant d18Ow equal to the warmest three clusters to test temperature
Monthly_data$Tmean_clusterVSMOW <- T_from_d18O_d18Ow(
    d18Oc = Monthly_data$d18Omean,
    d18Ow = mean(Clumped_data$d18Ow[which(Clumped_data$location_cluster %in% c("R_01-R_07", "R_08-R_11", "RB_2"))]),
    mineral = "calcite"
) - 273.15

# Now repeat for all data
Saiwan_SI$Tmean[which(Saiwan_SI$species %in% c("Oscillopha figari", "Vaccinites vesiculosus"))] <- T_from_d18O_d18Ow(
    d18Oc = Saiwan_SI$d18O[which(Saiwan_SI$species %in% c("Oscillopha figari", "Vaccinites vesiculosus"))],
    d18Ow = d18Ow_from_d18O_T(mean(Clumped_data$d18O), mean(Clumped_data$T47) + 273.15),
    mineral = "calcite"
) - 273.15
Saiwan_SI$d18Owmean[which(Saiwan_SI$species %in% c("Oscillopha figari", "Vaccinites vesiculosus"))] <- d18Ow_from_d18O_T(
    d18Oc = Saiwan_SI$d18O[which(Saiwan_SI$species %in% c("Oscillopha figari", "Vaccinites vesiculosus"))],
    T = Saiwan_SI$Tmean[which(Saiwan_SI$species %in% c("Oscillopha figari", "Vaccinites vesiculosus"))] + 273.15,
    mineral = "calcite"
)

# Apply constant d18Ow of -1 per mille to test temperature
Saiwan_SI$Tmean_1VSMOW <- T_from_d18O_d18Ow(
    d18Oc = Saiwan_SI$d18O,
    d18Ow = -1,
    mineral = "calcite"
) - 273.15

# Apply constant d18Ow equal to the warmest three clusters to test temperature
Saiwan_SI$Tmean_clusterVSMOW <- T_from_d18O_d18Ow(
    d18Oc = Saiwan_SI$d18O,
    d18Ow = mean(Clumped_data$d18Ow[which(Clumped_data$location_cluster %in% c("R_01-R_07", "R_08-R_11", "RB_2"))]),
    mineral = "calcite"
) - 273.15

# Merge clumped dataset into SI dataset
Saiwan_SI <- full_join(
    Saiwan_SI,
    data.frame(
        D = NA,
        d13C = Clumped_data$d13C,
        d18O = Clumped_data$d18O,
        d13C_SD = Clumped_data$d13Csd,
        d18O_SD = Clumped_data$d18Osd,
        species = "Torreites sanchezi",
        specimen_name = "HU-027",
        Tmean = predict(loess_T47_d18Oc_loc, newdata = data.frame(d18O = Clumped_data$d18O)),
        d18Owmean = predict(loess_d18Ow_d18Oc_loc, newdata = data.frame(d18O = Clumped_data$d18O))
    )
)

Monthly_data <- full_join(
    Monthly_data,
    data.frame(
        month = 1:4,
        N = Clumped_clusters$N,
        d18Omean = Clumped_clusters$d18Omean,
        d18Osd = Clumped_clusters$d18Osd,
        d13Cmean = Clumped_clusters$d13Cmean,
        d13Csd = Clumped_clusters$d13Csd,
        specimen = "HU-027",
        age_model = "DOY.d13C",
        species = "Torreites sanchezi",
        proxy = NA,
        Tmean = Clumped_clusters$Tmean,
        d18Owmean = Clumped_clusters$dwmean
    )
)

# Select the right age models, picking the highest precision age model per specimen
Monthly_data$age_model_select <- FALSE
Monthly_data$age_model_select[which(Monthly_data$specimen %in% c("B6", "B10", "H579C", "H579D", "H579E") & Monthly_data$age_model == "DOY.d13C")] <- TRUE
Monthly_data$age_model_select[which(Monthly_data$specimen %in% c("B11", "H576", "H579A", "H579B", "H585") & Monthly_data$age_model == "DOY.d18O")] <- TRUE
Monthly_data$age_model_select[which(Monthly_data$specimen == "HU-027")] <- TRUE # Keep the cluster data from specimen HU-027

# Isolate only the most precise growth model results
# For monthly data
Monthly_data <- Monthly_data %>%
filter(age_model_select) %>%
mutate(
    mean.GR = ifelse(age_model == "DOY.d13C", mean.GR.d13Cmean, mean.GR.d18Omean),
    sd.GR = ifelse(age_model == "DOY.d13C", mean.GR.d13Csd, mean.GR.d18Osd)
) %>%
select(-mean.GR.d18Omean, -mean.GR.d13Cmean, -mean.GR.d18Osd, -mean.GR.d13Csd)

# for complete dataset
Saiwan_SI <- Saiwan_SI %>%
mutate(
    mean.GR = ifelse(specimen_name %in% c("B6", "B10", "H579C", "H579D", "H579E"), mean.GR.d13C, mean.GR.d18O),
    CL95.GR = ifelse(specimen_name %in% c("B6", "B10", "H579C", "H579D", "H579E"), CL95.GR.d13C, CL95.GR.d18O)
)

# Find warmest and coldest month statistics + statistics of growth rates
monthly_extremes <- filter(Monthly_data, age_model_select == TRUE) |>
    group_by(specimen) |>
    summarize(
        species = first(species),
        d18O_mean = mean(d18Omean),
        d18Omax = max(d18Omean),
        d18Omin = min(d18Omean),
        d13C_mean = mean(d13Cmean),
        d13Cmax = max(d13Cmean),
        d13Cmin = min(d13Cmean),
        T_mean = mean(Tmean),
        Tmax = max(Tmean),
        Tmin = min(Tmean),
        T_mean_1VSMOW = mean(Tmean_1VSMOW),
        Tmax_1VSMOW = max(Tmean_1VSMOW),
        Tmin_1VSMOW = min(Tmean_1VSMOW),
        T_mean_clusterVSMOW = mean(Tmean_clusterVSMOW),
        Tmax_clusterVSMOW = max(Tmean_clusterVSMOW),
        Tmin_clusterVSMOW = min(Tmean_clusterVSMOW),
        dw_mean = mean(d18Owmean),
        dwmax = max(d18Owmean),
        dwmin = min(d18Owmean),
        GR_mean = mean(mean.GR),
        GRmax = max(mean.GR),
        GRmin = min(mean.GR)
    )

# Find extremes of T. sanchezi and others
monthly_extremes2 <- data.frame(
    season = c("warmest", "coldest"),
    d18Omean = c(
        mean(monthly_extremes$d18Omin[which(monthly_extremes$species == "Torreites sanchezi")]),
        mean(monthly_extremes$d18Omax[which(monthly_extremes$species != "Torreites sanchezi")])
    ),
    d18Omin = c(
        min(monthly_extremes$d18Omin[which(monthly_extremes$species == "Torreites sanchezi")]),
        min(monthly_extremes$d18Omax[which(monthly_extremes$species != "Torreites sanchezi")])
    ),
    d18Omax = c(
        max(monthly_extremes$d18Omin[which(monthly_extremes$species == "Torreites sanchezi")]),
        max(monthly_extremes$d18Omax[which(monthly_extremes$species != "Torreites sanchezi")])
    ),
    Tmean = c(
        mean(monthly_extremes$Tmax[which(monthly_extremes$species == "Torreites sanchezi")]),
        mean(monthly_extremes$Tmin[which(monthly_extremes$species != "Torreites sanchezi")])
    ),
    Tmin = c(
        min(monthly_extremes$Tmax[which(monthly_extremes$species == "Torreites sanchezi")]),
        min(monthly_extremes$Tmin[which(monthly_extremes$species != "Torreites sanchezi")])
    ),
    Tmax = c(
        max(monthly_extremes$Tmax[which(monthly_extremes$species == "Torreites sanchezi")]),
        max(monthly_extremes$Tmin[which(monthly_extremes$species != "Torreites sanchezi")])
    ),
    dwmean = c(
        mean(monthly_extremes$dwmax[which(monthly_extremes$species == "Torreites sanchezi")]),
        mean(monthly_extremes$dwmin[which(monthly_extremes$species == "Torreites sanchezi")])
    ),
    dwmin = c(
        min(monthly_extremes$dwmax[which(monthly_extremes$species == "Torreites sanchezi")]),
        min(monthly_extremes$dwmin[which(monthly_extremes$species == "Torreites sanchezi")])
    ),
    dwmax = c(
        max(monthly_extremes$dwmax[which(monthly_extremes$species == "Torreites sanchezi")]),
        max(monthly_extremes$dwmin[which(monthly_extremes$species == "Torreites sanchezi")])
    )
)

# Add growth rate extremes, excluding specimen HU-027 for which no growth modelling was done
monthly_extremes2 <- monthly_extremes2 |>
    mutate(
        GRmin = c(
            min(monthly_extremes$GRmax[which(monthly_extremes$species == "Torreites sanchezi" & monthly_extremes$specimen != "HU-027")]),
            min(monthly_extremes$GRmin[which(monthly_extremes$species != "Torreites sanchezi")])
        ),
        GRmax = c(
            max(monthly_extremes$GRmax[which(monthly_extremes$species == "Torreites sanchezi" & monthly_extremes$specimen != "HU-027")]),
            max(monthly_extremes$GRmin[which(monthly_extremes$species != "Torreites sanchezi")])
        ),
        GRmean = c(
            mean(monthly_extremes$GRmax[which(monthly_extremes$species == "Torreites sanchezi" & monthly_extremes$specimen != "HU-027")]),
            mean(monthly_extremes$GRmin[which(monthly_extremes$species != "Torreites sanchezi")])
        )
    )

# Find monthly extremes of entire dataset
monthly_extremes_all <- monthly_extremes %>%
summarise(
    across(
        c(d18Omax, d18Omin, d18O_mean, Tmin, Tmax, T_mean, dwmin, dwmax, dw_mean, GRmin, GRmax, GR_mean),
        list(
            mean = ~mean(.x, na.rm = TRUE),
            sd = ~sd(.x, na.rm = TRUE)
        )
    )
)

# Export data
write.csv(Monthly_data, "8_Results_monthly_extremes/Monthly_SI_summary2.csv", row.names = FALSE)
write.csv(Saiwan_SI, "8_Results_monthly_extremes/Saiwan_SI_dated2.csv", row.names = FALSE)
write.csv(monthly_extremes2, "8_Results_monthly_extremes/Monthly_extremes.csv", row.names = FALSE)
write.csv(monthly_extremes, "8_Results_monthly_extremes/Monthly_extremes_per_specimen.csv", row.names = FALSE)
write.csv(monthly_extremes_all, "8_Results_monthly_extremes/Monthly_extremes_entire_dataset.csv", row.names = FALSE)
